/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineConnector.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.jasmine;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;

import org.ow2.jasmine.monitoring.mbeancmd.Outer;

/**
 * Static methods for connecting to a JASMINe Event Switch.
 */
public class JasmineConnector {
    /**
     * JASMINe Connector Version string.
     */
    static private final String version = "v1";

    /**
     * Dummy constructor to avoid the class from being used in any other way
     * than calling the JasmineConnector.connect method.
     */
    protected JasmineConnector() {
        // Nothing
    }

    /**
     * Connects an Outer to the JASMINe Event Switch.
     * 
     * @param src
     *            Source Outer.
     * @param jasmineURI
     *            URI at which the JASMINe Event Switch is present. For more
     *            details, see "Mule Endpoint URIs".
     * @return A new Outer object, that will be set as the sink for src and will
     *         have the JASMINe Event Switch at jasmineURI as output.
     * @throws IOException
     *             If connection fails.
     */
    public static Outer connect(final Outer src, final String jasmineURI)
            throws IOException {
        PipedOutputStream stream = new PipedOutputStream();

        final LineNumberReader reader = new LineNumberReader(
                new InputStreamReader(new PipedInputStream(stream)));

        PipedOutputStream input = new PipedOutputStream();
        src.setSink(input);
        Outer outer = new Outer(input, new PrintStream(stream));

        // Start the thread that will read outputs from the pipe.
        new Thread(new MuleDispatcher(reader, jasmineURI)).start();

        return outer;
    }

    /**
     * Connects a PipedOutputStream to the JASMINe Event Switch.
     * 
     * @param src
     *            Source PipedOutputStream.
     * @param jasmineURI
     *            URI at which the JASMINe Event Switch is present. For more
     *            details, see "Mule Endpoint URIs".
     * @return A new Outer object, that will use src as input and the JASMINe
     *         Event Switch at jasmineURI as output.
     * @throws IOException
     *             If connection fails.
     */
    public static Outer connect(final PipedOutputStream src,
            final String jasmineURI) throws IOException {
        PipedOutputStream stream = new PipedOutputStream();

        final LineNumberReader reader = new LineNumberReader(
                new InputStreamReader(new PipedInputStream(stream)));

        Outer outer = new Outer(src, new PrintStream(stream));

        // Start the thread that will read outputs from the pipe.
        new Thread(new MuleDispatcher(reader, jasmineURI)).start();

        return outer;
    }
}