/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MuleDispatcher.java 9498 2011-12-15 13:09:57Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.monitoring.mbeancmd.jasmine;

import java.io.BufferedReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.mule.extras.client.MuleClient;
import org.mule.umo.UMOException;

/**
 *
 */
public class MuleDispatcher implements Runnable {

    private String version = "v1";

    /**
     * Allows to read lines printed out by the commands to PrintStream <code>pout</code>
     */
    private BufferedReader reader;

    /**
     * Keeps a header for each alias.
     */
    private Map<String, String> headers = new HashMap<String, String>();

    /**
     * Keeps attribute indexes for each alias.
     */
    private Map<String, List<Integer>> indexes = new HashMap<String, List<Integer>>();

    /**
     * Number of elements in the header from the beginning to the attributes list.
     */
    private int nbBaseHeaderElements = 7;
    /**
     * Mule entry point address.
     */
    private String jasmineURI;

    public MuleDispatcher(BufferedReader reader, String jasmineURI) {
        this.reader = reader;
        this.jasmineURI = jasmineURI;
    }

    public void run() {

        try {
            MuleClient client = null;
            while (client == null) {
                Thread.sleep(1000);
                client = new MuleClient();
            }
            /**
             * The header printed out by the command.
             * If no aliases, it will be used, at each iteration, to construct the message to be dispatched.
             * Otherwise, it is replaced by a header from <code>headers</code>
             */
            String uniqueHeader = reader.readLine();
            if (uniqueHeader == null) {
                System.err.println("Nothing to send: stop MuleDispatcher");
                return;
            }

            /**
             *  Split uniqueHeader in order to parse aliased attributes
             */
            String[] msgHeader = uniqueHeader.split(";");
            /**
             * The common part of any header (from the beginning to the attributes list)
             */
            String baseHeader = "";
            for (int i = 0; i < nbBaseHeaderElements; i ++) {
                baseHeader = baseHeader + msgHeader[i] + ";";
            }
            /**
             *  Supposes that aliases are used
             */
            boolean aliases = true;

            /**
             *  If the attributes are prefixed by aliases,
             *  get the position of the attributes for each alias and put it into <code>indexes</code>.
             *  Otherwise, set <code>aliases</code> to false
             */
            for (int i = 0; i < msgHeader.length; i++) {
                String headerElement = msgHeader[i];
                // skip the first 6 elements : date, time, sname, server, domain, mbean, cmdid
                if (i > nbBaseHeaderElements -1) {
                    // Check for alias:attName
                    int index = headerElement.indexOf(":");
                    if (index == -1) {
                        aliases = false;
                        break;
                    }
                    String alias = headerElement.substring(0, index);
                    if (!indexes.containsKey(alias)) {
                        List<Integer> indexList = new ArrayList<Integer>();
                        indexes.put(alias, indexList);
                    }
                    // add i to the indexList of the alias
                    // indexes are added in increasing order ex: 6, 7, ...
                    indexes.get(alias).add(i);
                }
            }

            if (aliases) {
                // Construct the header for each alias
                for (int i = 0; i < msgHeader.length; i++) {
                    String headerElement = msgHeader[i];
                    // skip the first 6 elements : date, time, sname, server, domain, mbean, cmdid
                    if (i > nbBaseHeaderElements -1) {
                        // Check for alias:attName
                        int index = headerElement.indexOf(":");
                        String alias = headerElement.substring(0, index);
                        String attName = headerElement.substring(index + 1, headerElement.length());
                        if (!headers.containsKey(alias)) {
                            headers.put(alias, baseHeader);
                        }
                        String currentHeader = headers.get(alias);
                        //currentHeader = currentHeader + attName + ";";

                        if (i == (msgHeader.length -1)) {
                            currentHeader = currentHeader + attName;
                        } else {
                            currentHeader = currentHeader + attName + ";";
                        }
                        headers.put(alias, currentHeader);
                    }
                }
                for (String alias : headers.keySet()) {
                    System.out.println(alias + " header: " + headers.get(alias));
                }
            }

            String header = "mbeancmd_values_" + version + "\n"
                    + uniqueHeader + "\n";

            while (true) {

                if (reader.ready()) {

                    String nextLine = reader.readLine();
                    String msg = null;
                    if (!aliases) {
                        msg = header + nextLine;
                    } else {
                        // guess the alias from the line
                        String[] msgElements = nextLine.split(";");
                        for (String alias : indexes.keySet()) {
                            List<Integer> indexList = indexes.get(alias);
                            String firstElement = msgElements[indexList.get(0)];
                            if (firstElement.length() > 0) {
                                // this line corresponds to the current alias
                                String fixedLine = fixLine(msgElements, indexList);
                                msg = "mbeancmd_values_" + version + "\n"
                                + headers.get(alias) + "\n" + fixedLine;
                                break;
                            } else {
                                continue;
                            }
                        }
                    }
                    try {
                        client.dispatch(jasmineURI, msg, null);
                    } catch (NullPointerException e) {

                        Logger.getLogger(MuleDispatcher.class.getName()).log(
                                Level.INFO,
                                "Mule stopped ... Message is not delivered anymore to "
                                        + jasmineURI);
                        break;
                    }
                }

                Thread.sleep(900);
            }

        } catch (UMOException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }

    }

    private String fixLine(final String[] msgElements, final List<Integer> indexList) {
        String newLine = "";
        // put elements before attributes
        int nbElements = 7;
        for (int i = 0; i < nbElements; i++) {
            newLine = newLine + msgElements[i] + ";";
        }
        // put index elements
        for (int index : indexList) {
            newLine = newLine + msgElements[index] + ";";
        }
        // remove the last ;
        newLine = newLine.substring(0, (newLine.length() - 1));
        return newLine;
    }

}
