/**
 * JASMINe
 * Copyright (C) 2005-2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServerSampler.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.Iterator;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Sampler for statistics on various parts of the J2EE server. Includes :
 *
 *      - machine cpu, if available
 *      - JVM cpu, if available
 *      - heap committed
 *      - heap used
 *      - number of threads
 *      - pending requests
 *      - current sessions
 *      - database connections
 *      - database busy cnx
 *      - waiters
 *      - dsleaks
 *      - dsfail
 *      - sfb instances
 *      - ssb instances
 *      - ent instances
 *      - committed tx
 *      - served cnx
 */
public class ServerSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public ServerSampler(final SamplerContext context) {
        super(context);
        try {
            // Retrieve J2EE Server
            onServer = queryON(J2EEServer_ON_PATTERN);

            // Retrieve JVM
            onJvm = queryON(JVM_ON_PATTERN);

            // Retrieve global request processors
            onGRP = queryON(GlobalRequestProcessor_ON_PATTERN);

            // Retrieve ThreadPool Object Name (jk)
            onThreadPool = queryON(ThreadPool_ON_PATTERN);

            // Retrieve Session managers
            onSessions = queryON(Manager_ON_PATTERN);

            // Retrieve JTM Object Name
            onJtm = queryON(JTAResource_ON_PATTERN);

            // Retrieve DataSource Object Names
            onDS = queryON(JDBCDataSource_ON_PATTERN);

            // Retrieve Entity bean Object Names
            onEntity = queryON(EntityBean_ON_PATTERN);

            // Retrieve StatelessSessionBean Object Names
            onSSB = queryON(StatelessSessionBean_ON_PATTERN);

            // Retrieve StatefulSessionBean Object Names
            onSFB = queryON(StatefulSessionBean_ON_PATTERN);
         } catch (Exception e) {
            e.printStackTrace();
         }
    }

    /**
     * Implementation of inherited abstract method. Returns null since
     * ServerSampler doesn't have one default object name.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOnPattern() {
        return "*:j2eeType=J2EEServer,*";
    }

    public static final String J2EEServer_ON_PATTERN = "*:j2eeType=J2EEServer,*";
    public static final String JVM_ON_PATTERN = "*:j2eeType=JVM,*";
    public static final String GlobalRequestProcessor_ON_PATTERN = "*:type=GlobalRequestProcessor,*";
    public static final String ThreadPool_ON_PATTERN = "*:type=ThreadPool,*";
    public static final String Manager_ON_PATTERN = "*:type=Manager,*";
    public static final String JTAResource_ON_PATTERN = "*:j2eeType=JTAResource,*";
    public static final String JDBCDataSource_ON_PATTERN = "*:j2eeType=JDBCDataSource,*";
    public static final String EntityBean_ON_PATTERN = "*:j2eeType=EntityBean,*";
    public static final String StatelessSessionBean_ON_PATTERN = "*:j2eeType=StatelessSessionBean,*";
    public static final String StatefulSessionBean_ON_PATTERN = "*:j2eeType=StatefulSessionBean,*";

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new ServerData();
    }

    /**
     * Implementation of inherited abstract method, will call all pollers of
     * this class.
     *
     * @see Sampler#poll(SampleData)
     *
     * @see ServerSampler#pollServer(ServerData)
     * @see ServerSampler#pollJVM(ServerData)
     * @see ServerSampler#pollSessions(ServerData)
     * @see ServerSampler#pollThreadPool(ServerData)
     * @see ServerSampler#pollJTM(ServerData)
     * @see ServerSampler#pollDataSources(ServerData)
     * @see ServerSampler#pollGlobalRequestProcessor(ServerData)
     * @see ServerSampler#pollEntity(ServerData)
     * @see ServerSampler#pollSSB(ServerData)
     * @see ServerSampler#pollSFB(ServerData)
     */
    protected SampleData poll(final SampleData data) {
        // Status is ok if all polls are OK
        boolean status = true;

        pollServer((ServerData) data);
        status = status && data.isValid();

        pollJVM((ServerData) data);
        status = status && data.isValid();

        pollSessions((ServerData) data);
        status = status && data.isValid();

        pollThreadPool((ServerData) data);
        status = status && data.isValid();

        pollJTM((ServerData) data);
        status = status && data.isValid();

        pollDataSources((ServerData) data);
        status = status && data.isValid();

        pollGlobalRequestProcessor((ServerData) data);
        status = status && data.isValid();

        pollEntity((ServerData) data);
        status = status && data.isValid();

        pollSSB((ServerData) data);
        status = status && data.isValid();

        pollSFB((ServerData) data);
        status = status && data.isValid();

        data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
        data.setCmdId(getCmdId());
        data.setValid(status);

        return data;
    }

    /**
     * Polls memory information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollServer(final ServerData data) {
        long totalMem = 0;
        long usedMem = 0;
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "currentTotalMemory",
                "currentUsedMemory"
        };

        if (onServer == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", J2EEServer_ON_PATTERN);
            return;
        }
        for (int i = 0; i < onServer.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onServer[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                if (al.isEmpty()) {
                    /**
                     *  Skip MBean with unexpected attributes.
                     */
                    continue;
                }
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();                  // currentTotalMemory
                totalMem += Long.parseLong(att.getValue().toString());
                att = (Attribute) it.next();                            // currentUsedMemory
                usedMem += Long.parseLong(att.getValue().toString());
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                if (e.getCause() != null) {
                    e.getCause().printStackTrace(System.err);
                }
            }
        }
        data.setUsedMemory(usedMem);
        data.setMemory(totalMem);
        data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
        data.setCmdId(getCmdId());
        if( onServer.length > 0) {
            data.setObjectName(onServer[0]);
        }

    }

    /**
     * Polls thread count.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollJVM(final ServerData data) {
        int nbThreads = 0;
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "allThreadsCount"
        };

        if (onJvm == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", JVM_ON_PATTERN);
            return;
        }
        for (int i = 0; i < onJvm.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onJvm[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                if (al.isEmpty()) {
                    /**
                     *  Skip MBean with unexpected attributes.
                     */
                    continue;
                }
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();                  // allThreadsCount
                nbThreads += Integer.parseInt(att.getValue().toString());
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                if (e.getCause() != null) {
                    e.getCause().printStackTrace(System.err);
                }
            }
        }
        data.setThreads(nbThreads);
    }

    /**
     * Polls session info.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollSessions(final ServerData data ) {
        long nbSess = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "activeSessions"
        };

        if (onSessions == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", Manager_ON_PATTERN);
            return;
        }
        for (int i = 0; i < onSessions.length; i++) {
            if (onSessions[i] != null) {
                try {
                    cnx = getMBeanServerConnection();
                    al = cnx.getAttributes(onSessions[i], attIds);
                    releaseMbeanServerConnection();
                    cnx = null;
                    if (al.isEmpty()) {
                        /**
                         *  Skip MBean with unexpected attributes.
                         */
                        continue;
                    }
                    Iterator it = al.iterator();
                    Attribute att = (Attribute) it.next();
                    nbSess += ((Integer) att.getValue()).intValue();
                    data.setValid(true);
                } catch (Exception e) {
                    e.printStackTrace(System.err);
                    if (e.getCause() != null) {
                        e.getCause().printStackTrace(System.err);
                    }
                }
            }
        }
        data.setSessions(nbSess);
    }

    /**
     * Polls global request information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollGlobalRequestProcessor(final ServerData data ) {
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "requestCount",
                "errorCount"
        };

        long reqCount = 0;
        long errCount = 0;

        if (onGRP == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", GlobalRequestProcessor_ON_PATTERN);
            return;
        }
        for (int i = 0; i < onGRP.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onGRP[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                if (al.isEmpty()) {
                    /**
                     *  Skip MBean with unexpected attributes.
                     */
                    continue;
                }
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();              // requestCount
                reqCount += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // errorCount
                errCount += ((Integer) att.getValue()).intValue();
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                if (e.getCause() != null) {
                    e.getCause().printStackTrace(System.err);
                }
            }
        }

        data.setHttpRequests(reqCount);
        data.setHttpErrors(errCount);
    }

    /**
     * Polls thread information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollThreadPool(final ServerData data ) {
        long nbBusy = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "currentThreadsBusy"
        };
        if (onThreadPool == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", ThreadPool_ON_PATTERN);
            return;
        }
        for (int i = 0; i < onThreadPool.length ;i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onThreadPool[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                if (al.isEmpty()) {
                    /**
                     *  Skip MBean with unexpected attributes.
                     */
                    continue;
                }
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();
                // Even if there is no pending request
                // there is always a busy thread =>
                // if busy = 1 don't count it
                long busy = ((Integer) att.getValue()).intValue();
                nbBusy += (busy == 1 ? 0 : busy);
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                if (e.getCause() != null) {
                    e.getCause().printStackTrace(System.err);
                }
            }
        }
        data.setPendingHttp(nbBusy);
    }

    /**
     * Polls the transaction manager.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollJTM(final ServerData data) {
        long txCount = 0;
        long currentTxGauge = 0;
        long rolledbackTx = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "totalCurrentTransactions",
                "totalCommittedTransactions",
                "totalRolledbackTransactions"
        };

        if (onJtm == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", JTAResource_ON_PATTERN);
            return;
        }
        for (int i = 0; i < onJtm.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onJtm[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                if (al.isEmpty()) {
                    /**
                     *  Skip MBean with unexpected attributes.
                     */
                    continue;
                }
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();                          // totalCurrentTransactions
                currentTxGauge += Long.parseLong(att.getValue().toString());
                att = (Attribute) it.next();                                    // totalCommittedTransactions
                txCount += Long.parseLong(att.getValue().toString());
                att = (Attribute) it.next();                                    // totalRolledbackTransactions
                rolledbackTx +=  Long.parseLong(att.getValue().toString());
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                if (e.getCause() != null) {
                    e.getCause().printStackTrace(System.err);
                }
            }
        }
        data.setCommitedTx(txCount);
        data.setRollbackTx(rolledbackTx);
        data.setPendingTx(currentTxGauge);
    }

    /**
     * Polls datasources.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollDataSources(final ServerData data) {
        int cnxGauge = 0;
        int cnxFails = 0;
        int cnxLeaks = 0;
        int cnxWaits = 0;
        int cnxRejects = 0;
        int cnxBusy = 0;
        int cnxServed = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "currentOpened",
                "connectionFailures",
                "connectionLeaks",
                "waiterCount",
                "rejectedOpen",
                "currentBusy",
                "servedOpen"
        };
        if (onDS == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", JDBCDataSource_ON_PATTERN);
            return;
        }
        for (int i = 0; i < onDS.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                al = cnx.getAttributes(onDS[i], attIds);
                releaseMbeanServerConnection();
                cnx = null;
                if (al.isEmpty()) {
                    /**
                     *  Skip MBean with unexpected attributes.
                     */
                    continue;
                }
                Iterator it = al.iterator();
                Attribute att = (Attribute) it.next();              // currentOpened
                cnxGauge += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // connectionFailures
                cnxFails += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // connectionLeaks
                cnxLeaks += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // waiterCount
                cnxWaits += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // rejectedOpen
                cnxRejects += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // currentBusy
                cnxBusy += ((Integer) att.getValue()).intValue();
                att = (Attribute) it.next();                        // servedopen
                cnxServed += ((Integer) att.getValue()).intValue();
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                if (e.getCause() != null) {
                    e.getCause().printStackTrace(System.err);
                }
            }
        }

        data.setDSCnx(cnxGauge);
        data.setDSFails(cnxFails);
        data.setDSLeaks(cnxLeaks);
        data.setDSWaits(cnxWaits);
        data.setDSRejects(cnxRejects);
        data.setDSCnxBusy(cnxBusy);
        data.setDSServed(cnxServed);
    }

    /**
     * Polls entity management information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollEntity(final ServerData data) {
        int entCacheNb = 0;
        int entPoolNb = 0;

        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = {
                "cacheSize",
                "currentInstancePoolSize"
        };
        if (onEntity == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", EntityBean_ON_PATTERN);
            return;
        }
        if (onEntity != null) {
            for (int i = 0; i < onEntity.length ;i++) {
                try {
                    cnx = getMBeanServerConnection();
                    al = cnx.getAttributes(onEntity[i], attIds);
                    releaseMbeanServerConnection();
                    cnx = null;
                    if (al.isEmpty()) {
                        /**
                         *  Skip MBean with unexpected attributes.
                         */
                        continue;
                    }
                    Iterator it = al.iterator();
                    Attribute att = (Attribute) it.next();                  // cacheSize
                    entCacheNb += ((Integer) att.getValue()).intValue();
                    att = (Attribute) it.next();                            // currentInstancePoolSize
                    entPoolNb += ((Integer) att.getValue()).intValue();
                    data.setValid(true);
                } catch (Exception e) {
                    e.printStackTrace(System.err);
                }
            }
        }

        data.setEntityCacheNb(entCacheNb);
        data.setEntityPoolNb(entPoolNb);
    }

    /**
     * Poll stateless session bean information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollSSB(final ServerData data) {
        int ssbCacheNb = 0;
        int ssbPoolNb = 0;
        long ssbCallsNb = 0;

        MBeanServerConnection cnx = null;
        String attId = null;
        Object att = null;
        if (onSSB == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", StatelessSessionBean_ON_PATTERN);
            return;
        }
        for (int i = 0; i < onSSB.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                attId = "cacheSize";
                try {
                    att = cnx.getAttribute(onSSB[i], attId);
                    ssbCacheNb += ((Integer) att).intValue();
                } catch (javax.management.AttributeNotFoundException ae) {
                     logger.warn("MBeanCmd poll -server : attribute {0} not found in {1} MBean", attId, onSSB[i]);
                }
                attId = "poolSize";
                try {
                    att = cnx.getAttribute(onSSB[i], attId);
                    ssbPoolNb += ((Integer) att).intValue();
                } catch (javax.management.AttributeNotFoundException ae) {
                     logger.warn("MBeanCmd poll -server : attribute {0} not found in {1} MBean", attId, onSSB[i]);
                }
                attId = "numberOfCalls";
                try {
                    att = cnx.getAttribute(onSSB[i], attId);
                    ssbCallsNb += ((Long) att).longValue();
                } catch (javax.management.AttributeNotFoundException ae) {
                     logger.warn("MBeanCmd poll -server : attribute {0} not found in {1} MBean", attId, onSSB[i]);
                }
                releaseMbeanServerConnection();
                cnx = null;
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                if (e.getCause() != null) {
                    e.getCause().printStackTrace(System.err);
                }
            }
        }

        data.setSsbCacheNb(ssbCacheNb);
        data.setSsbPoolNb(ssbPoolNb);
        data.setSsbCallsNb(ssbCallsNb);
    }

    /**
     * Pools stateful session bean information.
     *
     * @param data  ServerData to add polled information to.
     */
    private void pollSFB(final ServerData data) {
        int sfbCacheNb = 0;
        int sfbPoolNb = 0;

        String attId = null;
        Object att = null;
        MBeanServerConnection cnx = null;
        if (onSFB == null) {
            logger.warn("MBeanCmd poll -server : cannot poll {0} MBeans", StatefulSessionBean_ON_PATTERN);
            return;
        }
        for (int i = 0; i < onSFB.length; i++) {
            try {
                cnx = getMBeanServerConnection();
                attId = "cacheSize";
                try {
                    att = cnx.getAttribute(onSFB[i], attId);
                    sfbCacheNb += ((Integer) att).intValue();
                } catch (javax.management.AttributeNotFoundException ae) {
                     logger.warn("MBeanCmd poll -server : attribute {0} not found in {1} MBean", attId, onSFB[i]);
                }
                attId = "poolSize";
                try {
                    att = cnx.getAttribute(onSFB[i], attId);
                    sfbPoolNb += ((Integer) att).intValue();
                } catch (javax.management.AttributeNotFoundException ae) {
                     logger.warn("MBeanCmd poll -server : attribute {0} not found in {1} MBean", attId, onSFB[i]);
                }
                releaseMbeanServerConnection();
                cnx = null;
                data.setValid(true);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                if (e.getCause() != null) {
                    e.getCause().printStackTrace(System.err);
                }
            }
        }

        data.setSfbCacheNb(sfbCacheNb);
        data.setSfbPoolNb(sfbPoolNb);
    }

    /**
     * J2EE server.
     */
    private ObjectName[] onServer = null;

    /**
     * JVM.
     */
    private ObjectName[] onJvm = null;

    /**
     * JTM.
     */
    private ObjectName[] onJtm = null;

    /**
     * Global request processors.
     */
    private ObjectName[] onGRP = null;

    /**
     * Thread pools.
     */
    private ObjectName[] onThreadPool = null;

    /**
     * Session managers.
     */
    private ObjectName[] onSessions = null;

    /**
     * DataSource Object Names.
     */
    private ObjectName[] onDS = null;

    /**
     * Entity bean Object Names.
     */
    private ObjectName[] onEntity = null;

    /**
     * StatelessSessionBean Object Names.
     */
    private ObjectName[] onSSB = null;

    /**
     * StatefulSessionBean Object Names.
     */
    private ObjectName[] onSFB = null;
}
