/**
 * JASMINe
 * Copyright (C) 2005-2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: TxSampler.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.util.Iterator;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;

/**
 * Transaction statistics sampler.
 */
public class TxSampler extends Sampler {
    /**
     * Implementation of inherited method.
     *
     * @see Sampler#Sampler(String)
     */
    public TxSampler(final SamplerContext context) {
        super(context);
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#getDefaultOn()
     */
    public String getDefaultOnPattern() {
        return "*:j2eeType=JTAResource,*";
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#newSampleData()
     */
    protected SampleData newSampleData() {
        return new TxData();
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see Sampler#poll(SampleData)
     * @see TxSampler#pollJTM(TxData)
     */
    protected SampleData poll(final SampleData data) {
        pollJTM((TxData) data);
        return data;
    }

    /**
     * Polls the transaction manager.
     *
     * @param data
     *            TxData to add polled information to.
     */
    private void pollJTM(final TxData data) {
        long txCount = 0;
        long currentTxGauge = 0;
        long rolledbackTx = 0;
        long timedOutTx = 0;
        AttributeList al = null;
        MBeanServerConnection cnx = null;
        String[] attIds = { "totalCurrentTransactions", "totalCommittedTransactions", "totalRolledbackTransactions",
                "totalExpiredTransactions" };

        ObjectName[] oNames;
        try {
            oNames = getRefreshedJTAONames();
        } catch (Exception e) {
            // Sample.queryON may return with MalformedObjectNameException
            return;
        }

        if (oNames != null) {
            for (int i = 0; i < oNames.length; i++) {
                try {
                    cnx = getMBeanServerConnection();
                    al = cnx.getAttributes(oNames[i], attIds);
                    releaseMbeanServerConnection();
                    cnx = null;
                    if (al.isEmpty()) {
                        /**
                         *  Skip MBean with unexpected attributes.
                         */
                        continue;
                    }
                    Iterator it = al.iterator();
                    Attribute att = (Attribute) it.next(); // totalCurrentTransactions
                    currentTxGauge += Long.parseLong(att.getValue().toString());
                    att = (Attribute) it.next(); // totalCommittedTransactions
                    txCount += Long.parseLong(att.getValue().toString());
                    att = (Attribute) it.next(); // totalRolledbackTransactions
                    rolledbackTx += Long.parseLong(att.getValue().toString());
                    att = (Attribute) it.next(); // totaltimedOutTransactions
                    timedOutTx += Long.parseLong(att.getValue().toString());
                    data.setValid(true);
                } catch (Exception e) {
                    refreshable = true;
                    data.setValid(false);
                    System.err.println("Error on mbean " + oNames[i] + " for " + context.getName());
                    e.printStackTrace(System.err);
                    if (e.getCause() != null) {
                        e.getCause().printStackTrace(System.err);
                    }
                }
            }
            data.setCommittedTxCnt(txCount);
            data.setRolledbackTxCnt(rolledbackTx);
            data.setPendingTxGauge(currentTxGauge);
            data.setTimedOutTxCnt(timedOutTx);
            data.setServerInfo(getName(), getJmxUrl(), getServer(), getDomain());
            data.setCmdId(getCmdId());
            if (oNames.length > 0) {
                data.setObjectName(oNames[0]);
            }
        }
    }

    private ObjectName[] getRefreshedJTAONames() throws Exception {

        ObjectName[] oNames = null;
        Set<ObjectName> son = context.getOnames();
        if (son == null) {
            refreshable = true;
        }
        if (refreshable) {
            // note: queryON updates the context too
            oNames = queryON(this.getOnPattern());
            if (oNames != null) {
                refreshable = false;
            }
        } else {
            if ((son != null) && !son.isEmpty()) {
                oNames = (ObjectName[]) son.toArray(new ObjectName[son.size()]);
            }
        }

        return oNames;
    }

    private boolean refreshable = false;
}
