/**
 * JASMINe
 * Copyright (C) 2005-2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XSampleData.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;


/**
 * SampleData where sampling operates on a Set.
 */
public class XSampleData extends SampleData {
    /**
     * Implementation of inherited abstract method.
     *
     * Calls @link{SampleData#compute(SampleData)} on each member of
     * {@link XSampleData#samples}.
     */
    public void compute(final SampleData needs_cast) throws Exception {
        XSampleData prev = (XSampleData) needs_cast;
        Iterator<ObjectName> it = samples.keySet().iterator();
        while (it.hasNext()) {
            ObjectName on = it.next();
            if (prev != null) {
                this.get(on).compute(prev.get(on));
            } else {
                this.get(on).compute(null);
            }
        }
    }

    /**
     * Prints out all data in attributes based on the print header.
     *
     * @see SampleData#getPrintHeader()
     * @see SampleData#getInnerPrintData()
     *
     * @param out  Stream to print data values into.
     */
    public final void printData(final PrintStream out) {

        Iterator<SampleData> it = samples.values().iterator();
        while (it.hasNext()) {
            SampleData data = it.next();
            data.printData(out);
        }
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintHeader()
     */
    protected String getInnerPrintHeader() {
        String header = "";
        Iterator<SampleData> it = samples.values().iterator();
        if (it.hasNext()) {
            SampleData sample = it.next();
            header += sample.getInnerPrintHeader();
        }
        return header;
    }

    /**
     * Implementation of inherited abstract method.
     *
     * @see SampleData#getInnerPrintData()
     */
    protected String getInnerPrintData() {
        String data = "";
        Iterator<SampleData> it = samples.values().iterator();
        if (it.hasNext()) {
            SampleData sample = it.next();
            data += sample.getInnerPrintData();
        }
        return data;
    }

    /**
     * Puts an object in {@link XSampleData#samples}.
     *
     * @param on    ObjectName to use.
     * @param data  SampleData instance to link with that name.
     *
     * @return  data that's just been added.
     */
    public SampleData put(final ObjectName on, final SampleData data) {
        return samples.put(on, data);
    }

    /**
     * Gets the SampleData associated with a given ObjectName.
     *
     * @param on  ObjectName to look for.
     *
     * @return  SampleData instance that's linked with that object name.
     */
    public SampleData get(final ObjectName on) {
        return samples.get(on);
    }

    /**
     * @return SampleData included in this instance
     */
    public Set<SampleData> getSampleData() {
        return new HashSet<SampleData>(samples.values());
    }

    /**
     * SampleData included in this instance.
     *
     * TODO: Implementation note: ObjectName in Java 4 or 5 doesn't implement
     *       Comparable, therefore TreeMap will throw a ClassCastException.
     *       That problem has been fixed in Java 6.
     */
    private Map<ObjectName, SampleData> samples = new HashMap<ObjectName, SampleData>();
}
