/**
 * JASMINe
 * Copyright (C) 2005-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XServerSampler.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.sampling;

import java.io.IOException;
import java.io.PrintStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;

import javax.management.MalformedObjectNameException;

import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAp;
import org.ow2.jasmine.monitoring.mbeancmd.JmxHelper;
import org.ow2.jasmine.monitoring.mbeancmd.context.SamplerContext;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
/**
 * Server sampler.
 */
public class XServerSampler {
    private Log logger = LogFactory.getLog(this.getClass());

    private boolean stopped = false;

    /**
     * Constructor: saves the sampler type.
     *
     * @param type
     *            Sampler type
     */
    public XServerSampler(final String type) {
        sampleType = type;
    }

    public void stopProcessing() {
        stopped = true;
    }

    /**
     * Adds a manangee.
     *
     * @param name
     *            Managee name.
     * @param jmxUrl
     *            JMX URL for managee.
     */
    public void addManagee(final String name, final String jmxUrl) {
        manageeURLs.put(name, jmxUrl);
    }

    /**
     * @param objectNameAsString
     *            Object name filter pattern.
     */
    public void setOnPattern(final String objectNameAsString) {
        onPattern = objectNameAsString;
    }

    /**
     * Polls the MBean.
     *
     * @param period
     *            Period of polling
     * @param out
     *            Stream to output results into.
     */
    public void mbeanPoll(final long period, final PrintStream out) throws Exception{
        // Infinite sampling loop
        boolean printHeader = true;
        boolean setrunning = false;
        while (! stopped) {
            SamplerContext context = null;
            SampleData data = null;
            Sampler sampler;

            for (int i = 0; i < contexts.length; i++) {
                context = contexts[i];
                sampler = context.getSampler();
                try {
                    String thePattern = onPattern;
                    if (onPattern == null) {
                        thePattern = sampler.getOnPattern();
                    }
                    // Get MBean ObjectNames
                    try {
                        logger.debug("Get MBean ObjectNames for " + thePattern);
                        context.updateOnames(thePattern);
                    } catch (MalformedObjectNameException moe) {
                        // cannot do sampling on a wrong pattern
                        commandDispatcher.setFailed(moe.getMessage());
                        throw new Exception("moe.getMessage()");
                    } catch (IOException ioe) {
                        sampler.setHasFailed(true);
                        throw ioe;
                    }

                    if (! setrunning) {
                        commandDispatcher.setRunning();
                        setrunning = true;
                    }
                    // Construct SampleData
                    data = null;
                    try {
                        // SampleData on the previous iteration
                        SampleData prevData = context.getPreviousData();
                        // Get new sample data
                        data = sampler.sample(prevData);

                        boolean printed = printData(data, out, printHeader);
                        if (printed) {
                            printHeader = false;
                            if (!context.isStarted()) {
                                context.setStarted(true);
                            }
                        }
                    } catch (Exception e) {
                        if (context.isStarted()) {
                            System.err.println("Error sampling managee " + context.getName());
                        }
                    }
                    context.setPreviousData(data);
                } catch (Exception me) {
                    // Already logged in the throw
                } finally {
                    // ???
                   // context.getJmxap().releaseMBeanServerConnection();
                }
            }

            try {
                Thread.sleep(period * SECOND);
            } catch (InterruptedException e) {
                break;
            }
        }
    }

    /**
     * Print out data
     * @param data SampleData to print
     * @param out output
     * @param printHeader if true, header must be printed and its value passes to false.
     * @return true of data was really printed
     */
    private boolean printData(final SampleData data, final PrintStream out, boolean printHeader) {
        if (data == null) {
            return false;
        }
        boolean emptyData = false;
        if (data instanceof XSampleData) {
             XSampleData xData = (XSampleData) data;
             emptyData = xData.getSampleData().isEmpty();
        }
        if (emptyData) {
            return false;
        }
        if (printHeader) {
            out.println(data.getPrintHeader());
            printHeader = false;
        }
        data.printData(out);
        return true;
    }
    /**
     * Create a SamplerContext for a target which implies also creating the sampler.
     * @param name the target name
     * @return the SamplerContext of the given target
     */
    private SamplerContext createSamplerContext(final String name) throws Exception {
        SamplerContext samplerContext = new SamplerContext();
        samplerContext.setName(name);
        String jmxUrl = JmxHelper.getJmxUrl(name);
        samplerContext.setJmxUrl(jmxUrl);
        samplerContext.setJmxap(new JmxAp(jmxUrl, commandDispatcher));
        if (jmxUrl != null && jmxUrl.equals(name)) {
            // Support use-case where -target has jmxUrl value
            samplerContext.setAdminNaming(false);
        }
        samplerContext.setUpManagentInfos();
        if (samplerContext.getServer() == null) {
            samplerContext.setServer("unknown_server_name");
            samplerContext.setDomain("unknown_domain");
        }
        samplerContext.setCmdId(cmdid);
        // create and init the sampler
        Sampler sampler = SamplerFactory.newSampler(sampleType, onPattern, samplerContext);
        samplerContext.setSampler(sampler);
        return samplerContext;
    }

    /**
     * Start processing. Will never return, except on failure.
     *
     * @param period
     *            Poll period.
     * @param out
     *            Stream to output results into.
     * @param cmdDispatcher
     *            The command dispatcher is MBeanCmd is not autonomous
     */
    public final void process(final long period, final PrintStream out) throws Exception {
        // Create a SamplerContext with associated Sampler for each target
        List<SamplerContext> samplerContextList = new LinkedList<SamplerContext>();
        Iterator<Entry<String, String>> it = manageeURLs.entrySet().iterator();
        while (it.hasNext()) {
            /**
             * Implementation note on manageeURLs use:
             * We use only the key but not the value (the jmxUrl)
             *  In order to get the jmxUrl, the JmxHelper is used in the createSamplerContext():
             *  		JmxHelper.getJmxUrl(name)
             *  This could be probably simplified.
             */
            Entry<String, String> e = it.next();
            String targetName = e.getKey();
            try {
                SamplerContext ctx = createSamplerContext(targetName);
                samplerContextList.add(ctx);
            } catch (Exception ex) {
                // Could not create SamplerContext because of MalformedObjectNameException
                commandDispatcher.setFailed(ex.getMessage());
                throw ex;
            }
        }
        contexts = samplerContextList.toArray(new SamplerContext[samplerContextList.size()]);
        // process each target by polling the MBeans
        mbeanPoll(period, out);
    }

    private CommandDispatcher commandDispatcher = null;


    public void setCommandDispatcher(CommandDispatcher commandDispatcher) {
        this.commandDispatcher = commandDispatcher;
    }

    public void setCmdId(String c) {
        cmdid = c;
    }

    public String getCmdId() {
        return cmdid;
    }

    /**
     * URLs of managees.
     */
    private Map<String, String> manageeURLs = new TreeMap<String, String>();

    /**
     * Sample type.
     */
    private String sampleType = null;

    /**
     * Object name pattern.
     */
    private String onPattern = null;

    /**
     * Array of contexts.
     */
    private SamplerContext[] contexts = null;

    /**
     * Cmd Id
     */
    private String cmdid = null;

    /**
     * A second in millisecond.
     */
    private static final long SECOND = 1000;
}
