/**
 * JASMINe
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Lewys.java 4225 2009-07-01 15:15:52Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.IOException;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import javax.management.MalformedObjectNameException;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.objectweb.lewys.common.exceptions.NoResourceToProbeException;
import org.objectweb.lewys.probe.Probe;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.graph.Grapher;
import org.ow2.jasmine.monitoring.mbeancmd.jasmine.JasmineConnector;

/**
 * Manipulate LeWYS probes (for now, just this for network). Largely inspired from a Clif probe.
 * TODO: Support of all LeWYS probes.
 * @author Loris Bouzonnet
 */
public class Lewys extends AbstractCommand {

    public Lewys() {
        setOptions();
    }

    /**
     * Tests the Stat implementation.
     *
     * @param args  Arguments to pass to Stat#setArgs.
     */
    public static void main(final String[] args) {
        Lewys p = new Lewys();
        p.setArgs("lewys", args);
        p.exec(null);
    }

    @Override
    public int exec(final CommandDispatcher cmdDispatcher) {
        try {
            if (System.getProperty("os.name").equalsIgnoreCase("linux")) {
                probe = new org.objectweb.lewys.probe.linux.NetworkProbe();
            } else if (System.getProperty("os.name").startsWith("Windows")) {
                probe = new org.objectweb.lewys.probe.windows.NetworkProbe();
            } else if (System.getProperty("os.name").equals("Mac OS X")) {
                probe = new org.objectweb.lewys.probe.macosx.NetworkProbe();
            } else {
                throw new RuntimeException("No Network probe available for " + System.getProperty("os.name"));
            }
        } catch (NoResourceToProbeException ex) {
            throw new RuntimeException("Can't set Network probe", ex);
        }
        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }

        /*
         * out (or pout) is : - the output of the sampler - the input for the
         * Outer to print into System.out or into a file
         *
         * sink is: - the sink of the Outer - the input for the graph, if any
         */
        PipedOutputStream out = new PipedOutputStream();
        pout = new PrintStream(out);

        Outer outer = null;
        try {
            if (this.outputFilePath != null) {
                outer = new Outer(out, new File(outputFilePath));
            } else if (this.jasmineURI == null) {
                outer = new Outer(out, System.out);
            }

            if (this.jasmineURI != null) {
                if(outer == null) {
                    outer = JasmineConnector.connect(out, this.jasmineURI);
                } else {
                    Outer oldOuter = outer;
                    outer = JasmineConnector.connect(outer, this.jasmineURI);
                    new Thread(oldOuter).start();
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        if (this.graphDef != null) {
            try {
                PipedOutputStream sink = new PipedOutputStream();
                outer.setSink(sink);
                Grapher gr = new Grapher(sink, this.graphDef);
                gr.start();
            } catch (IOException e) {
                e.printStackTrace(System.err);
            }
        }
        new Thread(outer).start();

        process();
        return 0;
    }

    @Override
    public String summary() {
        return "Periodically probes with LeWYS.";
    }


    /**
     * Parses the command line arguments into {@link Stat#commandLine}.
     *
     * @param args  Arguments to parse.
     *
     * @throws ParseException                If parsing fails.
     * @throws MalformedObjectNameException  Object name given in the command
     *                                       line is invalid.
     */
    public void parseCommandLine(final String[] args) throws ParseException,
            MalformedObjectNameException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);

        arg_probe_config = Arrays.asList(commandLine.getOptionValues("a"));

        if (commandLine.hasOption("p")) {
            setPeriod();
        }

        if (commandLine.hasOption("graph")) {
            this.graphDef = commandLine.getOptionValue("graph");
        }
        if (commandLine.hasOption("f")) {
            this.outputFilePath = commandLine.getOptionValue("f");
        }
        if (commandLine.hasOption("jasmine")) {
            this.jasmineURI = commandLine.getOptionValue("jasmine");
        }

        if (commandLine.hasOption("s")) {
            this.separator = commandLine.getOptionValue("s");
            Outer.setSeparator(separator);
        }

        if(commandLine.hasOption("n")) {
            this.serverName = commandLine.getOptionValue("n");
        }
        if(commandLine.hasOption("d")) {
            this.domainName = commandLine.getOptionValue("d");
        }
    }

    private void process() {
        configProbe();
        lewysPoll();
    }

    private void configProbe() {

        String [] resourceNames = probe.getResourceNames();
        resourceIds = new int[NBRESOURCE];
        // the ArrayList specifies the inner parameters of the probe
        if (!arg_probe_config.isEmpty()) {
            // parameter 'networkcardName' appears in position zero.
            String networkcardName = (String)arg_probe_config.get(0);
            // locate the networkcard index in LeWYS resourceNames Array in order to get ressource ids.
            int networkcard_index = 0;
            while (networkcard_index < resourceNames.length && !(resourceNames[networkcard_index].endsWith(networkcardName + " bytes received")))
                networkcard_index++;

            if (networkcard_index < resourceNames.length) {
                resourceIds[0]  =  0 + networkcard_index;
                resourceIds[1]  =  1 + networkcard_index;
                resourceIds[2]  =  8 + networkcard_index;
                resourceIds[3]  =  9 + networkcard_index;
                return;
            }
            }

        String exceptionMessage = "";
        exceptionMessage+="\n. Bad probe parameters: A correct Networkcard name must be provided in section BladeArgument [pollingPeriod pollingTime Networkcardname]!!\n.\n";
        exceptionMessage+=". Networkcard Names for this host are:\n";
        int totalNbResource = 16;
        if (System.getProperty("os.name").equals("Mac OS X"))
        {
            totalNbResource = 17;
        }

        for( int j = 0 ; j < resourceNames.length; j+= totalNbResource) {
            String name = resourceNames[j].substring(0,resourceNames[j].length()-"bytes received".length());
            exceptionMessage+=". "+name+"\n";
        }
        exceptionMessage+=	".\n. HINT: If a NetworkCardName is too long, provide only the last discriminating characters.\n";
        exceptionMessage+=	". For Instance: \n\n" +
                            ".                  \"card1 3Gbits/s\"\n" +
                            ".                    in place of \n" +
                            ". \"My PowerFul NetworkCard FULL Duplex card1 3Gbits/s\" \n" +
                            ".\n. ATTENTION: The provided NetWorkCard Name must not comport character \";\" .\n";
        throw new RuntimeException (exceptionMessage);
    }

    private void lewysPoll() {
        boolean goOn = true;
        while (goOn) {
            try {
                long[] values = probe.getValue(resourceIds);
                if (previousValues == null) {
                    previousValues = values;
                } else {
                        long [] result = new long [values.length];
                        result [0] = (values[0]-previousValues[0])/1000; // Convert Bytes to KBytes
                        result [1] = values[1]-previousValues[1];
                        result [2] = (values[2]-previousValues[2])/1000; // Convert Bytes to KBytes
                        result [3] = values[3]-previousValues[3];
                        long t = System.currentTimeMillis();
                        printResult(t, result);
                        previousValues = values;
                }
            } catch (Exception ex) {
                ex.printStackTrace(System.err);
            }
            // Sleep for a while
            try {
                Thread.sleep(period * 1000);
            } catch (InterruptedException e) {
                goOn = false;
            }
        }

    }

    /**
     * Prints a given header.
     *
     * @param attl  Header attributes.
     *
     * @return  Printed header.
     */
    private String getMBeanPollHeader() {
        if (header == null) {
            StringBuffer buf = new StringBuffer(512);
            buf.append("date");
            buf.append(separator);
            buf.append("time");
            buf.append(separator);
            buf.append("sname");
            buf.append(separator);
            buf.append("server");
            buf.append(separator);
            buf.append("domain");
            buf.append(separator);
            buf.append("mbean");
            buf.append(separator);
            buf.append("KbytesReceived");
            buf.append(separator);
            buf.append("packetsReceived");
            buf.append(separator);
            buf.append("KbytesTransmit");
            buf.append(separator);
            buf.append("packetsTransmit");
            header = buf.toString();
        }
        return header;
    }

    private void printResult(final long t, final long[] result) {
        if (header == null) {
            pout.println(getMBeanPollHeader());
        }

        Date d = new Date(t);
        pout.print(SIMPLEDATEFORMAT.format(d));
        pout.print(separator);
        pout.print(t);
        pout.print(separator);
        pout.print("na");
        pout.print(separator);
        pout.print(serverName);
        pout.print(separator);
        pout.print(domainName);
        pout.print(separator);
        pout.print("network");
        pout.print(separator);
        pout.print(result[0]);
        pout.print(separator);
        pout.print(result[1]);
        pout.print(separator);
        pout.print(result[2]);
        pout.print(separator);
        pout.print(result[3]);

        pout.println();
    }

    /**
     * Sets the period if the command line has the "p" option.
     *
     * @throws NumberFormatException  Number after the "p" option is invalid.
     */
    private void setPeriod() throws NumberFormatException {
        period = Long.parseLong(commandLine.getOptionValue("p"));
    }

    /**
     * Sets all options that are parseable from the command line.
     *
     * @see Poll#options
     */
    private void setOptions() {
        options = new Options();
        options.addOption("v", "verbose", false, "Verbose mode");

        // Optional attribute list
        Option atts = new Option("a", "atts", true, "attributes to poll");
        atts.setRequired(false);
        atts.setOptionalArg(true);
        atts.setArgs(Option.UNLIMITED_VALUES);
        atts.setArgName("attributes");
        options.addOption(atts);

        // Period
        Option period = new Option("p", "period", true, "Polling period");
        atts.setRequired(false);
        period.setArgName("period");
        period.setArgs(1);
        options.addOption(period);

        // graph output
        Option graph = new Option("graph", "graph", true,
                "Enable graphical output");
        graph.setRequired(false);
        graph.setArgName("graphDefinition");
        graph.setArgs(1);
        options.addOption(graph);

        // file output
        Option file = new Option("f", "file", true,
                "Send output to file instead of stdout");
        file.setRequired(false);
        file.setArgName("path");
        file.setArgs(1);
        options.addOption(file);

        // jasmine connector output
        Option jasmine = new Option("jasmine", "jasmine", true,
                "Output logged data to jasmine (will disable stdout)");
        jasmine.setRequired(false);
        jasmine.setArgName("jasmineURI");
        jasmine.setArgs(1);
        options.addOption(jasmine);

        // Field separator
        Option separator = new Option("s", "separator", true,
                "Set the column separator");
        separator.setRequired(false);
        separator.setArgName("separator");
        separator.setArgs(1);
        options.addOption(separator);

        Option server = new Option("n", "name", true,
                "Server name");
        server.setRequired(false);
        server.setArgName("serverName");
        server.setArgs(1);
        options.addOption(server);

        Option domain = new Option("d", "domain", true,
                "Domain name");
        domain.setRequired(false);
        domain.setArgName("domainName");
        domain.setArgs(1);
        options.addOption(domain);
    }


    /**
     * List of options that should be parsed from the command line.
     */
    private Options options = null;

    /**
     * Command line arguments.
     */
    private CommandLine commandLine = null;

    private Probe probe;

    private Object serverName;

    private Object domainName;

    private int[] resourceIds;

    private int NBRESOURCE = 4;

    private long[] previousValues = null;

    private List arg_probe_config;

    /**
     * Poll period in seconds.
     */
    private long period = 10;

    /**
     * Graph definitions, null for no graph output.
     */
    private String graphDef = null;

    /**
     * Output file path, null for no file output.
     */
    private String outputFilePath = null;

    /**
     * URI of the JASMINe event switch, null for no JASMINe output.
     */
    private String jasmineURI = null;

    /**
     * Output stream.
     */
    private PrintStream pout = null;

    /**
     * Separator to use.
     */
    private String separator = ";";

    /**
     * Header to use.
     */
    private String header = null;
}
