/**
 * JASMINe
 * Copyright (C) 2008-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Disk.java 8094 2011-04-29 14:55:16Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.util.Date;

import org.objectweb.lewys.common.exceptions.NoResourceToProbeException;

/**
 * Lewys DiskProbe
 */
public class Disk extends Lewys {
    /**
     * Array of reource Ids to be polled with the Lewys probe
     */
    protected int[] resourceIds = new int[4];

    /**
     * Tests the command.
     * @param args Arguments to pass to command
     */
    public static void main(final String[] args) {
        Lewys p = new Network();
        p.setArgs("disk", args);
        p.exec(null);
    }

    /**
     * Compute resourceIds to be polled
     */
    protected void configProbe() {
        // Create the Probe
        try {
            switch (osType) {
                case LINUX_2_4:
                    probe = new org.objectweb.lewys.probe.linux.DiskProbe();
                    break;
                case LINUX_2_6:
                    probe = new org.objectweb.lewys.probe.linux.Linux2_6DiskProbe();
                    break;
                case WINDOWS:
                    probe = new org.objectweb.lewys.probe.windows.DiskProbe();
                    break;
                case MAC_OS_X:
                    probe = new org.objectweb.lewys.probe.macosx.DiskProbe();
                    break;
                default:
                    throw new RuntimeException("No DiskProbe available for " + System.getProperty("os.name"));
            }
        } catch (NoResourceToProbeException ex) {
            throw new RuntimeException("Can't set Disk probe", ex);
        }

        //String[] resourceNames = probe.getResourceNames();
        //for (int i = 0; i < resourceNames.length; i++) {
        //    System.out.println(resourceNames[i]);
        //}

        String diskName = (String) arg_probe_config.get(0);

        // Compute ids of the 4 resources to poll
        try {
            resourceIds[0] =  probe.getResourceId(diskName + " write sectors");
            resourceIds[1] =  probe.getResourceId(diskName + " read sectors");
        } catch (Exception e) {
            throw new RuntimeException("Cannot get Disk Info for " + diskName + " : " + e);
        }
    }

    protected void lewysPoll() {
        boolean goOn = true;
        long[] previousValues = null;
        while (goOn) {
            try {
                long[] values = probe.getValue(resourceIds);
                if (previousValues == null) {
                    previousValues = values;
                } else {
                    long[] result = new long[values.length];
                    result[0] = values[0] - previousValues[0];
                    result[1] = values[1] - previousValues[1];
                    long t = System.currentTimeMillis();
                    printResult(t, result);
                    previousValues = values;
                }
            } catch (Exception ex) {
                ex.printStackTrace(System.err);
            }
            // Sleep for the time of the period
            try {
                Thread.sleep(period * 1000);
            } catch (InterruptedException e) {
                goOn = false;
            }
        }
    }

    /**
     * Prints a given header.
     *
     * @return Printed header.
     */
    private String getMBeanPollHeader() {
        if (header == null) {
            StringBuffer buf = new StringBuffer(512);
            buf.append("date");
            buf.append(separator);
            buf.append("time");
            buf.append(separator);
            buf.append("sname");
            buf.append(separator);
            buf.append("server");
            buf.append(separator);
            buf.append("domain");
            buf.append(separator);
            buf.append("mbean");
            buf.append(separator);
            buf.append("sectors written");
            buf.append(separator);
            buf.append("sectors read");
            header = buf.toString();
        }
        return header;
    }

    private void printResult(final long t, final long[] result) {
        if (header == null) {
            pout.println(getMBeanPollHeader());
        }

        Date d = new Date(t);
        pout.print(SIMPLEDATEFORMAT.format(d));
        pout.print(separator);
        pout.print(t);
        pout.print(separator);
        pout.print("na");
        pout.print(separator);
        pout.print(serverName);
        pout.print(separator);
        pout.print(domainName);
        pout.print(separator);
        pout.print("disk");
        pout.print(separator);
        pout.print(result[0]);
        pout.print(separator);
        pout.print(result[1]);
        pout.println();
    }
}
