/**
 * JASMINe
 * Copyright (C) 2008-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Lewys.java 8094 2011-04-29 14:55:16Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.io.File;
import java.io.IOException;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.List;

import javax.management.MalformedObjectNameException;

import org.apache.commons.cli.*;
import org.objectweb.lewys.probe.Probe;
import org.ow2.jasmine.monitoring.mbeancmd.AbstractCommand;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher;
import org.ow2.jasmine.monitoring.mbeancmd.Outer;
import org.ow2.jasmine.monitoring.mbeancmd.graph.Grapher;
import org.ow2.jasmine.monitoring.mbeancmd.jasmine.JasmineConnector;

/**
 * Manipulate LeWYS probes
 * @author Loris Bouzonnet
 */
public abstract class Lewys extends AbstractCommand {

    /**
     * Supported OS
     */
    protected int osType;
    protected final static int LINUX_2_4 = 10;
    protected final static int LINUX_2_6 = 11;
    protected final static int WINDOWS = 20;
    protected final static int MAC_OS_X = 30;

    /**
     * List of options that should be parsed from the command line.
     */
    protected Options options = null;

    /**
     * Command line arguments.
     */
    protected CommandLine commandLine = null;

    /**
     * List of Attributes (-a option)
     */
    protected List arg_probe_config;

    /**
     * Poll period in seconds.
     */
    protected long period = 10;

    /**
     * Graph definitions, null for no graph output.
     */
    protected String graphDef = null;

    /**
     * Output file path, null for no file output.
     */
    protected String outputFilePath = null;

    /**
     * URI of the JASMINe event switch, null for no JASMINe output.
     */
    protected String jasmineURI = null;

    /**
     * Separator to use.
     */
    protected String separator = ";";

    /**
     * Server name to be printed in results
     */
    protected Object serverName;

    /**
     * Domain name to be printed in results
     */
    protected Object domainName;

    /**
     * Lewys Probe
     */
    protected Probe probe;

    /**
     * Output stream.
     */
    protected PrintStream pout = null;

    /**
     * Header to use.
     */
    protected String header = null;

    /**
     * Constructor
     */
    public Lewys() {
        setOptions();
    }

    /**
     * Executes the command.
     *
     * @return 0 if succeeded, an error code otherwise.
     */
    @Override
    public int exec(final CommandDispatcher cmdDispatcher) {

        // Check System version
        if (System.getProperty("os.name").equalsIgnoreCase("linux")) {
            if (System.getProperty("os.version").startsWith("2.6")) {
                osType = LINUX_2_6;
            } else {
                // Assumes that other versions will be as 2.4
                osType = LINUX_2_4;
            }
        }
        else if (System.getProperty("os.name").startsWith("Windows")) {
            osType = WINDOWS;
        }
        else if (System.getProperty("os.name").equals("Mac OS X")) {
            osType = MAC_OS_X;
        }
        //System.out.println(System.getProperty("os.name")+" "+System.getProperty("os.version"));

        // Parse command line args
        try {
            parseCommandLine(arguments);
        } catch (Exception e) {
            e.printStackTrace();
            return 1;
        }

        /*
         * out (or pout) is : - the output of the sampler - the input for the
         * Outer to print into System.out or into a file
         *
         * sink is: - the sink of the Outer - the input for the graph, if any
         */
        PipedOutputStream out = new PipedOutputStream();
        pout = new PrintStream(out);
        Outer outer = null;
        try {
            if (this.outputFilePath != null) {
                outer = new Outer(out, new File(outputFilePath));
            } else if (this.jasmineURI == null) {
                outer = new Outer(out, System.out);
            }

            if (this.jasmineURI != null) {
                if (outer == null) {
                    outer = JasmineConnector.connect(out, this.jasmineURI);
                } else {
                    Outer oldOuter = outer;
                    outer = JasmineConnector.connect(outer, this.jasmineURI);
                    new Thread(oldOuter).start();
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        if (this.graphDef != null) {
            try {
                PipedOutputStream sink = new PipedOutputStream();
                outer.setSink(sink);
                Grapher gr = new Grapher(sink, this.graphDef);
                gr.start();
            } catch (IOException e) {
                e.printStackTrace(System.err);
            }
        }
        new Thread(outer).start();

        // Process
        configProbe();
        lewysPoll();
        return 0;
    }

    @Override
    public String summary() {
        return "Periodically probes with LeWYS.";
    }

    protected abstract void configProbe();
    protected abstract void lewysPoll();

    /**
     * Parses the command line arguments into {@link Stat#commandLine}.
     *
     * @param args Arguments to parse.
     * @throws ParseException               If parsing fails.
     * @throws MalformedObjectNameException Object name given in the command
     *                                      line is invalid.
     */
    protected void parseCommandLine(final String[] args) throws ParseException,
            MalformedObjectNameException {
        BasicParser bp = new BasicParser();
        commandLine = bp.parse(options, args);

        arg_probe_config = Arrays.asList(commandLine.getOptionValues("a"));
        if (arg_probe_config.isEmpty()) {
            throw new RuntimeException("Empty list of args");
        }

        if (commandLine.hasOption("p")) {
            period = Long.parseLong(commandLine.getOptionValue("p"));
        }

        if (commandLine.hasOption("graph")) {
            this.graphDef = commandLine.getOptionValue("graph");
        }
        if (commandLine.hasOption("f")) {
            this.outputFilePath = commandLine.getOptionValue("f");
        }
        if (commandLine.hasOption("jasmine")) {
            this.jasmineURI = commandLine.getOptionValue("jasmine");
        }
        if (commandLine.hasOption("s")) {
            this.separator = commandLine.getOptionValue("s");
            Outer.setSeparator(separator);
        }
        if (commandLine.hasOption("n")) {
            this.serverName = commandLine.getOptionValue("n");
        }
        if (commandLine.hasOption("d")) {
            this.domainName = commandLine.getOptionValue("d");
        }
    }


    /**
     * Sets all options that are parseable from the command line.
     *
     * @see Poll#options
     */
    protected void setOptions() {
        options = new Options();
        options.addOption("v", "verbose", false, "Verbose mode");

        // Optional attribute list
        Option atts = new Option("a", "atts", true, "attributes to poll");
        atts.setRequired(false);
        atts.setOptionalArg(true);
        atts.setArgs(Option.UNLIMITED_VALUES);
        atts.setArgName("attributes");
        options.addOption(atts);

        // Period
        Option period = new Option("p", "period", true, "Polling period");
        atts.setRequired(false);
        period.setArgName("period");
        period.setArgs(1);
        options.addOption(period);

        // graph output
        Option graph = new Option("graph", "graph", true,
                "Enable graphical output");
        graph.setRequired(false);
        graph.setArgName("graphDefinition");
        graph.setArgs(1);
        options.addOption(graph);

        // file output
        Option file = new Option("f", "file", true,
                "Send output to file instead of stdout");
        file.setRequired(false);
        file.setArgName("path");
        file.setArgs(1);
        options.addOption(file);

        // jasmine connector output
        Option jasmine = new Option("jasmine", "jasmine", true,
                "Output logged data to jasmine (will disable stdout)");
        jasmine.setRequired(false);
        jasmine.setArgName("jasmineURI");
        jasmine.setArgs(1);
        options.addOption(jasmine);

        // Field separator
        Option separator = new Option("s", "separator", true,
                "Set the column separator");
        separator.setRequired(false);
        separator.setArgName("separator");
        separator.setArgs(1);
        options.addOption(separator);

        Option server = new Option("n", "name", true,
                "Server name");
        server.setRequired(false);
        server.setArgName("serverName");
        server.setArgs(1);
        options.addOption(server);

        Option domain = new Option("d", "domain", true,
                "Domain name");
        domain.setRequired(false);
        domain.setArgName("domainName");
        domain.setArgs(1);
        options.addOption(domain);
    }

}
