/**
 * JASMINe
 * Copyright (C) 2008-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Network.java 8094 2011-04-29 14:55:16Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.commands;

import java.util.Date;

import org.objectweb.lewys.common.exceptions.NoResourceToProbeException;

/**
 * Lewys NetworkProbe
 */
public class Network extends Lewys {

    /**
     * Array of reource Ids to be polled with the Lewys probe
     */
    protected int[] resourceIds = new int[4];

    /**
     * Tests the command.
     * @param args Arguments to pass to command
     */
    public static void main(final String[] args) {
        Lewys p = new Network();
        p.setArgs("network", args);
        p.exec(null);
    }

    /**
     * Compute resourceIds to be polled
     */
    protected void configProbe() {
        // Create the Probe
        try {
            switch (osType) {
                case LINUX_2_4:
                case LINUX_2_6:
                    probe = new org.objectweb.lewys.probe.linux.NetworkProbe();
                    break;
                case WINDOWS:
                    probe = new org.objectweb.lewys.probe.windows.NetworkProbe();
                    break;
                case MAC_OS_X:
                    probe = new org.objectweb.lewys.probe.macosx.NetworkProbe();
                    break;
                default:
                    throw new RuntimeException("No Network probe available for " + System.getProperty("os.name"));
            }
        } catch (NoResourceToProbeException ex) {
            throw new RuntimeException("Can't set Network probe", ex);
        }

        //String[] resourceNames = probe.getResourceNames();
        //for (int i = 0; i < resourceNames.length; i++)
        //    System.out.println(resourceNames[i]);

        // parameter 'networkcardName' appears in position zero.
        String networkcardName = (String) arg_probe_config.get(0);

        // Compute ids of the 4 resources to poll
        try {
            resourceIds[0] =  probe.getResourceId(networkcardName + " bytes received");
            resourceIds[1] =  probe.getResourceId(networkcardName + " packets received");
            resourceIds[2] =  probe.getResourceId(networkcardName + " bytes transmit");
            resourceIds[3] =  probe.getResourceId(networkcardName + " packets transmit");
        } catch (Exception e) {
            throw new RuntimeException(errorMessage());
        }
    }

    protected void lewysPoll() {
        boolean goOn = true;
        long[] previousValues = null;
        while (goOn) {
            try {
                long[] values = probe.getValue(resourceIds);
                if (previousValues == null) {
                    previousValues = values;
                } else {
                    long[] result = new long[values.length];
                    result[0] = (values[0] - previousValues[0]) / 1000; // Convert Bytes to KBytes
                    result[1] = values[1] - previousValues[1];
                    result[2] = (values[2] - previousValues[2]) / 1000; // Convert Bytes to KBytes
                    result[3] = values[3] - previousValues[3];
                    long t = System.currentTimeMillis();
                    printResult(t, result);
                    previousValues = values;
                }
            } catch (Exception ex) {
                ex.printStackTrace(System.err);
            }
            // Sleep for the time of the period
            try {
                Thread.sleep(period * 1000);
            } catch (InterruptedException e) {
                goOn = false;
            }
        }
    }

    /**
     * Prints a given header.
     *
     * @return Printed header.
     */
    private String getMBeanPollHeader() {
        if (header == null) {
            StringBuffer buf = new StringBuffer(512);
            buf.append("date");
            buf.append(separator);
            buf.append("time");
            buf.append(separator);
            buf.append("sname");
            buf.append(separator);
            buf.append("server");
            buf.append(separator);
            buf.append("domain");
            buf.append(separator);
            buf.append("mbean");
            buf.append(separator);
            buf.append("KbytesReceived");
            buf.append(separator);
            buf.append("packetsReceived");
            buf.append(separator);
            buf.append("KbytesTransmit");
            buf.append(separator);
            buf.append("packetsTransmit");
            header = buf.toString();
        }
        return header;
    }

    private void printResult(final long t, final long[] result) {
        if (header == null) {
            pout.println(getMBeanPollHeader());
        }

        Date d = new Date(t);
        pout.print(SIMPLEDATEFORMAT.format(d));
        pout.print(separator);
        pout.print(t);
        pout.print(separator);
        pout.print("na");
        pout.print(separator);
        pout.print(serverName);
        pout.print(separator);
        pout.print(domainName);
        pout.print(separator);
        pout.print("network");
        pout.print(separator);
        pout.print(result[0]);
        pout.print(separator);
        pout.print(result[1]);
        pout.print(separator);
        pout.print(result[2]);
        pout.print(separator);
        pout.print(result[3]);
        pout.println();
    }

    private String errorMessage() {
        String[] resourceNames = probe.getResourceNames();
        String mess = "";
        mess += "\n. Bad probe parameters: A correct Networkcard name must be provided in section BladeArgument [pollingPeriod pollingTime Networkcardname]!!\n.\n";
        mess += ". Networkcard Names for this host are:\n";
        int totalNbResource = osType == MAC_OS_X ? 17 : 16;

        for (int j = 0; j < resourceNames.length; j += totalNbResource) {
            String name = resourceNames[j].substring(0, resourceNames[j].length() - "bytes received".length());
            mess += ". " + name + "\n";
        }
        mess += ".\n. HINT: If a NetworkCardName is too long, provide only the last discriminating characters.\n";
        mess += ". For Instance: \n\n" +
                ".                  \"card1 3Gbits/s\"\n" +
                ".                    in place of \n" +
                ". \"My PowerFul NetworkCard FULL Duplex card1 3Gbits/s\" \n" +
                ".\n. ATTENTION: The provided NetWorkCard Name must not comport character \";\" .\n";
        return mess;
    }
}
