/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ProbeServiceImpl.java 7505 2011-01-24 16:33:31Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.osgi.internal;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Validate;

import org.osgi.framework.BundleContext;
import org.ow2.jasmine.monitoring.mbeancmd.JmxAp;
import org.ow2.jasmine.monitoring.mbeancmd.CommandDispatcher.CommandStatus;
import org.ow2.jasmine.monitoring.mbeancmd.generated.Cmdname;
import org.ow2.jasmine.monitoring.mbeancmd.generated.Media;
import org.ow2.jasmine.monitoring.mbeancmd.generated.ObjectFactory;
import org.ow2.jasmine.monitoring.mbeancmd.generated.PollType;
import org.ow2.jasmine.monitoring.mbeancmd.generated.ProbeConfig;
import org.ow2.jasmine.monitoring.mbeancmd.generated.ProbeData;
import org.ow2.jasmine.monitoring.mbeancmd.generated.Output;
import org.ow2.jasmine.monitoring.mbeancmd.generated.StatusType;
import org.ow2.jasmine.monitoring.mbeancmd.generated.Target;
import org.ow2.jasmine.probe.api.*;
import org.ow2.jonas.lib.bootstrap.JProp;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This class exposes an OSGi service bundle that implements the JasmineProbeManager
 * interface. It reads its configuration in a probe-config.xml file.
 * Called today by the jasmine-eos module ProbeManager.
 * It replaces the former part embedded in the jasmine eventswitch.
 * @author Philippe Durieux
 */
@Component(name="JasmineProbeManager")
@Provides
public class ProbeServiceImpl implements JasmineProbeManager {

    private Log logger = LogFactory.getLog(this.getClass());

    private BundleContext context;

    private ProbeConfig probeConfig = null;
    private Map<Integer, JasmineProbe> probeList = new HashMap<Integer, JasmineProbe>();
    private Map<String, JasmineOutput> outputList = new HashMap<String, JasmineOutput>();
    private Map<String, JasmineTarget> targetList = new HashMap<String, JasmineTarget>();

    private ArrayList<Integer> startlist = new ArrayList<Integer>();

    private Map<Integer, JProbeCmdThread> mbeanCmdList = new HashMap<Integer, JProbeCmdThread>();

    private int probeCount = 0;

    private List<JasmineProbeListener> probeListeners = new ArrayList<JasmineProbeListener>();


    /**
     * Inject the BundleContext in the IPojo
     * @param bc
     */
    public ProbeServiceImpl(BundleContext bc) {
        context = bc;
    }

    /**
     * Read the configuration when module is set.
     */
    @Validate
    public synchronized void start() {
        logger.debug("Start JasmineProbeManager");

        JProbeCommandDispatcher.setContext(context);

        try {
            readConfig();
        } catch (Exception e) {
            // Cannot read configuration
            logger.error("Cannot read configuration");
        }

        // Start all probes found in startlist
        for (int probeid : startlist) {
            try {
                startProbe(probeid);
            } catch (JasmineProbeException e) {
                logger.error("Cannot start probe " + probeid + ":" + e.getMessage());
            }
        }
    }

    /**
     * module is invalidated.
     */
    @Invalidate
    public synchronized void stop() {
        logger.debug("Stop JasmineProbeManager");

        // Stop all running probes
        try {
            stopAllProbes();
        } catch (Exception e) {
            // Cannot stop all probes
            logger.error("Cannot stop all probes");
        }

        // reset standard CommandDispatcher in MBeanCmd
        JProbeCommandDispatcher.removeContext();
    }

    // ------------------------------------------------------------
    // JasmineProbeManager Implementation
    // ------------------------------------------------------------

    /**
     * Create a new Output
     * @param newoutput output description
     */
    public synchronized void createOutput(final JasmineOutput newoutput) throws JasmineProbeException {
        // Id chosen by the user.
        String id = newoutput.getName();
        logger.debug("Create Output " + id);

        // Check if already known
        JasmineOutput output = outputList.get(id);
        if (output != null) {
            logger.debug("output already known: " + newoutput);
            if (output.equals(newoutput)) {
                // If same output already created, just return.
                return;
            }
            if (isOutputUsed(output)) {
                logger.warn("oldoutput:" + output);
                logger.warn("newoutput:" + newoutput);
                throw new JasmineProbeException(id + " already known with another definition");
            }
            // remove old definition and replace by the new one.
            outputList.remove(id);
        }
        outputList.put(id, newoutput);
    }

    public synchronized void changeOutput(final JasmineOutput newoutput) throws JasmineProbeException {
        // Id chosen by the user.
        String name = newoutput.getName();
        logger.debug("Change Output " + name);

        // Check if already known
        JasmineOutput output = outputList.get(name);
        if (output == null) {
            logger.warn("output unknown: " + newoutput);
            throw new JasmineProbeException(name + " unknown");
        }

        // Find probes where this output is used
        List<JasmineProbe> plist = new ArrayList<JasmineProbe>();
        for (JasmineProbe probe : probeList.values()) {
            if (probe.getOutputList().contains(output)) {
                if (probe.getState() == JasmineProbe.PROBE_RUNNING ||
                        probe.getState() == JasmineProbe.PROBE_STARTED) {
                    logger.warn("output " + name + " is used in a running probe");
                    throw new JasmineProbeException("Cannot change output currently used in a Probe");
                }
                plist.add(probe);
            }
        }

        // Replace output by new one, only if last check was OK.
        for (JasmineProbe probe : plist) {
            // replace output in probe output list
            probe.getOutputList().remove(output);
            probe.getOutputList().add(newoutput);
        }

        // replace output in the list
        outputList.put(name, newoutput);

    }


    /**
     * remove an output
     * @param name ident of the output to remove
     */
    public synchronized void removeOutput(final String name) throws JasmineProbeException {
        logger.debug("Remove output " + name);

        // Check if already known
        JasmineOutput output = outputList.get(name);
        if (output == null) {
            throw new JasmineProbeException("Unknown Output: " + name);
        }

        // Find probes where this output is used.
        List<JasmineProbe> plist = new ArrayList<JasmineProbe>();
        for (JasmineProbe probe : probeList.values()) {
            if (probe.getOutputList().contains(output)) {
                if (probe.getState() == JasmineProbe.PROBE_RUNNING ||
                        probe.getState() == JasmineProbe.PROBE_STARTED) {
                    logger.warn("output " + name + " is used in a running probe");
                    throw new JasmineProbeException("Cannot remove output currently used in a Probe");
                }
                plist.add(probe);
            }
        }

        // Remove output, only if last check was OK.
        for (JasmineProbe probe : plist) {
            // remove output in probe output list
            // TODO What if it was the last output of the probe ?
            probe.getOutputList().remove(output);
        }

        // remove output in the list
        outputList.remove(name);
    }

    /**
     * Get an OutputProbe by its Id
     * @param name ident of the output
     */
    public synchronized JasmineOutput findOutput(final String name) {
        JasmineOutput po = outputList.get(name);
        return po;
    }

    public synchronized Collection<JasmineOutput>getOutputs() throws JasmineProbeException {
        logger.debug("getOutputs");
        return outputList.values();
    }

    /**
     * Create a new Target
     * @param newtarget target description
     */
    public synchronized void createTarget(final JasmineTarget newtarget) throws JasmineProbeException {
        // Id chosen by the user.
        String id = newtarget.getName();
        logger.debug("create Target " + id);

        // Check if already known
        JasmineTarget target = targetList.get(id);
        if (target != null) {
            logger.debug("target already known: " + newtarget);
            if (target.equals(newtarget)) {
                // If same target already created, just return.
                return;
            }
            if (isTargetUsed(target)) {
                logger.warn("oldtarget:" + target);
                logger.warn("newtarget:" + newtarget);
                throw new JasmineProbeException(id + " already known with another definition");
            }
            // remove old definition and replace by the new one.
            targetList.remove(id);
        }
        targetList.put(id, newtarget);
    }

    public synchronized void changeTarget(final JasmineTarget newtarget) throws JasmineProbeException {
        String name = newtarget.getName();
        logger.debug("change Target " + name);

        // Check if already known
        JasmineTarget target = targetList.get(name);
        if (target == null) {
            logger.warn("target unknown: " + newtarget);
            throw new JasmineProbeException(name + " unknown");
        }

        // Find probes where this target is used
        List<JasmineProbe> plist = new ArrayList<JasmineProbe>();
        for (JasmineProbe probe : probeList.values()) {
            if (probe.getTargetList().contains(target)) {
                if (probe.getState() == JasmineProbe.PROBE_RUNNING ||
                        probe.getState() == JasmineProbe.PROBE_STARTED) {
                    logger.warn("target " + name + " is used in the running probe " + probe.getId());
                    throw new JasmineProbeException("Cannot change target currently used in a Probe");
                }
                plist.add(probe);
            }
        }

        // Replace target by new one, only if last check was OK.
        for (JasmineProbe probe : plist) {
            // replace target in probe target list
            probe.getTargetList().remove(target);
            probe.getTargetList().add(newtarget);
        }

        // replace target in the list
        targetList.put(name, newtarget);
    }

    /**
     * remove an target
     * @param name ident of the target to remove
     */
    public synchronized void removeTarget(final String name) throws JasmineProbeException {
        logger.debug("Remove target " + name);

        // Check if already known
        JasmineTarget target = targetList.get(name);
        if (target == null) {
            throw new JasmineProbeException("Unknown target: " + name);
        }

        // Find probes where this target is used.
        List<JasmineProbe> plist = new ArrayList<JasmineProbe>();
        for (JasmineProbe probe : probeList.values()) {
            if (probe.getTargetList().contains(target)) {
                if (probe.getState() == JasmineProbe.PROBE_RUNNING ||
                        probe.getState() == JasmineProbe.PROBE_STARTED) {
                    logger.warn("target " + name + " is used in a running probe");
                    throw new JasmineProbeException("Cannot remove target currently used in a Probe");
                }
                plist.add(probe);
            }
        }

        // Remove target, only if last check was OK.
        for (JasmineProbe probe : plist) {
            // remove target in probe target list
            // TODO What if it was the last target of the probe ?
            probe.getTargetList().remove(target);
        }

        // remove target in the list
        targetList.remove(name);
    }

    /**
     * Get a TargetProbe by its Id
     * @param targetId ident of the target
     */
    public synchronized JasmineTarget findTarget(final String targetId) {
        JasmineTarget pt = targetList.get(targetId);
        return pt;
    }

    public synchronized Collection<JasmineTarget> getTargets() {
        logger.debug("getTargets");
        return targetList.values();
    }

    /**
     * Create a Probe defined by a JasmineProbe object value
     * @param probe the object value
     * @return allocated probe id
     */
    public synchronized int createProbe(final JasmineProbe probe) throws JasmineProbeException {
        // Get the unique Id
        int probeId = getNewProbeId();
        probe.setId(probeId);
        logger.debug("creating probe #" + probeId + " : " + probe.getName());

        // Set its state to stopped
        probe.setState(JasmineProbe.PROBE_STOPPED);

        // Keep it in table
        probeList.put(probeId, probe);
        return probeId;
    }

    public synchronized void changeProbe(final JasmineProbe newprobe) throws JasmineProbeException {
        int probeId = newprobe.getId();
        JasmineProbe probe = probeList.get(probeId);
        if (probe == null) {
            throw new JasmineProbeException("Unknown Id: " + probeId);
        }
        // Don't change a probe that is running
        JProbeCmdThread cmd = mbeanCmdList.get(probeId);
        if (cmd != null) {
            if (cmd.getCmdStatus().equals(CommandStatus.RUNNING.toString())) {
                throw new JasmineProbeException("Cannot change a probe that is running");
            }
        }

        probeList.put(probeId, newprobe);
    }

    public synchronized void removeProbe(final int probeId) throws JasmineProbeException {
        JasmineProbe probe = probeList.get(probeId);
        if (probe == null) {
            throw new JasmineProbeException("Unknown Id: " + probeId);
        }
        logger.debug("removing " + probeId);
        probeList.remove(probeId);
        JProbeCmdThread cmd = mbeanCmdList.remove(probeId);
        if (cmd != null) {
            if (cmd.getCmdStatus().equals(CommandStatus.STARTED.toString())) {
                stopProbe(probeId);
            }
        }
    }

    /**
     * Remove all probes
     * @throws JasmineProbeException
     */
    public synchronized void removeAllProbes() throws JasmineProbeException {
        List<JasmineProbe> probes = new ArrayList<JasmineProbe>(probeList.values());
        for (JasmineProbe probe : probes) {
            int probeId = probe.getId();

            // Just in case it is still running: stop it.
            stopProbe(probeId);

            logger.debug("removing " + probeId);
            probeList.remove(probeId);
        }
    }


    public synchronized void startProbe(final int probeId) throws JasmineProbeException {
        JasmineProbe probe = probeList.get(probeId);
        if (probe == null) {
            throw new JasmineProbeException("Unknown Id: " + probeId);
        }
        logger.debug("starting " + probeId);
        String[] args = buildArgList(probe);
        JProbeCmdThread cmd = mbeanCmdList.get(probeId);
        if (cmd != null) {
            if (cmd.getCmdStatus().equals(CommandStatus.RUNNING.toString())) {
                throw new JasmineProbeException("Already running");
            }
            if (cmd.getCmdStatus().equals(CommandStatus.STARTED.toString())) {
                throw new JasmineProbeException("Already started");
            }
            logger.debug("Restarting probe");
            cmd.tryStop();
            mbeanCmdList.remove(probeId);
        }
        cmd = new JProbeCmdThread(args, 0, probe, this);
        mbeanCmdList.put(probeId, cmd);
        if (logger.isDebugEnabled()) {
            logger.debug("Starting Probe with args:");
            for (int i = 0; i < args.length; i++) {
                logger.debug("args[" + i + "]=" + args[i]);
            }
        }
        // start the thread running the command
        cmd.start();
    }

    public synchronized void stopProbe(final int probeId) throws JasmineProbeException {
        JasmineProbe probe = probeList.get(probeId);
        if (probe == null) {
            throw new JasmineProbeException("Unknown Id: " + probeId);
        }
        logger.debug("stopping " + probeId);
        JProbeCmdThread cmd = mbeanCmdList.remove(probeId);
        if (cmd == null) {
            logger.debug("Already stopped");
            return;
        }
        cmd.tryStop();
        logger.debug("probe has been stopped");
    }

    /**
     * Stop all the probes
     * Continues stopping probe even if one is in error
     */
    public synchronized void stopAllProbes() throws JasmineProbeException {
        for (JasmineProbe probe : probeList.values()) {
            try {
                stopProbe(probe.getId());
            } catch (Exception e) {
                logger.warn("could not stop probe " + probe.getId() + " :" + e);
            }
        }
    }

    /**
     * Start all the probes
     * Continues starting probe even if one is in error (maybe already started)
     */
    public synchronized void startAllProbes() throws JasmineProbeException {
        for (JasmineProbe probe : probeList.values()) {
            try {
                startProbe(probe.getId());
            } catch (Exception e) {
                logger.warn("could not start probe " + probe.getId() + " :" + e);
            }
        }
    }

    public synchronized Collection<JasmineProbe> getProbes() {
        logger.debug("listAllProbes");
        return probeList.values();
    }

    /**
     * Save back configuration in probe-config.xml
     * @param path LATER: path where configuration will be saved.
     */
    public synchronized void saveConfig(String path) throws JasmineProbeException {
        logger.debug("saveConfig");
        FileOutputStream outputStream = null;
        try {
            outputStream = new FileOutputStream(new File(JProp.getConfDir(), "probe-config.xml"));
        } catch (FileNotFoundException e) {
            throw new JasmineProbeException("Cannot write the config file 'probe-config.xml'" + e);
        }
        if (outputStream == null) {
            throw new JasmineProbeException("Cannot write the config file 'probe-config.xml'");
        }
        try {
            saveConfigFile(outputStream);
        } catch (Exception e) {
            logger.warn("could not save config: " + e);
            throw new JasmineProbeException("could not save config: " + e.getMessage());
        }
    }

    /**
     * Load configuration from the specified file
     * @param path File where config will be loaded
     */
    public synchronized void loadConfig(String path) throws JasmineProbeException {
        logger.debug("loadConfig: " + path);
        try {
            loadConfigFile(new FileInputStream(path));
        } catch (Exception e) {
            logger.warn("could not load config: " + e);
            throw new JasmineProbeException("could not load config: " + e.getMessage());
        }
    }

    /**
     * Execute a command.
     * @param command command definition containing the command's arguments
     * @return the command's result if any, null otherwise.
     * @exception JasmineProbeException ..
     */
    public synchronized Collection<String> execCommand(JasmineCommand command) throws JasmineProbeException {
        if (command instanceof JasmineMbeanCommand) {
            return execMbeanCommand((JasmineMbeanCommand) command);
        } else {
            logger.error("JasmineCommand type unknown");
            throw new JasmineProbeException("JasmineCommand type unknown");
        }
    }

    /**
     * Register a ProbeListener in order to be notified by probe state changes.
     * Maybe other events will be considered...
     * @param listener object that treats the probe state change
     */
    public synchronized void addProbeListener(JasmineProbeListener listener) {
        probeListeners.add(listener);
    }

    // ------------------------------------------------------------
    // Publisher on Probe Topic
    // ------------------------------------------------------------

    public void publishChange(JasmineProbe probe) {
        for (JasmineProbeListener listener : probeListeners) {
            listener.notifyEvent(probe);
        }
    }

    // ------------------------------------------------------------
    // Private methods
    // ------------------------------------------------------------

    /**
     * Execute a JasmineMbeanCommand
     * @param command command definition containing the command's arguments
     * @return the command's result if any, null otherwise.
     * @exception JasmineProbeException ..
     */
    public Collection<String> execMbeanCommand(JasmineMbeanCommand command) throws JasmineProbeException {
        if (command.isQuery()) {
            return getMbeanList(command.getTargetList().get(0), command.getFilter());
        } else {
            throw new JasmineProbeException("JasmineMbeanCommand not implemented yet");
        }
    }

    /**
     * Test if a Target is used at least in one Probe.
     * @param target the target to check
     * @return true if target used in at least 1 probe.
     */
    private boolean isTargetUsed(JasmineTarget target) {
        for (JasmineProbe probe : probeList.values()) {
            if (probe.getTargetList().contains(target)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Test if an Output is used at least in one Probe.
     * @param output the output to check
     * @return true if output used in at least 1 probe.
     */
    private boolean isOutputUsed(JasmineOutput output) {
        for (JasmineProbe probe : probeList.values()) {
            if (probe.getOutputList().contains(output)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Test a Target to see if it is running.
     * Not used today, because it may block if proxy problems
     */
    private void setTargetState(JasmineTarget target) {
        JmxAp jmxap = new JmxAp(target.getJmxUrl(), null);
        try {
            MBeanServerConnection cnx = jmxap.getMBeanServerConnection();
            cnx.getDefaultDomain();
            // target.setState(TargetDef.JasmineTarget);
        } catch (IOException e) {
            logger.error("cannot reach target: " + e);
         // target.setState(TargetDef.JasmineTarget);
        } finally {
            jmxap.releaseMBeanServerConnection();
        }
    }

    /**
     * Get the list of mbeans matching a filter
     */
    private Collection<String> getMbeanList(JasmineTarget target, String filter) throws JasmineProbeException {
        logger.debug(filter);
        JmxAp jmxap = new JmxAp(target.getJmxUrl(), null);
        Set<ObjectName> names = null;
        try {
            MBeanServerConnection cnx = jmxap.getMBeanServerConnection();
            ObjectName on = new ObjectName(filter);
            names = cnx.queryNames(on, null);
        } catch (IOException e) {
            logger.error("cannot reach target: " + e);
            throw new JasmineProbeException("cannot reach target: " + e.getMessage());
        } catch (MalformedObjectNameException e) {
            logger.error("Malformed ObjectName: " + e);
            throw new JasmineProbeException("Cannot Get MBean List: Malformed ObjectName");
        } catch (NullPointerException e) {
            logger.error("NPE: " + e);
            e.printStackTrace();
            throw new JasmineProbeException("Cannot Get MBean List");
        } finally {
            jmxap.releaseMBeanServerConnection();
        }
        Collection<String> result = new ArrayList<String>();
        for (ObjectName on : names) {
            result.add(on.toString());
        }
        return result;
    }

    /**
     * Read configuration.
     */
    private void readConfig() throws Exception {
        // Retrieve the probe config file
        String configurationFile = "probe-config.xml";
        ClassLoader loader = ClassLoader.getSystemClassLoader();
        InputStream resource = null;
        try {
            resource = loader.getResourceAsStream(configurationFile);
        } catch (Exception e) {
            e.printStackTrace();
            throw e;
        }
        if (resource == null) {
            logger.error("Cannot find probe-config.xml");
            return;
        }
        logger.debug("Parsing probe-config.xml");
        loadConfigFile(resource);
    }

    private void saveConfigFile(OutputStream resource) throws Exception {
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Marshaller marshaller = jaxbContext.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);

            // rebuild probeConfig from current configuration
            probeConfig = new ProbeConfig();

            ArrayList<Output> olist = new ArrayList<Output>();
            for (JasmineOutput jo : outputList.values()) {
                Output output = new Output();
                output.setId(jo.getName());
                if (jo instanceof ConsoleOutput) {
                    output.setDest(Media.CONSOLE);
                } else if (jo instanceof EventswitchOutput) {
                    output.setDest(Media.JASMINE);
                    output.setHost(((EventswitchOutput) jo).getHost());
                    output.setPort(((EventswitchOutput) jo).getPort());
                } else if (jo instanceof FileOutput) {
                    output.setDest(Media.LOGFILE);
                    output.setPath(((FileOutput) jo).getPath());
                }
                olist.add(output);
            }
            probeConfig.setOutput(olist);

            ArrayList<Target> tlist = new ArrayList<Target>();
            for (JasmineTarget jt : targetList.values()) {
                Target target = new Target();
                target.setId(jt.getName());
                target.setUrl(jt.getJmxUrl());
                target.setUser(jt.getUser());
                // TODO encrypt password
                target.setPassword(jt.getPassword());
                tlist.add(target);
            }
            probeConfig.setTarget(tlist);

            ArrayList<ProbeData> plist = new ArrayList<ProbeData>();
            for (JasmineProbe jp : probeList.values()) {
                ProbeData pdata = new ProbeData();
                if (jp instanceof JasmineStat) {
                    JasmineStat stat = (JasmineStat) jp;
                    pdata.setCmd(Cmdname.STAT);
                    pdata.setFilter(stat.getFilter());
                    String attlist = new String();
                    for (String attr : stat.getAttributes()) {
                        attlist += attr + " ";
                    }
                    pdata.setArgs(attlist);
                } else if (jp instanceof JasminePoll) {
                    JasminePoll poll = (JasminePoll) jp;
                    pdata.setCmd(Cmdname.POLL);
                    PollType ptype = null;
                    if (poll.getWhich().equals("jcacf")) ptype = PollType.JCACF;
                    if (poll.getWhich().equals("cpusun")) ptype = PollType.CPUSUN;
                    if (poll.getWhich().equals("tx")) ptype = PollType.TX;
                    if (poll.getWhich().equals("http")) ptype = PollType.HTTP;
                    if (poll.getWhich().equals("ds")) ptype = PollType.DS;
                    if (poll.getWhich().equals("ent")) ptype = PollType.ENT;
                    if (poll.getWhich().equals("slb")) ptype = PollType.SLB;
                    if (poll.getWhich().equals("sfb")) ptype = PollType.SFB;
                    if (poll.getWhich().equals("servlet")) ptype = PollType.SERVLET;
                    if (poll.getWhich().equals("joramq")) ptype = PollType.JORAMQ;
                    pdata.setWhich(ptype);
                    pdata.setFilter(poll.getFilter());
                }
                String period = new Long(jp.getPeriod()).toString();
                pdata.setPeriod(new BigInteger(period));
                String refreshperiod = new Long(jp.getRefreshPeriod()).toString();
                pdata.setRefreshPeriod(new BigInteger(refreshperiod));
                pdata.setSeparator(jp.getSeparator());
                ArrayList<String> tnlist = new ArrayList<String>();
                for (JasmineTarget target : jp.getTargetList()) {
                    tnlist.add(target.getName());
                }
                pdata.setTarget(tnlist);
                ArrayList<String> onlist = new ArrayList<String>();
                for (JasmineOutput output : jp.getOutputList()) {
                    onlist.add(output.getName());
                }
                pdata.setOutput(onlist);
                switch (jp.getState()) {
                case JasmineProbe.PROBE_FAILED:
                case JasmineProbe.PROBE_STOPPED:
                    pdata.setStatus(StatusType.STOPPED);
                    break;
                case JasmineProbe.PROBE_RUNNING:
                case JasmineProbe.PROBE_STARTED:
                    pdata.setStatus(StatusType.STARTED);
                    break;
                }
                plist.add(pdata);
            }
            probeConfig.setProbe(plist);

            // Build the xml file
            marshaller.marshal(probeConfig, resource);
        } catch (Exception e) {
            e.printStackTrace();
            throw e;
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }
    }

    private void loadConfigFile(InputStream resource) throws Exception {
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
            probeConfig = (ProbeConfig) unmarshaller.unmarshal(resource);

            // Create JasmineOutput objects
            for (Output output : probeConfig.getOutput()) {
                JasmineOutput jo = null;
                switch (output.getDest()) {
                case CONSOLE:
                    jo = new ConsoleOutput();
                    break;
                case JASMINE:
                    jo = new EventswitchOutput();
                    EventswitchOutput eo = (EventswitchOutput) jo;
                    eo.setHost(output.getHost());
                    eo.setPort(output.getPort());
                    break;
                case LOGFILE:
                    jo = new FileOutput();
                    FileOutput fo = (FileOutput) jo;
                    fo.setPath(output.getPath());
                    break;
                }
                jo.setName(output.getId());
                createOutput(jo);
            }

            // Create JasmineTarget objects
            for (Target target : probeConfig.getTarget()) {
                JasmineTarget jt = new JasmineTarget();
                jt.setName(target.getId());
                jt.setJmxUrl(target.getUrl());
                jt.setUser(target.getUser());
                // TODO decrypt password
                jt.setPassword(target.getPassword());
                createTarget(jt);
            }

            // Create JasmineProbe objects
            for (ProbeData pdata : probeConfig.getProbe()) {
                // Build the Probe Value Object with info found in config file
                JasmineProbe probe = null;
                switch (pdata.getCmd()) {
                case STAT:
                    JasmineStat stat = new JasmineStat();
                    probe = stat;
                    stat.setFilter(pdata.getFilter());
                    // attributes
                    ArrayList<String> sattr = new ArrayList<String>();
                    if (pdata.getArgs() != null) {
                        StringTokenizer stk = new StringTokenizer(pdata.getArgs());
                        while (stk.hasMoreTokens()) {
                            String tok = stk.nextToken();
                            sattr.add(tok);
                        }
                    }
                    stat.setAttributes(sattr.toArray(new String[sattr.size()]));
                    break;
                case POLL:
                    JasminePoll poll = new JasminePoll();
                    probe = poll;
                    if (pdata.getWhich() != null) {
                        poll.setWhich(pdata.getWhich().value());
                    } else {
                        logger.error("incorrect probe definition: missing attribute 'which'");
                        continue;
                    }
                    poll.setFilter(pdata.getFilter());
                    break;
                case SNAP:
                    logger.error("Not Supported probe type: SNAP");
                    continue;
                }

                // init fields common to all probe types
                probe.setName(pdata.getCmd().value());
                probe.setPeriod(pdata.getPeriod().intValue());
                probe.setRefreshPeriod(pdata.getRefreshPeriod().intValue());
                probe.setSeparator(pdata.getSeparator());

                // build target list
                ArrayList<JasmineTarget> tlist = new ArrayList<JasmineTarget>();
                for (String tid : pdata.getTarget()) {
                    JasmineTarget target = findTarget(tid);
                    if (target == null) {
                        logger.error("cannot find target " + tid);
                    } else {
                        tlist.add(target);
                    }
                }
                probe.setTargetList(tlist);

                // build output list
                ArrayList<JasmineOutput> olist = new ArrayList<JasmineOutput>();
                for (String oid : pdata.getOutput()) {
                    JasmineOutput output = findOutput(oid);
                    if (output == null) {
                        logger.error("cannot find output " + oid);
                    } else {
                        olist.add(output);
                    }
                }
                probe.setOutputList(olist);

                Integer id = createProbe(probe);
                if (pdata.getStatus() != null && pdata.getStatus().value().equals("started")) {
                    // autostart the Probe as set in the config file
                    startlist.add(id);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw e;
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }
    }

    private String [] buildArgList(JasmineProbe probe) throws JasmineProbeException {
        List<String> args = new LinkedList<String>();
        if (probe instanceof JasmineStat) {
            JasmineStat stat = (JasmineStat) probe;
            args.add("stat");
            args.add("-name");
            // filter (mandatory)
            if (stat.getFilter() != null && stat.getFilter().length() > 0) {
                args.add(stat.getFilter());
            }
            // attributes
            if (stat.getAttributes() != null) {
                for (int i = 0; i < stat.getAttributes().length; i++) {
                    args.add(stat.getAttributes()[i]);
                }
            }
            // refresh period (Only implemented for stat in current version)
            // LATER: do this for all probe types.
            if (probe.getRefreshPeriod() > 0) {
                args.add("-r");
                args.add("" + probe.getRefreshPeriod());
            }

        } else if (probe instanceof JasminePoll) {
            JasminePoll poll = (JasminePoll) probe;
            args.add("poll");
            args.add("-" + poll.getWhich());
            // filter (not for all poll commands)
            if (poll.getFilter() != null && poll.getFilter().length() > 0) {
                args.add(poll.getFilter());
            }
        } else {
            throw new JasmineProbeException("Probe type not supported");
        }
        // period
        if (probe.getPeriod() > 0) {
            args.add("-p");
            args.add("" + probe.getPeriod());
        }
        // separator
        if (probe.getSeparator() != null && probe.getSeparator().length() > 0) {
            args.add("-s");
            args.add(probe.getSeparator());
        }
        // cmd Ident
        args.add("-cmdid");
        args.add("" + probe.getId());

        // target
        if (probe.getTargetList().isEmpty()) {
            logger.warn("No target for probe " + probe.getId());
            throw new JasmineProbeException("Probe has no target defined");

        }
        for (JasmineTarget target : probe.getTargetList()) {
            args.add("-target");
            args.add(target.getJmxUrl());
        }

        // output
        if (probe.getOutputList().isEmpty()) {
            logger.warn("No output for probe " + probe.getId());
            throw new JasmineProbeException("Probe has no output defined");
        }
        for (JasmineOutput output : probe.getOutputList()) {
            if (output instanceof EventswitchOutput) {
                EventswitchOutput eout = (EventswitchOutput) output;
                args.add("-jasmine");
                //if (eout.getHost().equals("localhost")) {
                    // eventswitch local
                    args.add("vm://MBeanCmd_dispatcher");
                //} else {
                    // eventswitch remote
                    //args.add("tcp://" + eout.getHost() + ":" + eout.getPort() + "/MBeanCmd");
                //}
            } else if (output instanceof ConsoleOutput) {
                args.add("-console");
            } else if (output instanceof FileOutput) {
                FileOutput fout = (FileOutput) output;
                args.add("-f");
                if (fout.getPath().startsWith(File.separator)) {
                    // absolute path name
                    args.add(fout.getPath());
                } else {
                    // relative path: put the file in JONAS_BASE/monitoring directory
                    args.add(JProp.getJonasBase() + File.separator + "monitoring" +
                            File.separator + fout.getPath());
                }
            }
        }

        // Return as a String array
        logger.debug("arg list " + args);
        return args.toArray(new String[args.size()]);
    }

    /**
     * Generate a new Identifier for a Probe.
     * It will be unique for this session only.
     * @return Integer representing the probe identifier
     */
    private int getNewProbeId() {
        // for now just increment a count.
        int ret = ++probeCount;
        return ret;
    }
}
