/**
 * JASMINe
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: OuterProcessor.java 8156 2011-05-13 15:40:30Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd.event.processor.outer;

import java.io.PrintStream;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;

import org.ow2.jasmine.event.beans.JasmineEventEB;
import org.ow2.jasmine.event.processor.JasmineEventProcessor;

/**
 * Provide a processor that prints JASMINe event into a {@link PrintStream}.
 * @author Loris Bouzonnet
 */
public class OuterProcessor implements JasmineEventProcessor<OuterContext> {

    /**
     * Date format.
     */
    public static final SimpleDateFormat SIMPLEDATEFORMAT =
        new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    /**
     * A context to print.
     */
    private OuterContext context;

    /**
     * Process the given JASMINe event by printing it into the {@link PrintStream} belonging to {@link OuterContext#outer}.
     * @param event an event
     */
    public void process(final OuterContext context, final JasmineEventEB event) {
        this.context = context;
        printMBeanPoll(event.getDomain(), event.getServer(),
                event.getSource(), event.getProbe(), event.getSname(),
                event.getTimestamp().getTime(), event.getValue());
    }


    private void printMBeanPoll(final String domain, final String server,
            final String source, final String probe, final String sname, final Long timestamp, final Serializable value) {
        PrintStream pout = context.getPout();
        String separator = context.getSeparator();
        pout.println(this.getMBeanPollHeader(probe, value));
        Date d = new Date(timestamp);
        pout.print(SIMPLEDATEFORMAT.format(d));
        pout.print(separator);
        pout.print(d.getTime());
        pout.print(separator);
        pout.print(sname);
        pout.print(separator);
        pout.print(server);
        pout.print(separator);
        pout.print(domain);
        pout.print(separator);
        pout.print(source);
        pout.print(separator);
        pout.print(getStringValue(value));
        pout.println();
    }


    private String getStringValue(final Object value) {
        String separator = context.getSeparator();
        String stringValue = null;
        if(value != null) {
            if(value instanceof Map) {
                @SuppressWarnings("unchecked")
                Map<String, ?> mappedValue = (Map<String, ?>) value;
                String sep = "";
                for(String key : mappedValue.keySet()) {
                    stringValue += sep + mappedValue.get(key).toString();
                    sep = separator;
                }
            } else {
                stringValue = value.toString();
            }
        }
        return stringValue;
    }

    /**
     * Prints a given header.
     *
     * @param attl Header attributes.
     *
     * @return Printed header.
     */
    private String getMBeanPollHeader(final String probe, final Object value) {
        String separator = context.getSeparator();
        StringBuffer buf = new StringBuffer(512);
        buf.append("date");
        buf.append(separator);
        buf.append("time");
        buf.append(separator);
        buf.append("sname");
        buf.append(separator);
        buf.append("server");
        buf.append(separator);
        buf.append("domain");
        buf.append(separator);
        buf.append("mbean");
        buf.append(separator);
        buf.append(getStringKey(probe, value));
        context.setHeader(buf.toString());
        return context.getHeader();
    }

    private Object getStringKey(final String probe, final Object value) {
        String separator = context.getSeparator();
        String stringKey = null;
        if(value != null && value instanceof Map) {
            @SuppressWarnings("unchecked")
            Map<String, ?> mappedValue = (Map<String, ?>) value;
            String sep = "";
            for(String key : mappedValue.keySet()) {
                stringKey += sep + probe + "_" + key;
                sep = separator;
            }
        }
        if (stringKey == null) {
            stringKey = probe;
        }
        return stringKey;
    }

}
