/**
 * JASMINe
 * Copyright (C) 2007 SERLI Informatique
 * Contact: jasmine@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.interfaces.jmx.alarms;

import java.util.HashMap;
import java.util.List;

import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.NotificationBroadcasterSupport;
import javax.management.ObjectName;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeDataSupport;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenDataException;
import javax.management.openmbean.OpenType;
import javax.management.openmbean.SimpleType;
import javax.management.openmbean.TabularData;
import javax.management.openmbean.TabularDataSupport;
import javax.management.openmbean.TabularType;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.objectweb.jasmine.rules.logs.util.AlarmLog;
import org.objectweb.jasmine.rules.logs.beans.LogInterfaceLocal;


/**
 * Implementation of the MBean dedicated to alarms.<br><br>
 * This MBean supplies methods to retrieve alarm information by giving their id.<br>
 * It's also able to send notifications when an alarm is raised.<br>
 * <br>
 * This class must only be manipulated via its unique instance.
 * 
 * @author Awa OUEDRAOGO
 * @author Laurent RUAUD
 */
public class JMXAlarmManager extends NotificationBroadcasterSupport implements JMXAlarmManagerMBean {
	
    /** The unique instance */
    private static JMXAlarmManager instance;
    /** The object name of the unique instance */
    private static ObjectName objectName = null;
    
    /** The OpenData type for an alarm */
    private CompositeType alarmCompositeType;
    /** The OpenData type for a table of alarm */
    private TabularType alarmTabularType;
	

    
    /**
     * Gets the only instance available.
     * 
     * @return the unique instance.
     */
    @SuppressWarnings("unchecked")
    public static JMXAlarmManager getInstance() {
        if(instance == null)
            instance = new JMXAlarmManager();

        return instance;
    }

    /**
     * Gets the object name matching the unique instance.
     * 
     * @return the unique object name.
     */
    public static ObjectName getObjectName() {
        if(objectName == null)
            try {
                objectName = new ObjectName("JASMINe:name=Alarm Manager");
            } catch (MalformedObjectNameException e) {
            } catch (NullPointerException e) {
            }
        
        return objectName;
    }

    /**
     * Creates an MBean implementation.
     */
	private JMXAlarmManager() {
	    
	    //Creating JMX composite and tabular types
        try {
            alarmCompositeType = new CompositeType(
                    "JASMINe Alarm Composite", "JASMINe Alarm Informations",
                    new String[]{"ID", "Date", "Level", "Message"}, 
                    new String[]{"Alarm persistence ID", "Alarm occuration date","Alarm severity level", "Alarm brought message"}, 
                    new OpenType[] { SimpleType.INTEGER, SimpleType.DATE, SimpleType.STRING, SimpleType.STRING}
            );
            alarmTabularType = new TabularType(
                    "JASMINe Alarm Tabular", "JASMINe Alarm Table",
                    alarmCompositeType,
                    new String[]{"ID"}
            );
        } catch (OpenDataException e) {
            System.err.println("Invalid argument given during types creation: " + e.getMessage());
            e.printStackTrace();
        }
	}
    
	

	/**
     * Sends a notification to the MBean server.
     * 
     * @param id id of the alarm to be notified.
     */
	public void notifyAlarm(int id) {
	    Notification notif =  new Notification("JASMINe Alarm", objectName, id, "An alarm has been raised");
	    notif.setUserData(id);
	    sendNotification(notif);
	}
	
    /**
     * Retrieves information on the specified alarm from the log interface.
     *  
     * @param id id of the alarm.
     * @return a composite object containing a copy of alarm's information.
     */
	@SuppressWarnings("unchecked")
    public CompositeData getAlarm(int id) {
	    CompositeData log = null;
        try {
            Context ctx = new InitialContext();
            LogInterfaceLocal logInterface = (LogInterfaceLocal) ctx.lookup(
                    "org.objectweb.jasmine.rules.logs.LogInterface"
                    + "_" + LogInterfaceLocal.class.getName()
                    + "@Local"
            );
            log = compositeFromLog(logInterface.getLog(id));
        } catch (NamingException e) {
            e.printStackTrace();
        }
        return log;
    }
    
    /**
     * Retrieves information on the last raised alarms from the log interface.
     * The maximum number of alarm to retrieve must be specified.
     *  
     * @param nb number of alarms to retrieve.
     * @return a tabular object containing copies of alarm's information.
     */
    @SuppressWarnings("unchecked")
    public TabularData getLastAlarms(int nb) {
        TabularData logs = null;
        try {
            Context ctx = new InitialContext();
            LogInterfaceLocal logInterface = (LogInterfaceLocal) ctx.lookup(
                    "org.objectweb.jasmine.rules.logs.LogInterface"
                    + "_" + LogInterfaceLocal.class.getName()
                    + "@Local"
            );
            logs = tabularFromLogList(logInterface.getLastLogs(nb));;
        } catch (NamingException e) {
            e.printStackTrace();
        }
        return logs;
    }
    
    /**
     * Retrieves the alarms with an id contained in the specified range of id inclusive.
     * 
     * @param lowid the low threshold.
     * @param highid the high threshold.
     * @return a tabular object containing copies of alarm's information.
     */
    @SuppressWarnings("unchecked")
    public TabularData getAlarmRange(int lowid, int highid) {
        TabularData logs = null;
        try {
            Context ctx = new InitialContext();
            LogInterfaceLocal logInterface = (LogInterfaceLocal) ctx.lookup(
                    "org.objectweb.jasmine.rules.logs.LogInterface"
                    + "_" + LogInterfaceLocal.class.getName()
                    + "@Local"
            );
            logs = tabularFromLogList(logInterface.getLogRange(lowid, highid));;
        } catch (NamingException e) {
            e.printStackTrace();
        }
        return logs;
    }
    
    

    /**
     * Utility method that convert an alarm to a composite object.<br>
     * The alarm has been retrieved from the log interface and the composite object will be send through the MBean server.
     * 
     * @param log the alarm to convert.
     * @return the resulted composite object.
     */
    @SuppressWarnings("unchecked")
    private CompositeData compositeFromLog(AlarmLog log) {
        if(log == null) return null;
        try {
            HashMap alarmMapData = new HashMap();
            alarmMapData.put("ID", log.getId());     
            alarmMapData.put("Date", log.getDate());
            alarmMapData.put("Level", log.getLevel());
            alarmMapData.put("Message", log.getMessage());

            return new CompositeDataSupport(alarmCompositeType, alarmMapData);
        } catch (OpenDataException e) {
            System.err.println("Invalid argument given during composite creation: " + e.getMessage());
            e.printStackTrace();
        }
        return null;
    }
    
    /**
     * Utility method that convert a list of alarms to a tabular object.<br>
     * This uses the compositeFromLog method to convert each alarm of the list to a composite object.
     * 
     * @param logs the list of alarms to convert.
     * @return the resulted tabular object.
     */
    private TabularData tabularFromLogList(List<AlarmLog> logs) {
        TabularData alarmTabularData = new TabularDataSupport(alarmTabularType);
        for (AlarmLog log:logs) alarmTabularData.put(compositeFromLog(log));
        
        return alarmTabularData;
    }
    
}
