/**
 * JASMINe
 * Copyright (C) 2007 SERLI Informatique
 * Contact: jasmine@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.interfaces.jmx;

import java.util.ArrayList;
import java.util.List;

import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import javax.resource.ResourceException;
import javax.resource.spi.ActivationSpec;
import javax.resource.spi.BootstrapContext;
import javax.resource.spi.ResourceAdapter;
import javax.resource.spi.ResourceAdapterInternalException;
import javax.resource.spi.endpoint.MessageEndpointFactory;
import javax.transaction.xa.XAResource;

import org.ow2.jasmine.interfaces.jmx.alarms.JMXAlarmManager;

/**
 * This class implements a connector to provide support of MBean registration.
 *
 * @author Laurent RUAUD
 */
public class JMXResourceAdapter implements ResourceAdapter {

    /** The MBean server */
    private MBeanServer server;
    /** The ObjectNames that have been registered */
    private List<ObjectName> names;
    
    

    /**
     * Creates a connector for JMX external interfaces.
     */
    public JMXResourceAdapter() {
        names = new ArrayList<ObjectName>();
    }

    
    
    /**
     * This is called when the resource adapter instance is bootstrapped.
     * It finds the MBean server and registers MBeans of external interfaces.
     */
    @SuppressWarnings("unchecked")
    public void start(BootstrapContext arg0) throws ResourceAdapterInternalException {
        // Finding MBean server
        List servers = MBeanServerFactory.findMBeanServer(null);
        if (servers.size() > 0) server = (MBeanServer) servers.get(0);
        else server = MBeanServerFactory.createMBeanServer(null);
        
        // Registering Alarm Manager
        register(JMXAlarmManager.getInstance(), JMXAlarmManager.getObjectName());
        names.add(JMXAlarmManager.getObjectName());
    }

    /**
     * This is called when the resource adapter instance is undeployed or during application server shutdown.
     * It makes the MBean server unregister all registered MBeans.  
     */
    public void stop() {
        for(ObjectName name:names)
            try {
                server.unregisterMBean(name);
            } catch (InstanceNotFoundException e) {
                System.err.println("Unable to unregister a name of a JMX external interface: " + name);
                e.printStackTrace();
            } catch (MBeanRegistrationException e) {
                System.err.println("Unable to unregister a name of a JMX external interface: " + name);
                e.printStackTrace();
            }
    }

    
    
    /**
     * Registers an object with its name to the MBean server.
     * 
     * @param object the object to register.
     * @param name the object name of the object to register.
     */
    private void register(Object object, ObjectName name) {
        try {
            if(server.isRegistered(name))
                server.unregisterMBean(name);
            server.registerMBean(object, name);
        } catch (InstanceNotFoundException e) {
            System.err.println("Unable to register a name of a JMX external interface: " + name);
            e.printStackTrace();
        } catch (InstanceAlreadyExistsException e) {
            System.err.println("Unable to register a name of a JMX external interface: " + name);
            e.printStackTrace();
        } catch (MBeanRegistrationException e) {
            System.err.println("Unable to register a name of a JMX external interface: " + name);
            e.printStackTrace();
        } catch (NotCompliantMBeanException e) {
            System.err.println("Unable to register a name of a JMX external interface: " + name);
            e.printStackTrace();
        }
    }
    
    

    /**
     * Not used
     * 
     * @param arg0 Not used
     * @param arg1 Not used
     */
    public void endpointActivation(MessageEndpointFactory arg0, ActivationSpec arg1) throws ResourceException {}

    /**
     * Not used
     * 
     * @param arg0 Not used
     * @param arg1 Not used
     */
    public void endpointDeactivation(MessageEndpointFactory arg0, ActivationSpec arg1) {}
    
    /**
     * Not used
     * 
     * @param arg0 Not used
     */
    public XAResource[] getXAResources(ActivationSpec[] arg0) throws ResourceException {
        return null;
    }
}
