/**
 * JASMINe
 * Copyright (C) 2007 SERLI Informatique
 * Contact: jasmine@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.interfaces.snmp.alarms;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.objectweb.jasmine.rules.logs.AlarmLog;
import org.objectweb.jasmine.rules.logs.LogInterfaceLocal;
import org.ow2.jasmine.interfaces.snmp.SNMPAgent;
import org.snmp4j.agent.DuplicateRegistrationException;
import org.snmp4j.agent.MOGroup;
import org.snmp4j.agent.MOServer;
import org.snmp4j.agent.mo.DefaultMOFactory;
import org.snmp4j.agent.mo.MOAccessImpl;
import org.snmp4j.agent.mo.MOColumn;
import org.snmp4j.agent.mo.MOFactory;
import org.snmp4j.agent.mo.MOTable;
import org.snmp4j.agent.mo.MOTableIndex;
import org.snmp4j.agent.mo.MOTableSubIndex;
import org.snmp4j.smi.Gauge32;
import org.snmp4j.smi.OID;
import org.snmp4j.smi.OctetString;
import org.snmp4j.smi.SMIConstants;
import org.snmp4j.smi.VariableBinding;

/**
 * Implementation of the alarm part of the JASMINe MIB.<br><br>
 * This MIB only contains a tabular object: the alarms' data.<br>
 * <br>
 * This class must only be manipulated via its unique instance.
 * 
 * @author Laurent RUAUD
 */
public class SNMPAlarmMIB implements MOGroup {
    /** The identifier of the JASMINe branch */
	public static final OID oidJASMINe = new OID(new int[] { 1,3,6,1,2,1,133 });
    /** The identifier of the Alarms branch */
    public static final OID oidJASMINeAlarms = new OID(oidJASMINe.getValue(), new int[] { 1 });

    /** The identifier of the Alarm objects branch */
    public static final OID oidJASMINeAlarmObjects = new OID(oidJASMINeAlarms.getValue(), new int[] { 1 });
    /** The identifier of the branch containing the alarm data table */
    public static final OID oidAlarmDataTable = new OID(oidJASMINeAlarmObjects.getValue(), new int[] { 1 });
    /** The identifier of the branch containing the alarm data entries */
    public static final OID oidAlarmDataEntry = new OID(oidAlarmDataTable.getValue(), new int[] { 1 });
    /** The identifier of the branch containing the alarm data entries */
    public static final OID oidAlarmDataEntryIndex = new OID(oidAlarmDataEntry.getValue(), new int[] { 1 });

    /** The identifier of the Alarm notifications branch */
    public static final OID oidJASMINeAlarmNotifications = new OID(oidJASMINeAlarms.getValue(), new int[] { 2 });
    /** The identifier of the leaf representing the <i>alarm raised</i> notification*/
	public static final OID oidAlarmRaised = new OID(oidJASMINeAlarmNotifications.getValue(), new int[] { 0,1 });

    
    
    /** The singleton instance */
    private static SNMPAlarmMIB instance;

    
    
    /** The alarm data table */
	private MOTable  alarmDataTable;

    
    
    /**
     * Gets the only instance available.
     * 
     * @return the unique instance.
     */
    public static SNMPAlarmMIB getInstance() {
        if(instance == null)
            instance = new SNMPAlarmMIB();
        
        return instance;
    }
    
    /**
     * Creates and initializes a MIB for JASMINe Alarms.
     */
	private SNMPAlarmMIB() {
        createAlarmDataEntry();
        
	}

	
	
    /**
     * Sends a notification and adds a reference to the specified alarm in the MIB.
     * This creates a new row referencing the specified alarm.
     * 
     * @param id id of the alarm to be referenced.
     */
    public void notifyAlarm(int id) {
        SNMPAlarm alarm = new SNMPAlarm(id);
        alarmDataTable.addRow(alarm);

        SNMPAgent.getInstance().notify(
                new OctetString(), 
                SNMPAlarmMIB.oidAlarmRaised, 
                new VariableBinding[]{ new VariableBinding(
                        new OID(oidAlarmDataEntryIndex.getValue(), new int[] { id }),new Gauge32(id)
                ) }
        );
    }

    /**
     * Retrieves information on the specified alarm from the log interface.
     *  
     * @param id id of the alarm.
     * @return an AlarmLog object containing a copy of alarm's information
     */
    public AlarmLog getAlarm(int id) {
        AlarmLog log = null;
        try {
            Context ctx = new InitialContext();
            LogInterfaceLocal logInterface = (LogInterfaceLocal) ctx.lookup(
                    "org.objectweb.jasmine.rules.logs.LogInterface"
                    + "_" + LogInterfaceLocal.class.getName()
                    + "@Local"
            );
            log = logInterface.getLog(id);
        } catch (NamingException e) {
            e.printStackTrace();
        }
        return log;
    }
	
	
	
    /**
     * Registers the managed objects of this MIB to the specified server with the supplied context.
     * 
     * @param server the server where to register the managed objects.
     * @param context the context to use.
     */
	public void registerMOs(MOServer server, OctetString context)
			throws DuplicateRegistrationException {
		server.register(this.alarmDataTable, context);
	}

    /**
     * Unregisters the managed objects of this group from the supplied server and from the supplied context.
     * 
     * @param server the server where to unregister the managed objects.
     * @param context the context to use.
     */
	public void unregisterMOs(MOServer server, OctetString context) {
		server.unregister(this.alarmDataTable, context);
	}

    /**
     * Creates the alarm data table structure.
     */
	private void createAlarmDataEntry() {
	    MOFactory moFactory = DefaultMOFactory.getInstance();
	    
		MOColumn[] columns = new MOColumn[4];
		columns[0] = moFactory.createColumn(1, SMIConstants.SYNTAX_GAUGE32, MOAccessImpl.ACCESS_READ_ONLY);
		columns[1] = moFactory.createColumn(2, SMIConstants.SYNTAX_OCTET_STRING, MOAccessImpl.ACCESS_READ_ONLY);
		columns[2] = moFactory.createColumn(3, SMIConstants.SYNTAX_OCTET_STRING, MOAccessImpl.ACCESS_READ_ONLY);
		columns[3] = moFactory.createColumn(4, SMIConstants.SYNTAX_OCTET_STRING, MOAccessImpl.ACCESS_READ_ONLY);

	    MOTableIndex alarmDataTableIndex = moFactory.createIndex(
	        new MOTableSubIndex[] { moFactory.createSubIndex(oidAlarmDataEntryIndex, SMIConstants.SYNTAX_GAUGE32)  },
	        false
	    );
		
		
		alarmDataTable = moFactory.createTable(oidAlarmDataEntry, alarmDataTableIndex, columns);
	}
}
