/**
 * JASMINe
 * Copyright (C) 2007 SERLI Informatique
 * Contact: jasmine@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.interfaces.snmp;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import org.snmp4j.TransportMapping;
import org.snmp4j.agent.BaseAgent;
import org.snmp4j.agent.CommandProcessor;
import org.snmp4j.agent.DuplicateRegistrationException;
import org.snmp4j.agent.MOGroup;
import org.snmp4j.agent.mo.MOTableRow;
import org.snmp4j.agent.mo.snmp.RowStatus;
import org.snmp4j.agent.mo.snmp.SnmpCommunityMIB;
import org.snmp4j.agent.mo.snmp.SnmpNotificationMIB;
import org.snmp4j.agent.mo.snmp.SnmpTargetMIB;
import org.snmp4j.agent.mo.snmp.StorageType;
import org.snmp4j.agent.mo.snmp.TransportDomains;
import org.snmp4j.agent.mo.snmp.VacmMIB;
import org.snmp4j.mp.MPv3;
import org.snmp4j.mp.MessageProcessingModel;
import org.snmp4j.security.SecurityLevel;
import org.snmp4j.security.SecurityModel;
import org.snmp4j.security.USM;
import org.snmp4j.smi.Address;
import org.snmp4j.smi.GenericAddress;
import org.snmp4j.smi.Integer32;
import org.snmp4j.smi.OID;
import org.snmp4j.smi.OctetString;
import org.snmp4j.smi.UdpAddress;
import org.snmp4j.smi.Variable;
import org.snmp4j.smi.VariableBinding;
import org.snmp4j.transport.TransportMappings;

/**
 * JASMINe SNMP agent class.<br><br> This class provides basic SNMP features and allows to register MIB objects in order to supply information through SNMP protocol.<br> All external interfaces
 * dedicated to SNMP must use this class as a base for all its communications.<br> <br> This class must only be manipulated via its unique instance.
 * 
 * @author Laurent RUAUD
 */
public class SNMPAgent extends BaseAgent {

    /** The default request values */
    private static String DEFAULT_REQUEST_SOURCE = "127.0.0.1/1161";

    /** All MIBs that have been registered */
    private ArrayList<MOGroup> registeredMibs;
    /** All destinations for traps that have been registered */
    private ArrayList<String> registeredDestinations;
    /** All destinations for traps that have been registered */
    private String requestSource;

    /** The singleton instance */
    private static SNMPAgent instance;


    
    /**
     * Gets the only instance available.
     * 
     * @return the unique instance.
     */
    public static SNMPAgent getInstance() {
        if (instance == null)
            instance = new SNMPAgent();

        return instance;
    }

    /**
     * Creates and initializes an agent.
     */
    private SNMPAgent() {
        super(new File("AgentConfig.cfg"), new File("AgentBootCounter.cfg"), new CommandProcessor(new OctetString(MPv3.createLocalEngineID())));
        
        requestSource = DEFAULT_REQUEST_SOURCE;
        registeredMibs = new ArrayList<MOGroup>();
        registeredDestinations = new ArrayList<String>();
    }

    
    
    /**
     * Starts the agent.
     * This registers trap destinations and starts listening on request source.
     */
    public void start() {
        try {
            init();
        } catch (IOException e) {
            System.err.println("Unable to start the SNMP agent: " + e.getMessage());
            e.printStackTrace();
        }
        getServer().addContext(new OctetString("public"));
        finishInit();
        run();
    }
    
    /**
     * Stops the agent.
     * This unregisters trap destinations and stops listening on request source.
     * This also unregisters all managed objects and MIBs.
     */
    public void stop() {
        unregisterManagedObjects();
    }
    
    

    /**
     * Defines the source to listen to GET and SET request.<br>
     * Source must be formatted as followed: ipaddress/port (127.0.0.1/161).
     * 
     * @param requestSource source to listen to GET and SET request
     */
    public void setRequestSource(String source) {
        requestSource = source;
    }

    /**
     * Adds a new destination for SNMP traps.<br>
     * Destinations must be formatted as followed: ipaddress/port (127.0.0.1/162).
     * 
     * @param requestSource the destination to add.
     */
    public void addTrapDestination(String trapDestination) {
        registeredDestinations.add(trapDestination);
    }
    

    
    /**
     * Sends a notification through SNMP.
     * 
     * @param context the context name of the context on whose behalf this notification has been generated.
     * @param notificationID the object ID that uniquely identifies this notification.
     * @param vbs an array of <code>VariableBinding</code> instances representing the payload of the notification.
     */
    public void notify(OctetString context, OID notificationID, VariableBinding[] vbs) {
        notificationOriginator.notify(context, notificationID, vbs);
    }

    /**
     * Registers an MIB based on the specified manager.
     * 
     * @param manager the manager the MIB will be based on.
     * @return an MIB based on the specified manager.
     */
    public void registerMIB(MOGroup mib) {
        try {
            mib.registerMOs(server, null);
        } catch (DuplicateRegistrationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Registers an MIB based on the specified manager.
     * 
     * @param manager the manager the MIB will be based on.
     * @return an MIB based on the specified manager.
     */
    public void unregisterMIB(MOGroup mib) {
        mib.unregisterMOs(server, null);
    }

    
    
    /**
     * Register additional managed objects at the agent's server.
     */
    @Override
    protected void registerManagedObjects() {
    }

    /**
     * Unregister additional managed objects from the agent's server.
     */
    @Override
    protected void unregisterManagedObjects() {
        for (MOGroup mib : registeredMibs)
            mib.unregisterMOs(server, null);
    }

    
    
    /**
     * Adds community to security name mappings needed for SNMPv1 and SNMPv2c.
     * 
     * @param communityMIB the SnmpCommunityMIB holding coexistence configuration for community based security models.
     */
    @Override
    protected void addCommunities(SnmpCommunityMIB communityMIB) {
        Variable[] values = new Variable[] {
            new OctetString("public"),               // community name
            new OctetString("public"),               // security name
            getAgent().getContextEngineID(),         // local engine ID
            new OctetString(),                       // default context name
            new OctetString(),                       // transport tag
            new Integer32(StorageType.nonVolatile),  // storage type
            new Integer32(RowStatus.active)          // row status
        };
        MOTableRow row = communityMIB.getSnmpCommunityEntry().createRow(new OctetString("public2public").toSubIndex(true), values);
        communityMIB.getSnmpCommunityEntry().addRow(row);
    }

    /**
     * Adds initial notification targets and filters.
     * 
     * @param targetMIB the SnmpTargetMIB holding the target configuration.
     * @param notificationMIB the SnmpNotificationMIB holding the notification (filter) configuration.
     */
    @Override
    protected void addNotificationTargets(SnmpTargetMIB targetMIB, SnmpNotificationMIB notificationMIB) {
        targetMIB.addDefaultTDomains();

        for(String destination:registeredDestinations)
            targetMIB.addTargetAddress(
                    new OctetString("notif" + destination), 
                    TransportDomains.transportDomainUdpIpv4, 
                    new OctetString(new UdpAddress(destination).getValue()), 
                    1500, 
                    3, 
                    new OctetString("notify"), 
                    new OctetString("v2c"), 
                    StorageType.permanent
            );
        targetMIB.addTargetParams(
                new OctetString("v2c"), 
                MessageProcessingModel.MPv2c, 
                SecurityModel.SECURITY_MODEL_SNMPv2c, 
                new OctetString("public"), 
                SecurityLevel.NOAUTH_NOPRIV, 
                StorageType.permanent
        );
        notificationMIB.addNotifyEntry(
                new OctetString("default"), 
                new OctetString("notify"), 
                SnmpNotificationMIB.SnmpNotifyTypeEnum.trap, 
                StorageType.permanent
        );
    }

    /**
     * Adds all the necessary initial users to the USM.
     * 
     * @param usm the USM instance used by this agent.
     */
    @Override
    protected void addUsmUser(USM usm) {
    }

    /**
     * Adds initial VACM configuration.
     * 
     * @param vacmMIB the VacmMIB holding the agent's view configuration.
     */
    @Override
    protected void addViews(VacmMIB vacm) {
        vacm.addGroup(
                SecurityModel.SECURITY_MODEL_SNMPv1, 
                new OctetString("public"), 
                new OctetString("v1v2group"), 
                StorageType.nonVolatile
        );
        vacm.addGroup(
                SecurityModel.SECURITY_MODEL_SNMPv2c, 
                new OctetString("public"),
                new OctetString("v1v2group"), 
                StorageType.nonVolatile
        );

        vacm.addAccess(
                new OctetString("v1v2group"), 
                new OctetString(), 
                SecurityModel.SECURITY_MODEL_ANY, 
                SecurityLevel.NOAUTH_NOPRIV, 
                VacmMIB.vacmExactMatch, 
                new OctetString("fullReadView"), 
                new OctetString("fullWriteView"), 
                new OctetString("fullNotifyView"), 
                StorageType.nonVolatile
        );

        vacm.addViewTreeFamily(
                new OctetString("fullReadView"), 
                new OID("1.3"), 
                new OctetString(), 
                VacmMIB.vacmViewIncluded, 
                StorageType.nonVolatile
        );
        vacm.addViewTreeFamily(
                new OctetString("fullWriteView"), 
                new OID("1.3"), 
                new OctetString(), 
                VacmMIB.vacmViewIncluded, 
                StorageType.nonVolatile
        );
        vacm.addViewTreeFamily(
                new OctetString("fullNotifyView"), 
                new OID("1.3"), 
                new OctetString(), 
                VacmMIB.vacmViewIncluded, 
                StorageType.nonVolatile
        );
    }

    /**
     * Initializes the transport mappings (ports) to be used by the agent.
     * 
     * @throws IOException
     */
    @Override
    protected void initTransportMappings() throws IOException {
        Address addr = GenericAddress.parse(requestSource);
        TransportMapping tm = TransportMappings.getInstance().createTransportMapping(addr);

        if(tm == null) {
            System.err.println("Unable to establish a connection using address: " + requestSource);
            
            if(!requestSource.equals(DEFAULT_REQUEST_SOURCE)) {
                System.err.println("Will try with default values: " + DEFAULT_REQUEST_SOURCE);
                
                addr = GenericAddress.parse(DEFAULT_REQUEST_SOURCE);
                tm = TransportMappings.getInstance().createTransportMapping(addr);
                
                if(tm == null) System.err.println("Unable to establish a connection using default values.");
            }
        }
        
        transportMappings = new TransportMapping[1];
        transportMappings[0] = tm;
    }

}
