/**
 * JASMINe
 * Copyright (C) 2007 SERLI Informatique
 * Contact: jasmine@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.interfaces.snmp;

import java.util.ArrayList;

import javax.resource.ResourceException;
import javax.resource.spi.ActivationSpec;
import javax.resource.spi.BootstrapContext;
import javax.resource.spi.ResourceAdapter;
import javax.resource.spi.ResourceAdapterInternalException;
import javax.resource.spi.endpoint.MessageEndpointFactory;
import javax.transaction.xa.XAResource;

import org.ow2.jasmine.interfaces.snmp.alarms.SNMPAlarmMIB;

/**
 * This class implements a connector to provide SNMP communications.
 *
 * @see SNMPAgent
 * @author Laurent RUAUD
 */
public class SNMPResourceAdapter implements ResourceAdapter {

    /** The ip address of incoming requests */
    private String incomingRequestAddress;
    /** The port of incoming requests */
    private String incomingRequestPort;
    /** The ip address of traps destination */
    private ArrayList<String> trapDestinationAddresses = new ArrayList<String>();
    /** The port of traps destination */
    private String trapDestinationPort;

       
    
    /**
     * This is called when the resource adapter instance is bootstrapped.
     * It initiates the singleton instance of SNMP agent and registers MIB of external interfaces.
     */
    public void start(BootstrapContext arg0) throws ResourceAdapterInternalException {
        SNMPAgent.getInstance().setRequestSource(incomingRequestAddress + "/" + incomingRequestPort);
        for(String address:trapDestinationAddresses) {
            System.out.println("Agent.add : " + address + "/" + trapDestinationPort);
            SNMPAgent.getInstance().addTrapDestination(address + "/" + trapDestinationPort);
        }
        SNMPAgent.getInstance().start();
        SNMPAgent.getInstance().registerMIB(SNMPAlarmMIB.getInstance());
    }

    /**
     * This is called when the resource adapter instance is undeployed or during application server shutdown.
     * It makes the SNMP agent unregister all registered MIBs.  
     */
    public void stop() {
        SNMPAgent.getInstance().stop();
    }

    

    /**
     * Defines the ip address of incoming requests.
     * 
     * @param incomingRequestAddress the ip address of incoming requests.
     */
    public void setIncomingRequestAddress(String incomingRequestAddress) {
        this.incomingRequestAddress = incomingRequestAddress;
    }

    /**
     * Defines the port of incoming requests.
     * 
     * @param incomingRequestPort the port of incoming requests.
     */
    public void setIncomingRequestPort(String incomingRequestPort) {
        this.incomingRequestPort = incomingRequestPort;
    }

    /**
     * Defines ip addresses for traps destination.
     * Several addresses can be defined as a comma-separated list.
     * 
     * @param trapDestinationAddress an ip address for traps destination.
     */
    public void setTrapDestinationAddress(String trapDestinationAddress) {
        trapDestinationAddresses.clear();
        for(String address:trapDestinationAddress.split(","))
            this.trapDestinationAddresses.add(address.trim());
    }

    /**
     * Defines the port of traps destination.
     * 
     * @param trapDestinationPort the port of traps destination.
     */
    public void setTrapDestinationPort(String trapDestinationPort) {
        this.trapDestinationPort = trapDestinationPort;
    }

    
    
    /**
     * Not used
     * 
     * @param arg0 Not used
     * @param arg1 Not used
     */
    public void endpointActivation(MessageEndpointFactory arg0, ActivationSpec arg1) throws ResourceException {}

    /**
     * Not used
     * 
     * @param arg0 Not used
     * @param arg1 Not used
     */
    public void endpointDeactivation(MessageEndpointFactory arg0, ActivationSpec arg1) {}
    
    /**
     * Not used
     * 
     * @param arg0 Not used
     */
    public XAResource[] getXAResources(ActivationSpec[] arg0) throws ResourceException {
        return null;
    }
}
