/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * $Id: JasmineEventNotificationSLBRemote.java 6164 2010-03-20 17:03:30Z alitokmen $
 */

package org.ow2.jasmine.monitoring.eventswitch.beans;

import java.util.Date;
import java.util.List;

import org.ow2.jasmine.event.beans.JasmineEventNotification;
import org.ow2.jasmine.event.beans.JasmineEventNotificationExecutedAction;

/**
 * Interface for the JasmineEventNotification stateless bean.
 *
 * @author HAN Tianyi
 */
public interface JasmineEventNotificationSLBRemote {

    /**
     * Saves the action executed by a user in the DataBase.
     *
     * @param id the Id of the notification to update.
     * @param action the action to save.
     */
    void saveExecutedAction(final String id, final JasmineEventNotificationExecutedAction action);

    /**
     * Updates the status of the notification : acknowledge or not.
     *
     * @param id the id of the notification to update.
     * @param status the new status to save.
     */
    void updateNotificationStatus(final String id, final boolean status);

    /**
     * creates the JasmineEventNotification entity bean corresponding to given
     * datum and saves an event on the database.
     *
     * @see JasmineEventNotificationSLBRemote#saveEvent(JasmineEventNotificationEB)
     * @param n Notification to save.
     */
    void saveEvent(final JasmineEventNotification n);

    /**
     * Gets all the notification events from the DataBase.
     *
     * @return A list that contains all the {@link JasmineEventNotification}
     *         stored in the DataBase.
     */
    List<JasmineEventNotification> getAllNotifications();

    /**
     * looks for the beans searched for, converts the search into an array of
     * JasmineEventNotificationEB objects and returns.
     *
     * @param id Notification id, ignored if null.
     * @param date Notification date, ignored if null.
     * @param level Notification level, ignored if null.
     * @param message Associated message, ignored if null.
     * @param status Notification status, ignored if null.
     * @param rule The rule id that caused the notification, ignored if null.
     * @param orderBy Order using column, ignored if null..
     * @param limit Maximal number of results to get, ignored if null.
     * @return EventNotifications corresponding to the given search, null if
     *         none found.
     */
    JasmineEventNotification[] getEventNotifications(final String id, final Date date, final String level,
        final String message, final Boolean status, final String rule, final String orderBy, final Integer limit);

    /**
     * Gets the event notification between two dates.
     *
     * @param dateFrom the date from
     * @param dateTo the date to
     * @return a list of {@link JasmineEventNotification}
     */
    List<JasmineEventNotification> getEventNotifications(final Date dateFrom, final Date dateTo);

    /**
     * Gets the event notifications for a given rule, a "synchronized" flag and a
     * keepDisplayed value.
     *
     * @param keepDisplayed the keepDisplayed option, expressed in hours:
     *        <ul>
     *          <li>
     *            If "synchronized" is false, only notifications with a date
     *            inferior to now minus keepDisplayed hours will be returned.
     *          </li>
     *          <li>
     *            If "synchronized" is true, a merge of two types of
     *            notifications is returned:
     *            <ol>
     *              <li>
     *                Notifications with status false (i.e., not acknowledged
     *                yet)
     *              </li>
     *              <li>
     *                Notifications with status true AND at least one action of
     *                type ACKNOWLEDGE with a date to now minus keepDisplayed
     *                hours
     *              </li>
     *            </ol>
     *          </li>
     *        </ul>
     * @param acknowledgeButtonName the name of the ACKNOWLEDGE button. if
     *        null, "synchronized" is considered to be false; otherwise it is
     *        considered to be true.
     * @param rules rule names. might contain * characters at the end or at the
     *        beginning, meaning respectedly "rule starts with" or "rule ends
     *        "with".
     * @return a list of {@link JasmineEventNotification}
     */
    List<JasmineEventNotification> getEventNotifications(final int keepDisplayed,
        final String acknowledgeButtonName, final String... rules);

    /**
     * Looks for the given bean in the data base. It returns the associated
     * {@link JasmineEventNotification}.
     *
     * @param id Id of the notification to retrieve.
     * @return the retrieved notification.
     */
    JasmineEventNotification getEventNotification(final String id);
}
