/**
 * JASMINe
 * Copyright (C) 2007-2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: EJB3Connector.java 9538 2012-01-05 09:49:15Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eventswitch.connectors;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.rmi.RMISecurityManager;
import java.util.Hashtable;
import java.util.Map;
import java.util.TreeMap;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

/**
 * Defines an abstract EJB3 connector.
 *
 * @param <BeanType>  Type of bean accessed using this wrapper. This must be
 *                    set to the remote interface of the bean, bean can be
 *                    stateful or stateless.
 */
public class EJB3Connector<BeanType> {

    /**
     * Reference to the bean.
     */
    private BeanType remote = null;

    /**
     * Configuration.
     *
     */
    private Map<String, String> configuration = null;

    /**
     * Map containing Context configuration matches.
     */
    private static Map<String, String> keys = null;
    static {
        keys = new TreeMap<String, String>();
        for(Field f : Context.class.getFields()) {
            try {
                if(f.getType().equals(java.lang.String.class)) {
                    keys.put(f.getName(), (String) f.get(null));
                }
            } catch(Exception e) {
                // Nothing
            }
        }
    }

    /**
     * Sets the configuration to use for the EJB connection.
     *
     * @param conf  Configuration: configures all Context environment
     *              variables as well as the object name to look for.
     *              It contains mappings to:
     *                  - Any number of string constants defined in Context.
     *                    Example: to set the Context.INITIAL_CONTEXT_FACTORY
     *                    on the Map conf to factory, use:
     *                          conf.put("Context.INITIAL_CONTEXT_FACTORY", factory);
     *                  - The "Wrapper.LOOKUP_BEAN" key, the value of which is
     *                    looked up on the server and set as remote bean. That
     *                    bean is at @link MuleEJB3Wrapper#getBean()}.
     */
    public final void setConfiguration(final Map<String, String> conf) {
        configuration = conf;
    }

    /**
     * Returns a reference to the stateful or stateless bean. It will call
     * {@link EJB3Connector#connect()} if the reference hadn't been set yet.
     *
     * @return  Reference to the stateful or stateless bean.
     *
     * @throws NamingException  On naming failure.
     * @throws IOException  On configuration file failure.
     */
    protected final BeanType getBean() throws NamingException, IOException {
        if (remote == null) {
            connect();
        }
        return remote;
    }

    /**
     * Obtains the reference to the wanted stateless or stateful bean
     * using a given configuration. This will create the initial Context
     * and set the bean reference. If no security manager has been defined,
     * the default RMI security manager will be created.
     *
     * @throws NamingException  On naming failure.
     * @throws IOException  On configuration file failure.
     */
    @SuppressWarnings("unchecked")
    public void connect() throws NamingException, IOException {
        if (configuration == null) {
            throw new RuntimeException("This EJB3Connector has not been initialized");
        }
        if (Boolean.parseBoolean(configuration.get("Wrapper.INIT_SECURITY_MANAGER"))
                && System.getSecurityManager() == null) {
            if(System.getProperty("java.security.policy") == null) {
                File temp = File.createTempFile("java.security.policy.", ".tmp");
                FileOutputStream s = new FileOutputStream(temp);
                s.write("grant { permission java.security.AllPermission; };".getBytes());
                s.close();
                System.setProperty("java.security.policy", temp.toString());
            }
            System.setSecurityManager(new RMISecurityManager());
        }
        Hashtable<String, Object> env = new Hashtable<String, Object>(configuration.size() - 1);
        for(String k: keys.keySet()) {
            String value = configuration.get("Context."+k);
            if(value != null) {
                env.put(keys.get(k), value);
            }
        }
        Context initialContext = new InitialContext(env);
        remote = (BeanType) initialContext.lookup(configuration.get("Wrapper.LOOKUP_BEAN"));
    }

    /**
     * Sets the reference to the wanted stateless or stateful bean to null.
     */
    public void disconnect() {
        remote = null;
    }
}
