/**
 * JASMINe
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventEB;
import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventSLBRemote;

/**
 * Implementation of the JasmineEvent stateless bean.
 */
@Stateless
@Remote(JasmineEventSLBRemote.class)
public class JasmineEventSLBImpl implements JasmineEventSLBRemote {
    /**
     * Entity manager used by this session bean.
     */
    @PersistenceContext
    private EntityManager entityManager = null;

    /**
     * Implementation of inherited method: creates the JasmineEvent entity bean
     * corresponding to given datum and saves an event on the database.
     *
     * @see JasmineEventSLBRemote#saveEvent(JasmineEventEB)
     *
     * @param e  Event to save.
     */
    public void saveEvent(final JasmineEventEB e) {
        JasmineEventEBImpl bean = new JasmineEventEBImpl();
        bean.setDomain(e.getDomain());
        bean.setServer(e.getServer());
        bean.setSource(e.getSource());
        bean.setProbe(e.getProbe());
        bean.setValue(e.getValue());
        bean.setTimestamp(e.getTimestamp());
        bean.setSname(e.getSname());
        entityManager.persist(bean);
    }

    /**
     * Implementation of inherited method: looks for the beans searched for,
     * converts the search into an array of JasmineEventEB objects and returns.
     *
     * @see JasmineEventSLBRemote#getEvents(String, String, String, String, String, Date, Date, String, int)
     *
     * @param domain          Domain of the probed server.
     * @param server          Name of the probed server.
     * @param source          Event source (for example, MBeanCmd or JMX).
     * @param probe           Probe type.
     * @param value           Probed value.
     * @param startTimestamp  Probe time bigger than.
     * @param endTimestamp    Probe time smaller than.
     * @param orderBy         Order using column.
     * @param limit           Maximal number of results to get.
     *
     * @return  Events corresponding to the given search, null if none found.
     */
    public JasmineEventEB[] getEvents(final String domain, final String server, final String source, final String probe,
                                      final String value, final Date startTimestamp, final Date endTimestamp,
                                      final String orderBy, final int limit) {
        HashMap<String, Object> queryElements = new HashMap<String, Object>();
        queryElements.put("domain", domain);
        queryElements.put("server", server);
        queryElements.put("source", source);
        queryElements.put("probe", probe);
        queryElements.put("value", value);
        queryElements.put("startTimestamp", startTimestamp);
        queryElements.put("endTimestamp", endTimestamp);

        StringBuffer stringQuery = new StringBuffer("SELECT e FROM JasmineEventEBImpl e");
        for(String key : queryElements.keySet()) {
            if(queryElements.get(key) != null) {
                if(stringQuery.length() <= "SELECT e FROM JasmineEventEBImpl e".length()) {
                    stringQuery.append(" WHERE ");
                } else {
                    stringQuery.append(" OR ");
                }
                if(key.startsWith("start")) {
                    stringQuery.append(("e." + key.charAt(5)).toLowerCase(Locale.ENGLISH));
                    stringQuery.append(key.substring(6));
                    stringQuery.append(" >= :");
                } else if(key.startsWith("end")) {
                    stringQuery.append(("e." + key.charAt(3)).toLowerCase(Locale.ENGLISH));
                    stringQuery.append(key.substring(4));
                    stringQuery.append(" <= :");
                } else {
                    stringQuery.append("e.");
                    stringQuery.append(key);
                    stringQuery.append(" LIKE :");
                }
                stringQuery.append(key);
            }
        }
        if(orderBy != null) {
            stringQuery.append(" ORDER BY e.");
            stringQuery.append(orderBy);
            stringQuery.append(" ASC");
        }

        Query dbQuery = entityManager.createQuery(stringQuery.toString());
        dbQuery.setMaxResults(limit);
        for(String key : queryElements.keySet()) {
            if(queryElements.get(key) != null) {
                dbQuery.setParameter(key, queryElements.get(key));
            }
        }
        List<JasmineEventEBImpl> queryResult = dbQuery.getResultList();

        if(queryResult.size() < 1) {
            return null;
        } else {
            int i = 0;
            JasmineEventEB[] result = new JasmineEventEB[queryResult.size()];
            for(JasmineEventEBImpl bean : queryResult) {
                result[i++] = new JasmineEventEB(bean.getDomain(), bean.getServer(), bean.getSource(),
                                                 bean.getProbe(), bean.getValue(), bean.getTimestamp(),
                                                 bean.getSname());
            }
            return result;
        }
    }
}
