/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineEventNotificationEBImpl.java 5169 2009-09-08 12:32:01Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.Table;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;

import org.ow2.jasmine.event.beans.JasmineEventNotificationDetail;
import org.ow2.jasmine.event.beans.JasmineEventNotificationExecutedAction;

/**
 * Entity bean for a JASMINe Event Notification.
 *
 * @author HAN Tianyi
 */

@Entity
@Table(name = "JASMINE_EVENT_NOTIFICATIONS")
public class JasmineEventNotificationEBImpl implements Serializable {

    /**
     *
     */
    private static final long serialVersionUID = 3603043818875621406L;

    /**
     ** Primary key.
     */
    @Id
    private String id;

    /**
     ** Notification date.
     */
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "dateStamp")
    private Date date;

    /**
     ** Notification level.
     */
    @Column(name = "levelDegree")
    private String level;

    /**
     ** Associated message.
     */
    private String message;

    /**
     ** Notification status acquitted or not.
     */
    private Boolean status;

    /**
     ** The rule id that caused the notification.
     */
    private String rule;

    /**
     * notification additional informations.
     */
    @OneToMany(mappedBy = "notification", cascade = CascadeType.ALL)
    @OrderBy("key ASC")
    private List<JasmineEventDetailEB> details;

    /**
     * Actions executed on the notification.
     */
    @OneToMany(mappedBy = "notification", cascade = CascadeType.ALL)
    private List<JasmineEventNotificationActionEBImpl> actions;

    /**
     * default constructor.
     */
    public JasmineEventNotificationEBImpl() {
        super();
        details = new ArrayList<JasmineEventDetailEB>();
        actions = new ArrayList<JasmineEventNotificationActionEBImpl>();
    }

    // ------------------------------------------------------------------------
    // Getters & setters
    // ------------------------------------------------------------------------

    /**
     * @return Primary key
     */
    public String getId() {
        return id;
    }

    /**
     * @param i Primary key
     */
    public void setId(final String i) {
        this.id = i;
    }

    /**
     * @return Time of Notification (precise up to a second).
     */
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "dateStamp")
    public Date getDate() {
        return date;
    }

    /**
     * @param d Time of Notification (precise up to a second).
     */
    @Column(name = "dateStamp")
    public void setDate(final Date d) {
        this.date = d;
    }

    /**
     * @return Level of Notification
     */
    @Column(name = "levelDegree")
    public String getLevel() {
        return level;
    }

    /**
     * @param l Level of Notification
     */
    @Column(name = "levelDegree")
    public void setLevel(final String l) {
        this.level = l;
    }

    /**
     * @return Message of Notification
     */
    public String getMessage() {
        return message;
    }

    /**
     * @param m Message of Notification
     */
    public void setMessage(final String m) {
        this.message = m;
    }

    /**
     * @return Status of Notification
     */
    public Boolean getStatus() {
        return status;
    }

    /**
     * @param s Status of Notification
     */
    public void setStatus(final Boolean s) {
        this.status = s;
    }

    /**
     * @return The id of rule for Notification
     */
    public String getRule() {
        return rule;
    }

    /**
     * @param r id Id of rule for Notification
     */
    public void setRule(final String r) {
        this.rule = r;
    }

    /**
     * @return the details
     */
    public List<JasmineEventDetailEB> getDetails() {
        return details;
    }

    /**
     * @param details the details to set
     */
    public void setDetails(final List<JasmineEventDetailEB> details) {
        this.details = details;
    }

    /**
     * @param detail the detail to add
     */
    public void addDetail(final JasmineEventDetailEB detail) {
        if (!this.getDetails().contains(detail)) {
            this.getDetails().add(detail);
        }
    }

    /**
     * @return the details list of this notification
     */
    public List<JasmineEventNotificationDetail> getNotificationDetails() {
        List<JasmineEventNotificationDetail> detailsList = new ArrayList<JasmineEventNotificationDetail>();
        Iterator<JasmineEventDetailEB> ite = this.details.iterator();

        while (ite.hasNext()) {
            JasmineEventDetailEB detail = ite.next();
            detailsList.add(new JasmineEventNotificationDetail(detail.getKey(), detail.getValue()));
        }
        return detailsList;
    }

    /**
     * @return the actions
     */
    public List<JasmineEventNotificationExecutedAction> getActions() {

        List<JasmineEventNotificationExecutedAction> actions = new ArrayList<JasmineEventNotificationExecutedAction>();
        Iterator<JasmineEventNotificationActionEBImpl> iter = this.actions.iterator();

        while (iter.hasNext()) {
            JasmineEventNotificationActionEBImpl jasmineEventNotificationActionEBImpl = iter.next();
            actions.add(new JasmineEventNotificationExecutedAction(jasmineEventNotificationActionEBImpl.getComment(),
                jasmineEventNotificationActionEBImpl.getAction(), jasmineEventNotificationActionEBImpl.getDate(),
                jasmineEventNotificationActionEBImpl.getUser()));
        }

        return actions;
    }

    /**
     * @param actions the actions to set
     */
    public void setActions(final List<JasmineEventNotificationActionEBImpl> actions) {
        this.actions = actions;
    }

    public void addAction(final JasmineEventNotificationActionEBImpl action) {
        this.actions.add(action);
    }
}
