/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.ow2.jasmine.event.beans.JasmineEventNotification;
import org.ow2.jasmine.event.beans.JasmineEventNotificationDetail;
import org.ow2.jasmine.event.beans.JasmineEventNotificationExecutedAction;
import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventNotificationSLBRemote;
import org.ow2.util.log.LogFactory;

/**
 * Implementation of the JasmineEventNotification stateless bean.
 *
 * @author HAN Tianyi
 */
@Stateless(mappedName = "db-ejb/notification")
@Remote(JasmineEventNotificationSLBRemote.class)
public class JasmineEventNotificationSLBImpl implements JasmineEventNotificationSLBRemote {
    /**
     * Entity manager used by this session bean.
     */
    @PersistenceContext
    private EntityManager entityManager = null;

    /**
     * Saves the action executed by a user in the DataBase.
     *
     * @see JasmineEventNotificationSLBRemote#saveExecutedAction(String,
     *      JasmineEventNotificationExecutedAction)
     * @param id the Id of the notification to update.
     * @param action the action to save.
     */

    public void saveExecutedAction(final String id, final JasmineEventNotificationExecutedAction action) {

        // get the bean from the data base.
        JasmineEventNotificationEBImpl notif = entityManager.find(JasmineEventNotificationEBImpl.class, id);

        // create the entity bean of the action to add.
        JasmineEventNotificationActionEBImpl actionEB = new JasmineEventNotificationActionEBImpl(action.getComment(), action
            .getAction(), action.getDate(), action.getUser());

        actionEB.setNotification(notif);

        notif.addAction(actionEB);

    }

    /**
     * Updates the status of the notification : acknowledge or not.
     *
     * @param id the id of the notification to update.
     * @param status the new status to save.
     */
    public void updateNotificationStatus(final String id, final boolean status) {

        // get the bean from the data base.
        JasmineEventNotificationEBImpl notif = entityManager.find(JasmineEventNotificationEBImpl.class, id);

        notif.setStatus(status);

    }

    /**
     * Implementation of inherited method: creates the JasmineEventNotification
     * entity bean corresponding to given datum and saves an event on the
     * database.
     *
     * @see JasmineEventNotificationSLBRemote#saveEvent(JasmineEventNotificationEB)
     * @param n Notification to save.
     */
    public void saveEvent(final JasmineEventNotification n) {
        try {
            // bean to be persisted
            JasmineEventNotificationEBImpl bean = new JasmineEventNotificationEBImpl();
            // list of his given details
            List<JasmineEventDetail> details = new ArrayList<JasmineEventDetail>();
            List<JasmineEventNotificationActionEBImpl> actions = new ArrayList<JasmineEventNotificationActionEBImpl>();

            bean.setId(n.getId());
            bean.setDate(n.getDate());
            bean.setLevel(n.getLevel());
            bean.setMessage(n.getMessage());
            bean.setRule(n.getRule());
            bean.setStatus(n.getStatus());

            if (!n.getDetails().isEmpty()) {
                for (JasmineEventNotificationDetail entry : n.getDetails()) {
                    JasmineEventDetail detail = new JasmineEventDetail(entry.getLabel(), entry.getMessage());
                    detail.setNotification(bean);
                    details.add(detail);
                }
                bean.setDetails(details);
            }

            if (!n.getActions().isEmpty()) {
                for (JasmineEventNotificationExecutedAction entry : n.getActions()) {
                    JasmineEventNotificationActionEBImpl action = new JasmineEventNotificationActionEBImpl(entry.getComment(),
                        entry.getAction(), entry.getDate(), entry.getUser());
                    action.setNotification(bean);
                    actions.add(action);
                }
                bean.setActions(actions);
            }

            entityManager.persist(bean);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Implementation of inherited method: looks for the beans searched for,
     * converts the search into an array of JasmineEventNotificationEB objects
     * and returns.
     *
     * @see public JasmineEventNotification[] getEventNotifications(int, Date,
     *      String, String, Boolean, int, String, int);
     * @param id Notification id
     * @param date Notification date
     * @param level Notification level
     * @param message Associated message
     * @param status Notification status
     * @param rule The rule id that caused the notification
     * @param orderBy Order using column.
     * @param limit Maximal number of results to get.
     * @return EventNotifications corresponding to the given search, null if
     *         none found.
     */
    @SuppressWarnings( {"unchecked"})
    public JasmineEventNotification[] getEventNotifications(final String id, final Date date, final String level,
        final String message, final Boolean status, final String rule, final String orderBy, final Integer limit) {
        HashMap<String, Object> queryElements = new HashMap<String, Object>();
        queryElements.put("id", id);
        queryElements.put("date", date);
        queryElements.put("level", level);
        queryElements.put("message", message);
        queryElements.put("status", status);
        queryElements.put("rule", rule);

        StringBuffer stringQuery = new StringBuffer("SELECT e FROM JasmineEventNotificationEBImpl e");

        for (String key : queryElements.keySet()) {
            if (queryElements.get(key) != null) {
                if (stringQuery.length() > "SELECT e FROM JasmineEventNotificationEBImpl e".length()) {
                    stringQuery.append(" AND ");
                } else {
                    stringQuery.append(" WHERE ");
                }
                stringQuery.append("e.");
                stringQuery.append(key);
                stringQuery.append(" LIKE :");
                stringQuery.append(key);
            }
        }
        if (orderBy != null) {
            stringQuery.append(" ORDER BY e.");
            stringQuery.append(orderBy);
            stringQuery.append(" ASC");
        }

        Query dbQuery = entityManager.createQuery(stringQuery.toString());
        if (limit != null) {
            dbQuery.setMaxResults(limit);
        }
        for (String key : queryElements.keySet()) {
            if (queryElements.get(key) != null) {
                dbQuery.setParameter(key, queryElements.get(key));
            }
        }

        LogFactory.getLog(this.getClass()).debug("Executing the following query : \n\t{0}", stringQuery);

        List<JasmineEventNotificationEBImpl> queryResult = dbQuery.getResultList();

        if (queryResult.size() < 1) {
            return null;
        } else {
            int i = 0;
            JasmineEventNotification[] result = new JasmineEventNotification[queryResult.size()];

            for (JasmineEventNotificationEBImpl bean : queryResult) {
                JasmineEventNotification notificationResult = new JasmineEventNotification(bean.getId(), bean.getDate(), bean
                    .getLevel(), bean.getMessage(), bean.getStatus(), bean.getRule());
                notificationResult.setDetails(bean.getNotificationDetails());
                notificationResult.setActions(bean.getActions());
                result[i++] = notificationResult;
            }
            return result;
        }
    }

    /**
     * Gets the event notification between two dates.
     *
     * @param dateFrom the date from
     * @param dateTo the date to
     * @return a list of {@link JasmineEventNotification}
     */
    @SuppressWarnings("unchecked")
    public List<JasmineEventNotification> getEventNotifications(final Date dateFrom, final Date dateTo) {

        LogFactory.getLog(this.getClass()).debug("Query dateFrom {0} to dateTo {1}", dateFrom, dateTo);

        StringBuffer stringQuery = new StringBuffer("SELECT e FROM JasmineEventNotificationEBImpl e");

        stringQuery.append(" WHERE ");
        stringQuery.append(" e.date >= :dateFrom AND e.date <= :dateTo");

        HashMap<String, Object> queryElements = new HashMap<String, Object>();
        queryElements.put("dateFrom", dateFrom);
        queryElements.put("dateTo", dateTo);

        Query dbQuery = entityManager.createQuery(stringQuery.toString());

        for (String key : queryElements.keySet()) {
            if (queryElements.get(key) != null) {
                dbQuery.setParameter(key, queryElements.get(key));
            }
        }

        List<JasmineEventNotificationEBImpl> queryResult = dbQuery.getResultList();

        List<JasmineEventNotification> result = new ArrayList<JasmineEventNotification>();

        if (queryResult.size() < 1) {
            LogFactory.getLog(this.getClass()).debug("No result found for the following query : \n\t{0}", stringQuery);
            return null;
        } else {

            for (JasmineEventNotificationEBImpl bean : queryResult) {
                JasmineEventNotification notificationResult = new JasmineEventNotification(bean.getId(), bean.getDate(), bean
                    .getLevel(), bean.getMessage(), bean.getStatus(), bean.getRule());
                notificationResult.setDetails(bean.getNotificationDetails());
                notificationResult.setActions(bean.getActions());
                result.add(notificationResult);
            }
            return result;
        }
    }

    /**
     * Gets all the notification events from the DataBase.
     *
     * @return A list that contains all the {@link JasmineEventNotification}
     *         stored in the DataBase.
     */
    @SuppressWarnings("unchecked")
    public List<JasmineEventNotification> getAllNotifications() {
        LogFactory.getLog(this.getClass()).debug("Getting all the notifications from the DataBase");

        StringBuffer stringQuery = new StringBuffer("SELECT e FROM JasmineEventNotificationEBImpl e");

        Query dbQuery = entityManager.createQuery(stringQuery.toString());

        List<JasmineEventNotificationEBImpl> queryResult = dbQuery.getResultList();

        List<JasmineEventNotification> result = new ArrayList<JasmineEventNotification>();

        if (queryResult.size() < 1) {
            LogFactory.getLog(this.getClass()).debug("No result found for the following query : \n\t{0}", stringQuery);
            return null;
        } else {

            for (JasmineEventNotificationEBImpl bean : queryResult) {
                JasmineEventNotification notificationResult = new JasmineEventNotification(bean.getId(), bean.getDate(), bean
                    .getLevel(), bean.getMessage(), bean.getStatus(), bean.getRule());
                notificationResult.setDetails(bean.getNotificationDetails());
                notificationResult.setActions(bean.getActions());
                result.add(notificationResult);
            }
            return result;
        }
    }

    /**
     * Looks for the given bean in the data base. It returns the associated
     * {@link JasmineEventNotification}.
     *
     * @param id Id of the notification to retrieve.
     * @return the retrieved notification.
     */
    public JasmineEventNotification getEventNotification(final String id) {
        // get the bean from the data base.
        JasmineEventNotificationEBImpl bean = entityManager.find(JasmineEventNotificationEBImpl.class, id);

        JasmineEventNotification notificationResult = new JasmineEventNotification(bean.getId(), bean.getDate(), bean
            .getLevel(), bean.getMessage(), bean.getStatus(), bean.getRule());
        notificationResult.setDetails(bean.getNotificationDetails());
        notificationResult.setActions(bean.getActions());

        return notificationResult;
    }

}
