/**
 * JASMINe
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineEventSLBImpl.java 4798 2009-08-07 12:30:47Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.ow2.jasmine.event.beans.JasmineEventEB;
import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventSLBRemote;

/**
 * Implementation of the JasmineEvent stateless bean.
 */
@Stateless
@Remote(JasmineEventSLBRemote.class)
public class JasmineEventSLBImpl implements JasmineEventSLBRemote {
    /**
     * Entity manager used by this session bean.
     */
    @PersistenceContext
    private EntityManager entityManager = null;

    /**
     * Implementation of inherited method: creates the JasmineEvent entity bean
     * corresponding to given datum and saves an event on the database.
     *
     * @see JasmineEventSLBRemote#saveEvent(JasmineEventEB)
     *
     * @param e  Event to save.
     */
    public void saveEvent(final JasmineEventEB e) {
        JasmineEventEBImpl bean = new JasmineEventEBImpl();
        bean.setDomain(e.getDomain());
        bean.setServer(e.getServer());
        bean.setSource(e.getSource());
        bean.setProbe(e.getProbe());
        bean.setValue(e.getValue().toString());
        bean.setTimestamp(e.getTimestamp());
        bean.setSname(e.getSname());
        entityManager.persist(bean);
    }

    /**
     * Implementation of inherited method: looks for the beans searched for,
     * converts the search into an array of JasmineEventEB objects and returns.
     *
     * @see JasmineEventSLBRemote#getEvents(String, String, String, String, String, Date, Date, String, int)
     *
     * @param domain          Domain of the probed server.
     * @param server          Name of the probed servers.
     * @param source          Event source (for example, MBeanCmd or JMX).
     * @param probe           Probes type.
     * @param value           Probed value.
     * @param startTimestamp  Probe time bigger than.
     * @param endTimestamp    Probe time smaller than.
     * @param orderBy         Order using column.
     * @param limit           Maximal number of results to get.
     *
     * @return  Events corresponding to the given search, null if none found.
     */
    public JasmineEventEB[] getEvents(final String domain, final String server, final String source, final String probe,
                                      final String value, final Date startTimestamp, final Date endTimestamp,
                                      final String orderBy, final int limit) {
        HashMap<String, Object> queryElements = new HashMap<String, Object>();
        queryElements.put("domain", domain);
        queryElements.put("source", source);
        queryElements.put("value", value);
        queryElements.put("startTimestamp", startTimestamp);
        queryElements.put("endTimestamp", endTimestamp);

        //split the probe string to get all the probes
        String[] probes = probe.split("separator");
        //split the server string to get all the servers
        String[] servers = server.split("separator");
        for (int j = 0; j < probes.length; j++) {
            queryElements.put("probe"+j, probes[j]);
            queryElements.put("server"+j, servers[j]);
        }

        //create the query from the elements of the HashMap
        //if there's no value for one element, it's not added to the request
        StringBuffer stringQuery = new StringBuffer("SELECT e FROM JasmineEventEBImpl e WHERE ");
        Set<String> keysFromHT = queryElements.keySet();
        Object[] keys = keysFromHT.toArray();
        for (int i=0 ; i < keys.length; i++){
            if (queryElements.get(keys[i].toString()) != null){
                if (keys[i].toString().startsWith("domain")){
                    stringQuery.append("e.domain LIKE :");
                    stringQuery.append(keys[i]);
                    stringQuery.append(" AND ");
                }
                if (keys[i].toString().startsWith("source")){
                    stringQuery.append("e.source LIKE :");
                    stringQuery.append(keys[i]);
                    stringQuery.append(" AND ");
                }
                if (keys[i].toString().startsWith("value")){
                    stringQuery.append("e.value LIKE :");
                    stringQuery.append(keys[i]);
                    stringQuery.append(" AND ");
                }
                if (keys[i].toString().startsWith("startTimestamp")){
                    stringQuery.append("e.timestamp >= :");
                    stringQuery.append(keys[i]);
                    stringQuery.append(" AND ");
                }
                if (keys[i].toString().startsWith("endTimestamp")){
                    stringQuery.append("e.timestamp <= :");
                    stringQuery.append(keys[i]);
                    stringQuery.append(" AND ");
                }
            }
        }
        //this part of the request is special because the probes and the servers must correspond
        for (int i=0; i < keys.length; i++){
            if (keys[i].toString().startsWith("probe0")){
                stringQuery.append(" (");
            }
            if (keys[i].toString().startsWith("probe")){
                stringQuery.append("e.probe LIKE :");
                stringQuery.append(keys[i]);
                stringQuery.append(" AND ");
                for (int j=0; j < keys.length; j++){
                    if (keys[j].toString().startsWith("server"+keys[i].toString().charAt(5))){
                        stringQuery.append("e.server LIKE :");
                        stringQuery.append(keys[j]);
                        if (keys[i].toString().startsWith("probe"+(probes.length-1))){
                            stringQuery.append(") ");
                        }
                        else {
                            stringQuery.append(" OR ");
                        }
                    }
                }
            }
        }

        if(orderBy != null) {
            stringQuery.append(" ORDER BY e.");
            stringQuery.append(orderBy);
            stringQuery.append(" ASC");
        }

        //replace the keys by the real values contained in the HashMap
        Query dbQuery = entityManager.createQuery(stringQuery.toString());
        dbQuery.setMaxResults(limit);
        for(String key : queryElements.keySet()) {
            if(queryElements.get(key) != null) {
                dbQuery.setParameter(key, queryElements.get(key));
            }
        }

        List<JasmineEventEBImpl> queryResult = dbQuery.getResultList();

        if(queryResult.size() < 1) {
            return null;
        } else {
            int i = 0;
            JasmineEventEB[] result = new JasmineEventEB[queryResult.size()];
            for(JasmineEventEBImpl bean : queryResult) {
                result[i++] = new JasmineEventEB(bean.getDomain(), bean.getServer(), bean.getSource(),
                                                 bean.getProbe(), bean.getValue(), bean.getTimestamp(),
                                                 bean.getSname());
            }
            return result;
        }
    }
}
