/**
 * JASMINe
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineEventSLBImpl.java 5131 2009-09-02 13:50:03Z rouxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.logging.Logger;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.ejb.Remote;
import javax.ejb.SessionContext;
import javax.ejb.Stateless;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerService;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.ow2.jasmine.event.messages.JasmineEventTimer;
import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventTimerSLBRemote;

/**
 * Implementation of the JasmineEvent timer bean. Aims to purge depreciated
 * event entries on the database. Listens for a JasmineEventTimer to get
 * configured. Can execute the purge immediately or can trigger an EJB Timer to
 * execute periodically.
 *
 *@author Arda Aydin
 */
@Stateless(mappedName = "db-ejb/timer")
@Remote(JasmineEventTimerSLBRemote.class)
public class JasmineEventTimerSLBImpl implements JasmineEventTimerSLBRemote {
    /**
     * Jasmine timer event entity bean implementation class name. The name of
     * this class is used in the JPQL query that get events. Allow easy
     * refactoring.
     */
    static final String EVENT_CLASS_NAME = JasmineEventEBImpl.class.getName();

    Logger logger = Logger.getLogger(this.getClass().getName());

    // To persist only one instance of entity bean
    private static final int FIXED_UNIQUE_ID = 1;

    @Resource
    private SessionContext sessionContext;

    /**
     * Obtain a reference to the TimerService object
     */
    TimerService timerService = null;

    Timer currentTimer = null;

    /**
     * Entity manager used by this session bean.
     */
    @PersistenceContext
    private EntityManager entityManager = null;

    JasmineEventTimerEB bean = new JasmineEventTimerEB();

    @PostConstruct
    private void getTimerService() {
        timerService = sessionContext.getTimerService();
    }

    private void loadTimerEB() {
        bean = entityManager.find(JasmineEventTimerEB.class, 0);
    }

    private void saveTimerEB() {
        entityManager.persist(bean);
    }

    private void createTimer() {
        // clear all existent timers
        clearOldTimers();
        // create a new one
        currentTimer = timerService.createTimer(bean.getExecutionStartDate(), bean.getExecutionInterval(), "");
    }

    private void clearOldTimers() {
        Collection<Timer> timersCollection = timerService.getTimers();
        // For a set or list
        for (Iterator<Timer> it = timersCollection.iterator(); it.hasNext();) {
            Timer timer = it.next();
            timer.cancel();
        }
        timersCollection.clear();

    }


    /**
     *
     * Public method called via remote interface to configure the timer class.
     *
     */
    public int configureTimer(final JasmineEventTimer e) {

        bean.setId(FIXED_UNIQUE_ID);

        bean.setMaxEntryAge(e.getMaxEntryAge());
        bean.setMaxEntryNumber(e.getMaxEntryNumber());
        bean.setImmediateExecution(e.isImmediateExecution());
        bean.setExecuteOnAllServers(e.isExecuteOnAllServers());
        if (!e.isExecuteOnAllServers()) {
            bean.setDomainName(e.getDomainName());
            bean.setServerName(e.getServerName());
        }

        if (e.isImmediateExecution()) {
            clearOldTimers();
            deleteEventsFromDB();
        } else {
            Date date = e.getExecutionStartDate();
            int startHour = e.getExecutionStartHours() / 24;
            int startMinute = e.getExecutionStartHours() % 24;
            date.setHours(startHour);
            date.setMinutes(startMinute);
            bean.setExecutionStartDate(date);
            bean.setExecutionInterval(e.getExecutionInterval());

            // saveTimerEB();
            createTimer();
        }

        return 1;

    }

    /**
     * Method to make EJBQL queries to delete appropriate event entries from the database.
     *
     */
    private void deleteEventsFromDB() {

        Long maxPrimaryKeyId;
        Long minPrimaryKeyId;
        Long maxNumber = new Long(bean.getMaxEntryNumber());

        // query construction
        StringBuilder stringQueryMax = new StringBuilder();

        stringQueryMax.append("Select max(e.id) from ").append(EVENT_CLASS_NAME).append(" e");

        Query dbQueryMax = entityManager.createQuery(stringQueryMax.toString());
        maxPrimaryKeyId = (Long) dbQueryMax.getSingleResult();

        if (maxPrimaryKeyId != null) {
            minPrimaryKeyId = maxPrimaryKeyId - maxNumber; // last n entries

            Calendar cal = Calendar.getInstance();

            Date endDate = cal.getTime(); // time of now
            long timeDiff = bean.getMaxEntryAge();
            Date startDate = new Date(endDate.getTime() - timeDiff); // difference
            // of time
            // in
            // milliseconds

            // query construction
            StringBuilder stringQuery = new StringBuilder();
            stringQuery.append("DELETE FROM ").append(EVENT_CLASS_NAME).append(" e").append(
                " WHERE ((e.timestamp NOT BETWEEN :start AND :end)").append(" OR (e.id NOT BETWEEN :min AND :max))");

            if (!bean.isExecuteOnAllServers()) {
                stringQuery.append(" AND e.domain=:domain AND e.server=:server");
            }

            // Create query from stringQuery.
            Query dbQuery = entityManager.createQuery(stringQuery.toString());

            if (bean.isExecuteOnAllServers()) {
                dbQuery.setParameter("start", startDate).setParameter("end", endDate).setParameter("min", minPrimaryKeyId)
                    .setParameter("max", maxPrimaryKeyId);
            } else {
                dbQuery.setParameter("start", startDate).setParameter("end", endDate).setParameter("min", minPrimaryKeyId)
                    .setParameter("max", maxPrimaryKeyId).setParameter("domain", bean.getDomainName()).setParameter("server",
                        bean.getServerName());
            }

            int result = dbQuery.executeUpdate();

            logger.info("[PurgeTimer]Event Timer cleaned up " + result + " events ");
        } else {
            logger.info("[PurgeTimer]No events to delete in database");
        }

    }

    /**
     * Timeout handle method to be triggered every x times by EJB Timer service of the aplication server.
     *
     * @param timer
     */
    @Timeout
    public void handleTimeout(final Timer timer) {
        // loadTimerEB();
        deleteEventsFromDB();

    }

}