/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.ow2.jasmine.event.beans.JasmineEventNotification;
import org.ow2.jasmine.event.beans.JasmineEventNotificationDetail;
import org.ow2.jasmine.event.beans.JasmineEventNotificationExecutedAction;
import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventNotificationSLBRemote;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implementation of the JasmineEventNotification stateless bean.
 *
 * @author HAN Tianyi
 */
@Stateless(mappedName = "db-ejb/notification")
@Remote(JasmineEventNotificationSLBRemote.class)
public class JasmineEventNotificationSLBImpl implements JasmineEventNotificationSLBRemote {

    /**
     * Logger
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * Conversion constant.
     */
    private static final long HOURS_TO_MILLISECONDS = 1000 * 60 * 60;

    /**
     * Entity manager used by this session bean.
     */
    @PersistenceContext(unitName = "persistence-unit/notification")
    private EntityManager entityManager = null;

    /**
     * Saves the action executed by a user in the DataBase.
     *
     * @see JasmineEventNotificationSLBRemote#saveExecutedAction(String,
     *      JasmineEventNotificationExecutedAction)
     * @param id the Id of the notification to update.
     * @param action the action to save.
     */

    public void saveExecutedAction(final String id, final JasmineEventNotificationExecutedAction action) {

        // get the bean from the data base.
        JasmineEventNotificationEBImpl notif = entityManager.find(JasmineEventNotificationEBImpl.class, id);

        // create the entity bean of the action to add.
        JasmineEventNotificationActionEBImpl actionEB = new JasmineEventNotificationActionEBImpl(action.getComment(), action
            .getAction(), action.getDate(), action.getUser());

        actionEB.setNotification(notif);

        notif.addAction(actionEB);

    }

    /**
     * Updates the status of the notification : acknowledge or not.
     *
     * @param id the id of the notification to update.
     * @param status the new status to save.
     */
    public void updateNotificationStatus(final String id, final boolean status) {

        // get the bean from the data base.
        JasmineEventNotificationEBImpl notif = entityManager.find(JasmineEventNotificationEBImpl.class, id);

        notif.setStatus(status);

    }

    /**
     * Implementation of inherited method: creates the JasmineEventNotification
     * entity bean corresponding to given datum and saves an event on the
     * database.
     *
     * @see JasmineEventNotificationSLBRemote#saveEvent(JasmineEventNotificationEB)
     * @param n Notification to save.
     */
    public void saveEvent(final JasmineEventNotification n) {

        // bean to be persisted
        JasmineEventNotificationEBImpl bean = new JasmineEventNotificationEBImpl();
        // list of his given details
        List<JasmineEventDetailEB> details = new ArrayList<JasmineEventDetailEB>();
        List<JasmineEventNotificationActionEBImpl> actions = new ArrayList<JasmineEventNotificationActionEBImpl>();

        bean.setId(n.getId());
        bean.setDate(n.getDate());
        bean.setLevel(n.getLevel());
        bean.setMessage(n.getMessage());
        bean.setRule(n.getRule());
        bean.setStatus(n.getStatus());

        if (!n.getDetails().isEmpty()) {
            for (JasmineEventNotificationDetail entry : n.getDetails()) {
                JasmineEventDetailEB detail = new JasmineEventDetailEB(entry.getLabel(), entry.getMessage());
                detail.setNotification(bean);
                details.add(detail);
            }
            bean.setDetails(details);
        }

        if (!n.getActions().isEmpty()) {
            for (JasmineEventNotificationExecutedAction entry : n.getActions()) {
                JasmineEventNotificationActionEBImpl action = new JasmineEventNotificationActionEBImpl(entry.getComment(),
                    entry.getAction(), entry.getDate(), entry.getUser());
                action.setNotification(bean);
                actions.add(action);
            }
            bean.setActions(actions);
        }

        entityManager.persist(bean);
    }

    /**
     * Implementation of inherited method: Gets the event notifications for a
     * given rule, a "synchronized" flag and a keepDisplayed value.
     *
     * @param rule rule name. might contain * characters at the end or at the
     *        beginning, meaning respectedly "rule starts with" or "rule ends
     *        "with".
     * @param keepDisplayed the keepDisplayed option, expressed in hours:
     *        <ul>
     *          <li>
     *            If "synchronized" is false, only notifications with a date
     *            inferior to now minus keepDisplayed hours will be returned.
     *          </li>
     *          <li>
     *            If  "synchronized" is true, a merge of two types of
     *            notifications is returned:
     *            <ol>
     *              <li>
     *                Notifications with status false (i.e., not acknowledged
     *                yet)
     *              </li>
     *              <li>
     *                Notifications with status true AND at least one action of
     *                type ACKNOWLEDGE with a date to now minus keepDisplayed
     *                hours
     *              </li>
     *            </ol>
     *          </li>
     *        </ul>
     * @param acknowledgeButtonName the name of the ACKNOWLEDGE button. if
     *        null, "synchronized" is considered to be false; otherwise it is
     *        considered to be true.
     * @return a list of {@link JasmineEventNotification}
     */
    @SuppressWarnings("unchecked")
    public List<JasmineEventNotification> getEventNotifications(String rule, final int keepDisplayed,
        final String acknowledgeButtonName) {

        this.logger.debug("getEventNotifications({0}, " + keepDisplayed + ", {1})", rule, acknowledgeButtonName);

        StringBuilder stringQuery = new StringBuilder("SELECT e FROM JasmineEventNotificationEBImpl e WHERE e.rule LIKE :rule");
        Map<String, Object> parameters = new HashMap<String, Object>();

        // We do it this way because the Flex client also does it this way
        if (rule.startsWith("*")) {
            rule = "%%" + rule.substring(1);
        } else if (rule.endsWith("*")) {
            rule = rule.substring(0, rule.length() - 1) + "%%";
        }
        parameters.put("rule", rule);

        long keepDisplayedInMilliseconds = keepDisplayed;
        keepDisplayedInMilliseconds *= JasmineEventNotificationSLBImpl.HOURS_TO_MILLISECONDS;
        final Date keepDisplayedHoursAgo = new Date(System.currentTimeMillis() - keepDisplayedInMilliseconds);
        parameters.put("date", keepDisplayedHoursAgo);
        stringQuery.append(" AND ");

        if (acknowledgeButtonName == null) {
            // If "synchronized" is false, only notifications with a date
            // inferior to now minus keepDisplayed hours will be returned.
            stringQuery.append("e.date > :date");
        } else {
            // If  "synchronized" is true, a merge of two types of
            // notifications is returned:
            //
            //  1. Notifications with status false (i.e., not acknowledged yet)
            //  2. Notifications with status true AND at least one action of
            //     type ACKNOWLEDGE with a date to now minus keepDisplayed
            //     hours
            stringQuery.append("(e.status = false OR (e.status = true AND (");
            stringQuery.append("SELECT COUNT(*) FROM JasmineEventNotificationActionEBImpl a WHERE ");
            stringQuery.append("a.notification = e AND a.date < :date AND a.action = :action");
            stringQuery.append(") = 0))");

            parameters.put("action", acknowledgeButtonName);
        }

        this.logger.debug("executing DB query {0}, parameters {1}", stringQuery, parameters);

        Query dbQuery = entityManager.createQuery(stringQuery.toString());
        for (Map.Entry<String, Object> parameter : parameters.entrySet()) {
            dbQuery.setParameter(parameter.getKey(), parameter.getValue());
        }

        List<JasmineEventNotificationEBImpl> queryResult = dbQuery.getResultList();

        List<JasmineEventNotification> result = new ArrayList<JasmineEventNotification>();

        for (JasmineEventNotificationEBImpl bean : queryResult) {
            JasmineEventNotification notificationResult = new JasmineEventNotification(bean.getId(), bean.getDate(), bean
                .getLevel(), bean.getMessage(), bean.getStatus(), bean.getRule());
            notificationResult.setDetails(bean.getNotificationDetails());
            notificationResult.setActions(bean.getActions());
            result.add(notificationResult);
        }

        this.logger.debug("getEventNotifications returns {0}", result);
        return result;
    }

    /**
     * Gets the event notification between two dates.
     *
     * @param dateFrom the date from
     * @param dateTo the date to
     * @return a list of {@link JasmineEventNotification}
     */
    @SuppressWarnings("unchecked")
    public List<JasmineEventNotification> getEventNotifications(final Date dateFrom, final Date dateTo) {

        this.logger.debug("getEventNotifications({0}, {1})", dateFrom, dateTo);

        StringBuilder stringQuery = new StringBuilder("SELECT e FROM JasmineEventNotificationEBImpl e");

        stringQuery.append(" WHERE ");
        stringQuery.append(" e.date >= :dateFrom AND e.date <= :dateTo");

        Query dbQuery = entityManager.createQuery(stringQuery.toString());

        dbQuery.setParameter("dateFrom", dateFrom);
        dbQuery.setParameter("dateTo", dateTo);

        List<JasmineEventNotificationEBImpl> queryResult = dbQuery.getResultList();

        List<JasmineEventNotification> result = new ArrayList<JasmineEventNotification>();

        for (JasmineEventNotificationEBImpl bean : queryResult) {
            JasmineEventNotification notificationResult = new JasmineEventNotification(bean.getId(), bean.getDate(), bean
                .getLevel(), bean.getMessage(), bean.getStatus(), bean.getRule());
            notificationResult.setDetails(bean.getNotificationDetails());
            notificationResult.setActions(bean.getActions());
            result.add(notificationResult);
        }

        this.logger.debug("getEventNotifications returns {0}", result);
        return result;
    }

    /**
     * Implementation of inherited method: looks for the beans searched for,
     * converts the search into an array of JasmineEventNotificationEB objects
     * and returns.
     *
     * @param id Notification id, ignored if null.
     * @param date Notification date, ignored if null.
     * @param level Notification level, ignored if null.
     * @param message Associated message, ignored if null.
     * @param status Notification status, ignored if null.
     * @param rule The rule id that caused the notification, ignored if null.
     * @param orderBy Order using column, ignored if null..
     * @param limit Maximal number of results to get, ignored if null.
     * @return EventNotifications corresponding to the given search, null if
     *         none found.
     */
    @SuppressWarnings("unchecked")
    public JasmineEventNotification[] getEventNotifications(final String id, final Date date, final String level,
        final String message, final Boolean status, final String rule, final String orderBy, final Integer limit) {

        this.logger.debug("getEventNotifications({0}, {1}, {2}, {3}, {4}, {5}, {6}, {7})", id, date, level, message, status,
            rule, orderBy, limit);

        HashMap<String, Object> queryElements = new HashMap<String, Object>();
        queryElements.put("id", id);
        queryElements.put("date", date);
        queryElements.put("level", level);
        queryElements.put("message", message);
        queryElements.put("status", status);
        queryElements.put("rule", rule);

        StringBuilder stringQuery = new StringBuilder("SELECT e FROM JasmineEventNotificationEBImpl e");

        boolean WHERE = true;
        for (String key : queryElements.keySet()) {
            if (queryElements.get(key) != null) {
                if (WHERE) {
                    WHERE = false;
                    stringQuery.append(" WHERE ");
                } else {
                    stringQuery.append(" AND ");
                }
                stringQuery.append("e.");
                stringQuery.append(key);
                stringQuery.append(" LIKE :");
                stringQuery.append(key);
            }
        }
        if (orderBy != null) {
            stringQuery.append(" ORDER BY e.");
            stringQuery.append(orderBy);
            stringQuery.append(" ASC");
        }

        Query dbQuery = entityManager.createQuery(stringQuery.toString());
        if (limit != null) {
            dbQuery.setMaxResults(limit);
        }
        for (String key : queryElements.keySet()) {
            if (queryElements.get(key) != null) {
                dbQuery.setParameter(key, queryElements.get(key));
            }
        }

        this.logger.debug("executing DB query {0}, parameters {1}", stringQuery, queryElements);

        List<JasmineEventNotificationEBImpl> queryResult = dbQuery.getResultList();

        if (queryResult.size() < 1) {
            return null;
        } else {
            int i = 0;
            JasmineEventNotification[] result = new JasmineEventNotification[queryResult.size()];

            for (JasmineEventNotificationEBImpl bean : queryResult) {
                JasmineEventNotification notificationResult = new JasmineEventNotification(bean.getId(), bean.getDate(), bean
                    .getLevel(), bean.getMessage(), bean.getStatus(), bean.getRule());
                notificationResult.setDetails(bean.getNotificationDetails());
                notificationResult.setActions(bean.getActions());
                result[i++] = notificationResult;
            }
            return result;
        }
    }

    /**
     * Gets all the notification events from the DataBase.
     *
     * @return A list that contains all the {@link JasmineEventNotification}
     *         stored in the DataBase.
     */
    @SuppressWarnings("unchecked")
    public List<JasmineEventNotification> getAllNotifications() {
        this.logger.debug("Getting all the notifications from the DataBase");

        String stringQuery = "SELECT e FROM JasmineEventNotificationEBImpl e";

        Query dbQuery = entityManager.createQuery(stringQuery);

        List<JasmineEventNotificationEBImpl> queryResult = dbQuery.getResultList();

        List<JasmineEventNotification> result = new ArrayList<JasmineEventNotification>();

        for (JasmineEventNotificationEBImpl bean : queryResult) {
            JasmineEventNotification notificationResult = new JasmineEventNotification(bean.getId(), bean.getDate(), bean
                .getLevel(), bean.getMessage(), bean.getStatus(), bean.getRule());
            notificationResult.setDetails(bean.getNotificationDetails());
            notificationResult.setActions(bean.getActions());
            result.add(notificationResult);
        }
        return result;
    }

    /**
     * Looks for the given bean in the data base. It returns the associated
     * {@link JasmineEventNotification}.
     *
     * @param id Id of the notification to retrieve.
     * @return the retrieved notification.
     */
    public JasmineEventNotification getEventNotification(final String id) {
        // get the bean from the data base.
        JasmineEventNotificationEBImpl bean = entityManager.find(JasmineEventNotificationEBImpl.class, id);

        JasmineEventNotification notificationResult = new JasmineEventNotification(bean.getId(), bean.getDate(), bean
            .getLevel(), bean.getMessage(), bean.getStatus(), bean.getRule());
        notificationResult.setDetails(bean.getNotificationDetails());
        notificationResult.setActions(bean.getActions());

        return notificationResult;
    }

}
