/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineEventTimerSLBImpl.java 5869 2010-01-07 13:55:18Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.logging.Logger;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.ejb.Remote;
import javax.ejb.SessionContext;
import javax.ejb.Stateless;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerService;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.ow2.jasmine.event.messages.JasmineEventTimer;
import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventTimerSLBRemote;

/**
 * Implementation of the JasmineEvent timer bean. Aims to purge depreciated
 * event entries on the database. Listens for a JasmineEventTimer to get
 * configured. Can execute the purge immediately or can trigger an EJB Timer to
 * execute periodically.
 */
@Stateless(mappedName = "db-ejb/timer")
@Remote(JasmineEventTimerSLBRemote.class)
public class JasmineEventTimerSLBImpl implements JasmineEventTimerSLBRemote {
    /**
     * Jasmine timer event entity bean implementation class name. The name of
     * this class is used in the JPQL query that get events. Allow easy
     * refactoring.
     */
    static final String EVENT_CLASS_NAME = JasmineEventEBImpl.class.getName();

    Logger logger = Logger.getLogger(this.getClass().getName());

    // To persist only one instance of entity bean
    private static final int FIXED_UNIQUE_ID = 1;

    @Resource
    private SessionContext sessionContext;

    /**
     * Obtain a reference to the TimerService object
     */
    TimerService timerService = null;

    Timer currentTimer = null;

    /**
     * Entity manager used by this session bean.
     */
    @PersistenceContext(unitName = "persistence-unit/timer")
    private EntityManager entityManager = null;

    JasmineEventTimerEB bean = new JasmineEventTimerEB();



    @PostConstruct
    private void getTimerService() {
        timerService = sessionContext.getTimerService();
    }

    private void loadTimerEB() {
        bean = entityManager.find(JasmineEventTimerEB.class, 0);
    }

    private void saveTimerEB() {
        entityManager.persist(bean);
    }

    private void createTimer() {
        // clear all existent timers
        clearOldTimers();
        // create a new one
        if (bean.getExecutionInterval() > 0) {
            currentTimer = timerService.createTimer(bean.getExecutionStartDate(), bean.getExecutionInterval(), "");
            logger.info("[PurgeTimer]Timer configured, first execution : " + bean.getExecutionStartDate().toString());
        } else {
            logger.warning("Timer Interval must be greater than zero.");
        }

    }

    private void clearOldTimers() {
        Collection<Timer> timersCollection = timerService.getTimers();
        // For a set or list
        for (Iterator<Timer> it = timersCollection.iterator(); it.hasNext();) {
            Timer timer = it.next();
            timer.cancel();
        }
        timersCollection.clear();

    }

    /**
     * Public method called via remote interface to configure the timer class.
     */
    public int configureTimer(final JasmineEventTimer e) {

        logger.fine("------------------------------------\n\n\n");
        logger.fine(" Event Received with attributes:");
        logger.fine(" - MaxEntryAge :" + e.getMaxEntryAge());
        logger.fine(" - MaxEntryNumber :" + e.getMaxEntryNumber());
        logger.fine(" - ImmediateExecution :" + e.isImmediateExecution());
        logger.fine(" - ExecuteOnAllServers :" + e.isExecuteOnAllServers());
        logger.fine(" - DomainName :" + e.getDomainName());
        logger.fine(" - ServerName :" + e.getServerName());
        logger.fine(" - ExecutionStartDate :" + e.getExecutionStartDate());
        logger.fine(" - ExecutionStartHours :" + e.getExecutionStartHours());
        logger.fine(" - ExecutionInterval :" + e.getExecutionInterval());
        logger.fine("\n\n\n------------------------------------");

        bean.setId(FIXED_UNIQUE_ID);

        bean.setMaxEntryAge(e.getMaxEntryAge());
        bean.setMaxEntryNumber(e.getMaxEntryNumber());
        bean.setImmediateExecution(e.isImmediateExecution());
        bean.setExecuteOnAllServers(e.isExecuteOnAllServers());
        if (!e.isExecuteOnAllServers()) {
            bean.setDomainName(e.getDomainName());
            bean.setServerName(e.getServerName());
        }

        if (e.isImmediateExecution()) {
            deleteEventsFromDB();
        } else {

            bean.setExecutionStartDate(e.getExecutionStartDate());
            bean.setExecutionInterval(e.getExecutionInterval());

            // saveTimerEB();
            createTimer();
        }

        return 1;

    }

    /**
     * Method to make EJBQL queries to delete appropriate event entries from the
     * database.
     */
    private void deleteEventsFromDB() {

        Long maxPrimaryKeyId;
        Long minPrimaryKeyId;
        Long maxNumber = new Long(bean.getMaxEntryNumber());

        // query construction
        StringBuilder stringQueryMax = new StringBuilder();

        stringQueryMax.append("Select max(e.id) from ").append(EVENT_CLASS_NAME).append(" e");

        Query dbQueryMax = entityManager.createQuery(stringQueryMax.toString());
        maxPrimaryKeyId = (Long) dbQueryMax.getSingleResult();

        if (maxPrimaryKeyId != null) {
            minPrimaryKeyId = maxPrimaryKeyId - maxNumber; // last n entries

            Calendar cal = Calendar.getInstance();
            Date endDate = cal.getTime(); // time of now
            long timeDiff = bean.getMaxEntryAge();
            // difference
            // of time
            // in
            // milliseconds
            Date startDate = new Date(endDate.getTime() - timeDiff);

            // query construction
            StringBuilder stringQuery = new StringBuilder();

            if(maxNumber!=-1)
            {
                stringQuery.append("DELETE FROM ").append(EVENT_CLASS_NAME).append(" e").append(
                " WHERE (e.id NOT BETWEEN :min AND :max)");
            }
            else{
                stringQuery.append("DELETE FROM ").append(EVENT_CLASS_NAME).append(" e").append(
                " WHERE (e.timestamp NOT BETWEEN :start AND :end)");
            }


            if (!bean.isExecuteOnAllServers()) {
                stringQuery.append(" AND (e.domain=:domain AND e.server=:server)");
            }

            // Create query from stringQuery.
            Query dbQuery = entityManager.createQuery(stringQuery.toString());
            logger.fine(" - Query to db : " + dbQuery.toString());
            logger.fine(" - Query to min : " + minPrimaryKeyId);
            logger.fine(" - Query to max : " + maxPrimaryKeyId);
            logger.fine(" - Query to start : " + startDate);
            logger.fine(" - Query to end : " + endDate);


            if (bean.isExecuteOnAllServers()) {
                if(maxNumber!=-1)
                {
                    dbQuery
                    .setParameter("min", minPrimaryKeyId)
                    .setParameter("max", maxPrimaryKeyId);
                }
                else
                {
                    dbQuery
                    .setParameter("start", startDate)
                    .setParameter("end", endDate);
                }
            } else {
                if(maxNumber!=-1)
                {
                    dbQuery
                    .setParameter("min", minPrimaryKeyId)
                    .setParameter("max", maxPrimaryKeyId)
                    .setParameter("domain", bean.getDomainName())
                    .setParameter("server", bean.getServerName());
                }
                else
                {
                    dbQuery
                    .setParameter("start", startDate)
                    .setParameter("end", endDate)
                    .setParameter("domain", bean.getDomainName())
                    .setParameter("server",bean.getServerName());
                }
            }



            int result = dbQuery.executeUpdate();

            logger.info("[PurgeTimer]Event Timer cleaned up " + result + " events ");
        } else {
            logger.info("[PurgeTimer]No events to delete in database");
        }

    }

    /**
     * Timeout handle method to be triggered every x times by EJB Timer service
     * of the application server.
     *
     * @param timer
     */
    @Timeout
    public void handleTimeout(final Timer timer) {
        // loadTimerEB();
        deleteEventsFromDB();
        Calendar cal = Calendar.getInstance();
        logger.info("[PurgeTimer]Timer triggered");
    }

}