/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineEventNotificationEBImpl.java 7135 2010-11-17 17:31:11Z joaninh $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.Table;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;

import org.ow2.jasmine.event.beans.JasmineEventNotificationDetail;
import org.ow2.jasmine.event.beans.JasmineEventNotificationExecutedAction;

/**
 * Entity bean for a JASMINe Event Notification.
 *
 * @author HAN Tianyi
 */

@Entity
@Table(name = "JM_EVENT_NOTIFICATIONS")
public class JasmineEventNotificationEBImpl implements Serializable {

    /**
     *
     */
    private static final long serialVersionUID = 3603043818875621406L;

    /**
     * Primary key.
     */
    @Id
    private String id;

    /**
     * Notification date.
     */
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "dateStamp")
    private Date date;

    /**
     * Notification level.
     */
    @Column(name = "levelDegree")
    private String level;

    /**
     * Associated message.
     */
    private String message;

    /**
     * Notification status acknowledged or not.
     */
    private Boolean status;

    /**
     * The rule id that caused the notification.
     */
    private String rule;

    /**
     * notification additional informations.
     */
    @OneToMany(mappedBy = "notification", cascade = CascadeType.ALL)
    @OrderBy("key ASC")
    private List<JasmineEventDetailEB> details;

    /**
     * Actions executed on the notification.
     */
    @OneToMany(mappedBy = "notification", cascade = CascadeType.ALL)
    private List<JasmineEventNotificationActionEBImpl> actions;

    /**
     * default constructor.
     */
    public JasmineEventNotificationEBImpl() {
        super();
        details = new ArrayList<JasmineEventDetailEB>();
        actions = new ArrayList<JasmineEventNotificationActionEBImpl>();
    }

    // ------------------------------------------------------------------------
    // Getters & setters
    // ------------------------------------------------------------------------

    /**
     * @return Primary key
     */
    public String getId() {
        return id;
    }

    /**
     * @param i Primary key
     */
    public void setId(final String i) {
        this.id = i;
    }

    /**
     * @return Time of Notification (precise up to a second).
     */
    @Temporal(TemporalType.TIMESTAMP)
    @Column(name = "dateStamp")
    public Date getDate() {
        return date;
    }

    /**
     * @param d Time of Notification (precise up to a second).
     */
    @Column(name = "dateStamp")
    public void setDate(final Date d) {
        this.date = d;
    }

    /**
     * @return Level of Notification
     */
    @Column(name = "levelDegree")
    public String getLevel() {
        return level;
    }

    /**
     * @param l Level of Notification
     */
    @Column(name = "levelDegree")
    public void setLevel(final String l) {
        this.level = l;
    }

    /**
     * @return Message of Notification
     */
    public String getMessage() {
        return message;
    }

    /**
     * @param m Message of Notification
     */
    public void setMessage(final String m) {
        this.message = m;
    }

    /**
     * @return Status of Notification
     */
    public Boolean getStatus() {
        return status;
    }

    /**
     * @param s Status of Notification
     */
    public void setStatus(final Boolean s) {
        this.status = s;
    }

    /**
     * @return The id of rule for Notification
     */
    public String getRule() {
        return rule;
    }

    /**
     * @param r id Id of rule for Notification
     */
    public void setRule(final String r) {
        this.rule = r;
    }

    /**
     * @return the details
     */
    public List<JasmineEventDetailEB> getDetails() {
        return details;
    }

    /**
     * @param details the details to set
     */
    public void setDetails(final List<JasmineEventDetailEB> details) {
        this.details = details;
    }

    /**
     * Adds a detail to the notification. If an entry already exist with the
     * same key, the value is updated, and <code>true</code> is returned.
     *
     * @param p_Detail the detail to add
     * @return <code>true</code> if the detail has been updated,
     *         <code>false</code> otherwise (added or not).
     */
    public boolean addDetail(final JasmineEventDetailEB p_Detail) {
        boolean result;
        int index = -1;

        // find the index of a detail that may contains the same key as the one
        // to add.
        for (JasmineEventDetailEB detail : this.getDetails()) {
            if (null != detail.getKey() && detail.getKey().equals(p_Detail.getKey())) {
                index = this.getDetails().indexOf(detail);
                break;
            }
        }

        // if the index is -1, then no detail with the key is stored, simply
        // adds the detail, and return false.
        if (-1 == index) {
            if (null == p_Detail.getNotification()) {
                p_Detail.setNotification(this);
            }
            this.getDetails().add(p_Detail);
            result = false;
        }
        // otherwise, a detail with this key already exists, update it and
        // return true
        else {
            this.getDetails().get(index).setValue(p_Detail.getValue());
            result = true;
        }

        return result;
    }

    /**
     * @return the details list of this notification
     */
    public List<JasmineEventNotificationDetail> getNotificationDetails() {
        List<JasmineEventNotificationDetail> detailsList = new ArrayList<JasmineEventNotificationDetail>();
        Iterator<JasmineEventDetailEB> ite = this.details.iterator();

        while (ite.hasNext()) {
            JasmineEventDetailEB detail = ite.next();
            detailsList.add(new JasmineEventNotificationDetail(detail.getKey(), detail.getValue()));
        }
        return detailsList;
    }

    /**
     * @return the actions
     */
    public List<JasmineEventNotificationExecutedAction> getActions() {

        List<JasmineEventNotificationExecutedAction> actions = new ArrayList<JasmineEventNotificationExecutedAction>();
        Iterator<JasmineEventNotificationActionEBImpl> iter = this.actions.iterator();

        while (iter.hasNext()) {
            JasmineEventNotificationActionEBImpl jasmineEventNotificationActionEBImpl = iter.next();
            actions.add(new JasmineEventNotificationExecutedAction(jasmineEventNotificationActionEBImpl.getComment(),
                jasmineEventNotificationActionEBImpl.getAction(), jasmineEventNotificationActionEBImpl.getDate(),
                jasmineEventNotificationActionEBImpl.getUser()));
        }

        return actions;
    }

    /**
     * @param actions the actions to set
     */
    public void setActions(final List<JasmineEventNotificationActionEBImpl> actions) {
        this.actions = actions;
    }

    /**
     * Add an action to the list
     *
     * @param action the {@link JasmineEventNotificationActionEBImpl} to add
     */
    public void addAction(final JasmineEventNotificationActionEBImpl action) {
        if (null == action.getNotification()) {
            action.setNotification(this);
        }
        if (!this.actions.contains(action)) {
            this.actions.add(action);
        }

    }
}
