/**
 * JASMINe
 * Copyright (C) 2009-2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.ejb.Remote;
import javax.ejb.SessionContext;
import javax.ejb.Stateless;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerService;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Message;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.Topic;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.ow2.jasmine.event.messages.JasmineEventPurgeTask;
import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventPurgeTaskSLBRemote;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implementation of the JasmineEvent purge task bean. Aims to clean depreciated
 * event entries on the database. Listens for a JasmineEventPurgeTask to get
 * configured. Can execute the clean immediately or can trigger an EJB Timer to
 * execute periodically.
 */
@Stateless(mappedName = "db-ejb/purgetask")
@Remote(JasmineEventPurgeTaskSLBRemote.class)
public class JasmineEventPurgeTaskSLBImpl implements JasmineEventPurgeTaskSLBRemote {
    /**
     * Jasmine timer event entity bean implementation class name. The name of
     * this class is used in the JPQL query that get events. Allow easy
     * refactoring.
     */
    static final String EVENT_ABSTRACT_SCHEMA_NAME = JasmineEventEBImpl.class.getSimpleName();

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    @Resource
    private SessionContext sessionContext;

    /**
     * Obtain a reference to the TimerService object
     */
    private TimerService timerService = null;

    /**
     * Entity manager used by this session bean.
     */
    @PersistenceContext(unitName = "persistence-unit/purgetask")
    private EntityManager entityManager = null;

    /**
     * JMS Topic Connection Factory
     */
    @Resource(mappedName = "JTCF")
    private ConnectionFactory factory;

    /**
     * JMS Topic discussion
     */
    @Resource(mappedName = "jasminePurgeTask")
    private Topic topic;

    @SuppressWarnings("unused")
    @PostConstruct
    private void getTimerService() {
        timerService = sessionContext.getTimerService();
    }

    /**
     * @see org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventPurgeTaskSLBRemote#activatePeriodicPurgeTasks()
     */
    @SuppressWarnings("all")
    public void activatePeriodicPurgeTasks() {
        cancelAllTimers();

        List<JasmineEventPurgeTaskEB> beans = entityManager.createNamedQuery("ALL_PURGE_TASKS").getResultList();
        for (JasmineEventPurgeTaskEB bean : beans) {
            // Only create a task for a periodic purge task
            if (bean.getExecutionInterval() > 0) {
                createTask(bean);
            }
        }
    }

    /**
     * Timeout handle method to be triggered every x times by EJB Timer service
     * of the application server.
     *
     * @param timer
     */
    @Timeout
    public void handleTimeout(final Timer timer) {
        deleteEventsFromDB(timer.getInfo());
    }

    /**
     * Cancel all timers.
     */
    @SuppressWarnings("unchecked")
    private void cancelAllTimers() {
        Collection<Timer> timers = timerService.getTimers();
        for (Iterator<Timer> it = timers.iterator(); it.hasNext();) {
            Timer timer = it.next();
            timer.cancel();
        }
        timers.clear();
    }

    /**
     * Convert a purge task bean into a purge task event.
     * @param bean The ben to convert
     * @return A purge task event from a given bean.
     */
    private JasmineEventPurgeTask beanToEvent(final JasmineEventPurgeTaskEB bean) {
        JasmineEventPurgeTask event = new JasmineEventPurgeTask();

        event.setId(bean.getId());
        event.setMaxEntryAge(bean.getMaxEntryAge());
        event.setMaxEntryNumber(bean.getMaxEntryNumber());
        event.setExecuteOnAllServers(bean.isExecuteOnAllServers());
        event.setDomainName(bean.getDomainName());
        event.setServerName(bean.getServerName());
        event.setExecutionStartDate(bean.getExecutionStartDate());
        event.setExecutionInterval(bean.getExecutionInterval());
        event.setStarted(bean.isStarted());
        event.setRemovedEvents(bean.getRemovedEvents());
        event.setLastExecutionDate(bean.getLastExecutionDate());

        return event;
    }

    /**
     * Create a new timer corresponding to a purge task. This task can be periodic.
     * @param bean
     */
    private void createTask(final JasmineEventPurgeTaskEB bean) {
        // create a new one
        if (bean.getExecutionInterval() == 0) {
            timerService.createTimer(bean.getExecutionStartDate(), bean.getId());
            logger.info("[DataManagement] Purge Task configured, execution : " + bean.getExecutionStartDate());
        } else {
            timerService.createTimer(bean.getExecutionStartDate(), bean.getExecutionInterval(), bean.getId());
            bean.setStarted(true);
            logger.info("[DataManagement] Purge Task configured, first execution : " + bean.getExecutionStartDate());
        }
    }

    /**
     * Remove a timer corresponding to a purge task.
     * @param bean
     */
    private void removeTask(final JasmineEventPurgeTaskEB bean) {
        Collection timers = timerService.getTimers();
        for (Iterator<Timer> it = timers.iterator(); it.hasNext();) {
            Timer timer = it.next();
            if (Long.parseLong(timer.getInfo().toString()) == bean.getId()) {
                timer.cancel();
            }
        }
        bean.setStarted(false);
        logger.info("[DataManagement] Purge Task removed : " + bean.getId());
    }

    /**
     * @see org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventPurgeTaskSLBRemote#configurePurgeTask(org.ow2.jasmine.event.messages.JasmineEventPurgeTask)
     */
    public JasmineEventPurgeTask configurePurgeTask(final JasmineEventPurgeTask event) {
        JasmineEventPurgeTaskEB bean = new JasmineEventPurgeTaskEB();

        logger.debug("------------------------------------\n\n\n");
        logger.debug(" Event Received with attributes:");
        logger.debug(" - MaxEntryAge :" + event.getMaxEntryAge());
        logger.debug(" - MaxEntryNumber :" + event.getMaxEntryNumber());
        logger.debug(" - ExecuteOnAllServers :" + event.isExecuteOnAllServers());
        logger.debug(" - DomainName :" + event.getDomainName());
        logger.debug(" - ServerName :" + event.getServerName());
        logger.debug(" - ExecutionStartDate :" + event.getExecutionStartDate());
        logger.debug(" - ExecutionInterval :" + event.getExecutionInterval());
        logger.debug("\n\n\n------------------------------------");

        bean.setMaxEntryAge(event.getMaxEntryAge());
        bean.setMaxEntryNumber(event.getMaxEntryNumber());
        bean.setExecuteOnAllServers(event.isExecuteOnAllServers());
        if (!event.isExecuteOnAllServers()) {
            bean.setDomainName(event.getDomainName());
            bean.setServerName(event.getServerName());
        }

        if (event.getExecutionStartDate() == null) {
            // Add 2 seconds to work around bug MONITORING-354
            event.setExecutionStartDate(new Date(System.currentTimeMillis()+2000));
        }

        bean.setExecutionStartDate(event.getExecutionStartDate());
        bean.setExecutionInterval(event.getExecutionInterval());

        entityManager.persist(bean);

        // saveTimerEB();
        createTask(bean);

        event.setId(bean.getId());
        event.setStarted(bean.isStarted());

        return event;
    }

    /**
     * @see org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventPurgeTaskSLBRemote#listPurgeTasks()
     */
    @SuppressWarnings("unchecked")
    public List<JasmineEventPurgeTask> listPurgeTasks() {
        List<JasmineEventPurgeTaskEB> beans = entityManager.createNamedQuery("ALL_PURGE_TASKS").getResultList();
        List<JasmineEventPurgeTask> result = new LinkedList<JasmineEventPurgeTask>();
        for (JasmineEventPurgeTaskEB bean : beans) {
            result.add(beanToEvent(bean));
        }
        return result;
    }

    /**
     * @see org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventPurgeTaskSLBRemote#removePurgeTask(long)
     */
    public long removePurgeTask(final long purgeTaskId) {
        JasmineEventPurgeTaskEB bean = entityManager.find(JasmineEventPurgeTaskEB.class, purgeTaskId);
        if (bean == null) {
            return -1;
        }
        removeTask(bean);
        entityManager.remove(bean);
        return purgeTaskId;
    }

    /**
     * @see org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventPurgeTaskSLBRemote#startPurgeTask(long)
     */
    public long startPurgeTask(final long purgeTaskId) {
        JasmineEventPurgeTaskEB bean = entityManager.find(JasmineEventPurgeTaskEB.class, purgeTaskId);
        if (bean == null) {
            return -1;
        }
        createTask(bean);
        sendPurgeTaskOnTopic(bean);
        return purgeTaskId;
    }

    /**
     * @see org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventPurgeTaskSLBRemote#pausePurgeTask(long)
     */
    public long pausePurgeTask(final long purgeTaskId) {
        JasmineEventPurgeTaskEB bean = entityManager.find(JasmineEventPurgeTaskEB.class, purgeTaskId);
        if (bean == null) {
            return -1;
        }
        removeTask(bean);
        sendPurgeTaskOnTopic(bean);
        return purgeTaskId;
    }

    /**
     * Method to make EJBQL queries to delete appropriate event entries from the
     * database.
     */
    private void deleteEventsFromDB(final Serializable beanId) {
        JasmineEventPurgeTaskEB bean = entityManager.find(JasmineEventPurgeTaskEB.class, beanId);

        if (bean == null) {
            logger.info("[DataManagement] No cleanup parameters found in database");
            return;
        }

        Long maxPrimaryKeyId;
        Long minPrimaryKeyId;
        Long maxNumber = new Long(bean.getMaxEntryNumber());

        // query construction
        StringBuilder stringQueryMax = new StringBuilder();

        stringQueryMax.append("Select max(e.id) from ").append(EVENT_ABSTRACT_SCHEMA_NAME).append(" e");

        Query dbQueryMax = entityManager.createQuery(stringQueryMax.toString());
        maxPrimaryKeyId = (Long) dbQueryMax.getSingleResult();

        if (maxPrimaryKeyId == null) {
            logger.info("[DataManagement] No events to delete in database");
            return;
        }

        minPrimaryKeyId = maxPrimaryKeyId - maxNumber; // last n entries

        Calendar cal = Calendar.getInstance();
        Date endDate = cal.getTime(); // time of now
        long timeDiff = bean.getMaxEntryAge();
        // difference of time in milliseconds
        Date startDate = new Date(endDate.getTime() - timeDiff);

        // query construction
        StringBuilder stringQuery = new StringBuilder();

        if (maxNumber != -1) {
            stringQuery.append("DELETE FROM ").append(EVENT_ABSTRACT_SCHEMA_NAME).append(" e").append(
                " WHERE (e.id NOT BETWEEN :min AND :max)");
        } else {
            stringQuery.append("DELETE FROM ").append(EVENT_ABSTRACT_SCHEMA_NAME).append(" e").append(
                " WHERE (e.timestamp NOT BETWEEN :start AND :end)");
        }

        if (!bean.isExecuteOnAllServers()) {
            stringQuery.append(" AND (e.domain=:domain AND e.server=:server)");
        }

        // Create query from stringQuery.
        Query dbQuery = entityManager.createQuery(stringQuery.toString());
        logger.debug(" - Query to db : " + dbQuery.toString());
        logger.debug(" - Query to min : " + minPrimaryKeyId);
        logger.debug(" - Query to max : " + maxPrimaryKeyId);
        logger.debug(" - Query to start : " + startDate);
        logger.debug(" - Query to end : " + endDate);

        if (bean.isExecuteOnAllServers()) {
            if (maxNumber != -1) {
                dbQuery.setParameter("min", minPrimaryKeyId).setParameter("max", maxPrimaryKeyId);
            } else {
                dbQuery.setParameter("start", startDate).setParameter("end", endDate);
            }
        } else {
            if (maxNumber != -1) {
                dbQuery.setParameter("min", minPrimaryKeyId).setParameter("max", maxPrimaryKeyId).setParameter("domain",
                    bean.getDomainName()).setParameter("server", bean.getServerName());
            } else {
                dbQuery.setParameter("start", startDate).setParameter("end", endDate).setParameter("domain",
                    bean.getDomainName()).setParameter("server", bean.getServerName());
            }
        }

        int result = dbQuery.executeUpdate();
        bean.setRemovedEvents(result);

        // Set the last execution date
        bean.setLastExecutionDate(Calendar.getInstance().getTime());

        sendPurgeTaskOnTopic(bean);

        logger.info("[DataManagement] Purge Task cleaned up " + result + " events");

    }

    private void sendPurgeTaskOnTopic(final JasmineEventPurgeTaskEB bean) {
        try {
            Connection connection = factory.createConnection();
            Session session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
            Message message = session.createObjectMessage(beanToEvent(bean));
            MessageProducer producer = session.createProducer(topic);
            producer.send(message);

            // Close JMS objects
            session.close();
            connection.close();
        } catch (Exception e) {
            logger.error(e.getMessage());
        }
    }

}