/**
 * JASMINe
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eventswitch.beans.impl;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.ow2.jasmine.event.beans.JasmineEventEB;
import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventSLBRemote;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implementation of the JasmineEvent stateless bean.
 */
@Stateless(mappedName = "db-ejb/event")
@Remote(JasmineEventSLBRemote.class)
public class JasmineEventSLBImpl implements JasmineEventSLBRemote {
    /**
     * Jasmine event entity bean implementation class name.
     * The name of this class is used in the JPQL query that get events.
     * Allow easy refactoring.
     */
    static final String EVENT_ABSTRACT_SCHEMA_NAME = JasmineEventEBImpl.class.getSimpleName();

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * Entity manager used by this session bean.
     */
    @PersistenceContext(unitName = "persistence-unit/event")
    private EntityManager entityManager = null;

    /**
     * Implementation of inherited method: creates the JasmineEvent entity bean
     * corresponding to given datum and saves an event on the database.
     *
     * @see JasmineEventSLBRemote#saveEvent(JasmineEventEB)
     *
     * @param e  Event to save.
     */
    public void saveEvent(final JasmineEventEB e) {
        JasmineEventEBImpl bean = new JasmineEventEBImpl();
        bean.setDomain(e.getDomain());
        bean.setServer(e.getServer());
        bean.setSource(e.getSource());
        bean.setProbe(e.getProbe());
        bean.setValue(e.getValue().toString());
        bean.setTimestamp(e.getTimestamp());
        bean.setSname(e.getSname());
        entityManager.persist(bean);
    }

    /**
     * Gets the events corresponding to a given filter.
     * This method is used by the "replay from database" monitoring facility.
     *
     * @param servers   Name of the probed servers.
     * @param probes    Probes
     * @param from      Probe time bigger than.
     * @param to        Probe time smaller than.
     * @param orderBy   Order using column.
     * @param index     Set the index (position) of the first result to retrieve (start at 0).
     * @param limit     Maximal number of results to get.
     * @return  Events corresponding to the given search, null if none found.
     */
    public JasmineEventEB[] getEvents(
        final Collection<String> servers,
        final Collection<String> probes,
        final Date from, final Date to,
        final String orderBy,
        final int index,
        final int limit) {

        Map<String, Map<String, Object>> groupsCriteria = new HashMap<String, Map<String, Object>>();

        // server list
        if (servers != null) {
            Map<String, Object> map = new HashMap<String, Object>();
            int i = 0;
            for (String server : servers) {
                StringBuilder nameCriterion = new StringBuilder("server").append(i++);
                map.put(nameCriterion.toString(), server);
                logger.debug("server:" + server);
            }
            groupsCriteria.put("server", map);
        }

        // attribute list
        if (probes != null) {
            Map<String, Object> map = new HashMap<String, Object>();
            int i = 0;
            for (String probe : probes) {
                StringBuilder nameCriterion = new StringBuilder("probe").append(i++);
                map.put(nameCriterion.toString(), probe);
                logger.debug("probe:" + probe);
            }
            groupsCriteria.put("probe", map);
        }

        // dates from/to
        if (from != null) {
            Map<String, Object> map = new HashMap<String, Object>();
            map.put("from", from);
            groupsCriteria.put("from", map);
        }
        if (to != null) {
            Map<String, Object> map = new HashMap<String, Object>();
            map.put("to", to);
            groupsCriteria.put("to", map);
        }

        // String query construction.
        StringBuilder stringQuery = new StringBuilder();
        stringQuery.append("SELECT e FROM ").append(EVENT_ABSTRACT_SCHEMA_NAME).append(" e");

        if (groupsCriteria.size() != 0) {
            stringQuery.append(" WHERE");

            boolean isFirstGroup = true;
            for (String groupName : groupsCriteria.keySet()) {
                if (isFirstGroup) {
                    isFirstGroup = false;
                } else {
                    stringQuery.append(" AND");
                }
                stringQuery.append(" (");

                boolean isFirstCriterion = true;
                for (String criterionName : groupsCriteria.get(groupName).keySet()) {
                    if (isFirstCriterion) {
                        isFirstCriterion = false;
                    } else {
                        stringQuery.append(" OR");
                    }

                    if (groupName.equals("from")) {
                        stringQuery.append(" e.timestamp >=");
                    } else if (groupName.equals("to")) {
                        stringQuery.append(" e.timestamp <=");
                    } else {
                        stringQuery.append(" UPPER(").append(" e.").append(groupName).append(") LIKE");
                    }
                    stringQuery.append(" :").append(criterionName);
                }
                stringQuery.append(" )");
            }
        }

        // If the orderBy clause is specified.
        if (orderBy != null) {
            stringQuery.append(" ORDER BY e.").append(orderBy).append(" ASC");
        }

        logger.debug(stringQuery);

        // Create query from stringQuery.
        Query dbQuery = entityManager.createQuery(stringQuery.toString());

        if (index > 0) {
            dbQuery.setFirstResult(index);
        }

        // If limit clause is specified, set the query maximum number of results.
        if (limit > 0) {
            dbQuery.setMaxResults(limit);
        }


        // Replace the keys by the real values.
        for (Map<String, Object> map : groupsCriteria.values()) {
            for (Entry<String, Object> entry :  map.entrySet()) {
                if (entry.getValue() instanceof String) {
                    String val = (String) entry.getValue();
                    dbQuery.setParameter(entry.getKey(), "%" + val.toUpperCase() + "%");

                } else {
                    dbQuery.setParameter(entry.getKey(), entry.getValue());
                }

            }
        }

        List<JasmineEventEBImpl> queryResult = dbQuery.getResultList();
        JasmineEventEB[] resultTable = null;

        // If the query has returned results, put them in the result table.
        if (queryResult.size() != 0) {
            resultTable = new JasmineEventEB[queryResult.size()];
            int i = 0;
            for (JasmineEventEBImpl eventEB : queryResult) {
                resultTable[i++] = new JasmineEventEB(eventEB.getDomain(), eventEB.getServer(), eventEB.getSource(),
                                                 eventEB.getProbe(), eventEB.getValue(), eventEB.getTimestamp(),
                                                 eventEB.getSname());
            }
        }

        return resultTable;
    }

    /**
     * Implementation of inherited method: looks for the beans searched for,
     * converts the search into an array of JasmineEventEB objects and returns.
     *
     * @see JasmineEventSLBRemote#getEvents(String, String, String, String, String, Date, Date, String, int)
     *
     * @param domain          Domain of the probed server.
     * @param server          Name of the probed servers.
     * @param source          Event source (for example, JProbe or JMX).
     * @param probe           Probes type.
     * @param value           Probed value.
     * @param startTimestamp  Probe time bigger than.
     * @param endTimestamp    Probe time smaller than.
     * @param orderBy         Order using column.
     * @param limit           Maximal number of results to get.
     *
     * @return  Events corresponding to the given search, null if none found.
     */
    public JasmineEventEB[] getEvents(final String domain, final String server, final String source, final String probe,
                                      final String value, final Date startTimestamp, final Date endTimestamp,
                                      final String orderBy, final int limit) {
        return getEvents(domain, server, source, probe, value, startTimestamp, endTimestamp, orderBy, 0, limit);
    }

    /**
     * Implementation of inherited method: looks for the beans searched for,
     * converts the search into an array of JasmineEventEB objects and returns.
     *
     * @see JasmineEventSLBRemote#getEvents(String, String, String, String, String, Date, Date, String, int, int)
     *
     * @param domain            Domain of the probed server.
     * @param server            Name of the probed servers.
     * @param source            Event source (for example, JProbe or JMX).
     * @param probe             Probes type.
     * @param value             Probed value.
     * @param startTimestamp    Probe time bigger than.
     * @param endTimestamp      Probe time smaller than.
     * @param orderBy           Order using column.
     * @param indexFirstResult  Set the index (position) of the first result to retrieve (start at 0).
     * @param limit             Maximal number of results to get.
     *
     * @return  Events corresponding to the given search, null if none found.
     */
    public JasmineEventEB[] getEvents(final String domain, final String server, final String source, final String probe,
            final String value, final Date startTimestamp, final Date endTimestamp, final String orderBy,
            final int indexFirstResult, final int limit) {
        Map<String, Map<String, Object>> groupsCriteria = new HashMap<String, Map<String, Object>>();
        //Definition of the separator used for the criteria composed of several values.
        //String separator = "\\|";
        String separator = "separator";

        //If the criterion domain is specified.
        if (domain != null && !domain.equals("")) {
            //Split the domain string to get all the domains.
            String[] domains = domain.split(separator);
            //Create group criteria "domain".
            Map<String, Object> domainCriteria = new HashMap<String, Object>();
            for (int i = 0; i < domains.length; i++) {
                StringBuilder nameCriterion = new StringBuilder("domain").append(i);
                //Add a criterion to the "domain" group.
                domainCriteria.put(nameCriterion.toString(), domains[i]);
            }
            //Add "domain" group to groupsCriteria.
            groupsCriteria.put("domain", domainCriteria);
        }
        //If the criterion server is specified.
        if (server != null && !server.equals("")) {
            String[] servers = server.split(separator);
            Map<String, Object> map = new HashMap<String, Object>();
            for (int i = 0; i < servers.length; i++) {
                StringBuilder nameCriterion = new StringBuilder("server").append(i);
                map.put(nameCriterion.toString(), servers[i]);
            }
            groupsCriteria.put("server", map);
        }
        //If the criterion source is specified.
        if (source != null && !source.equals("")) {
            String[] sources = source.split(separator);
            Map<String, Object> map = new HashMap<String, Object>();
            for (int i = 0; i < sources.length; i++) {
                StringBuilder nameCriterion = new StringBuilder("source").append(i);
                map.put(nameCriterion.toString(), sources[i]);
            }
            groupsCriteria.put("source", map);
        }
        //If the criterion probe is specified.
        if (probe != null && !probe.equals("")) {
            String[] probes = probe.split(separator);
            Map<String, Object> map = new HashMap<String, Object>();
            for (int i = 0; i < probes.length; i++) {
                StringBuilder nameCriterion = new StringBuilder("probe").append(i);
                map.put(nameCriterion.toString(), probes[i]);
                logger.debug("probe"+i+"="+probes[i]);
            }
            groupsCriteria.put("probe", map);
        }
        //If the criterion value is specified.
        if (value != null) {
            String[] values = value.split(separator);
            Map<String, Object> map = new HashMap<String, Object>();
            for (int i = 0; i < values.length; i++) {
                StringBuilder nameCriterion = new StringBuilder("value").append(i);
                map.put(nameCriterion.toString(), values[i]);
            }
            groupsCriteria.put("value", map);
        }
        //If the criterion startTimestamp is specified.
        if (startTimestamp != null) {
            Map<String, Object> map = new HashMap<String, Object>();
            map.put("startTimestamp", startTimestamp);

            groupsCriteria.put("startTimestamp", map);
        }
        //If the criterion endTimestamp is specified.
        if (endTimestamp != null) {
            Map<String, Object> map = new HashMap<String, Object>();
            map.put("endTimestamp", endTimestamp);

            groupsCriteria.put("endTimestamp", map);
        }

        //String query construction.
        StringBuilder stringQuery = new StringBuilder();
        stringQuery.append("SELECT e FROM ").append(EVENT_ABSTRACT_SCHEMA_NAME).append(" e");

        //If no criteria are specified.
        if (groupsCriteria.size() != 0) {
            stringQuery.append(" WHERE");

            boolean isFirstGroup = true;
            for (String groupName : groupsCriteria.keySet()) {
                //For each group criteria.
                if (isFirstGroup) {
                    isFirstGroup = false;
                } else {
                    stringQuery.append(" AND");
                }
                stringQuery.append(" (");

                boolean isFirstCriterion = true;
                for (String criterionName : groupsCriteria.get(groupName).keySet()) {
                    //For each criterion.
                    if (isFirstCriterion) {
                        isFirstCriterion = false;
                    } else {
                        stringQuery.append(" OR");
                    }

                    if (groupName.equals("startTimestamp")) {
                        stringQuery.append(" e.timestamp >=");
                    } else if (groupName.equals("endTimestamp")) {
                        stringQuery.append(" e.timestamp <=");
                    } else {
                        stringQuery.append(" UPPER(").append(" e.").append(groupName).append(") LIKE");
                    }
                    stringQuery.append(" :").append(criterionName);
                }
                stringQuery.append(" )");
            }
        }

        //If the orderBy clause is specified.
        if (orderBy != null) {
            stringQuery.append(" ORDER BY e.").append(orderBy).append(" ASC");
        }

        logger.debug(stringQuery);

        //Create query from stringQuery.
        Query dbQuery = entityManager.createQuery(stringQuery.toString());

        //If indexFirstResult is specified, set the index of the first result.
        if (indexFirstResult > 0) {
            dbQuery.setFirstResult(indexFirstResult);
        }

        //If limit clause is specified, set the query maximum number of results.
        if (limit > 0) {
            dbQuery.setMaxResults(limit);
        }


        //Replace the keys by the real values.
        for (Map<String, Object> map : groupsCriteria.values()) {
            for (Entry<String, Object> entry :  map.entrySet()) {
                if (entry.getValue() instanceof String) {
                    String val = (String) entry.getValue();
                    dbQuery.setParameter(entry.getKey(), "%" + val.toUpperCase() + "%");

                } else {
                    dbQuery.setParameter(entry.getKey(), entry.getValue());
                }

            }
        }
        logger.debug(dbQuery.toString());

        List<JasmineEventEBImpl> queryResult = dbQuery.getResultList();
        JasmineEventEB[] resultTable = null;

        //If the query has returned results, put them in the result table.
        if (queryResult.size() != 0) {
            resultTable = new JasmineEventEB[queryResult.size()];
            int i = 0;
            for (JasmineEventEBImpl eventEB : queryResult) {
                resultTable[i++] = new JasmineEventEB(eventEB.getDomain(), eventEB.getServer(), eventEB.getSource(),
                                                 eventEB.getProbe(), eventEB.getValue(), eventEB.getTimestamp(),
                                                 eventEB.getSname());
            }
        }

        return resultTable;
    }
}
