/**
 * JASMINe
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eventswitch.extractor;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Set;
import java.util.TreeSet;

import javax.naming.NamingException;

import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventEB;
import org.ow2.jasmine.monitoring.eventswitch.beans.JasmineEventSLBRemote;
import org.ow2.jasmine.monitoring.eventswitch.connectors.EJB3Connector;

/**
 * The JASMINe Event Data Extractor is an executable utility class. It will
 * connect to a given JASMINe Event stateless bean, fetch information and print
 * it back on stdio using the CSV similar to mbeancmd's.
 */
public class JasmineEventDataExtractor extends EJB3Connector<JasmineEventSLBRemote> {
    /**
     * Launches {@link JasmineEventDataExtractor#JasmineEventDataExtractor(String[])}.
     * That will connect to the given JASMINe Event stateless bean, requests
     * data that satisfies the given constraints an prints it back on stdio.
     *
     * @param args  Accepts three types of variables:
     *                  - Any number of string constants defined in Context.
     *                    Example: to set the Context.INITIAL_CONTEXT_FACTORY
     *                    to factory, use the option:
     *                          -Context.INITIAL_CONTEXT_FACTORY factory
     *                  - The "Wrapper.LOOKUP_BEAN" option, the value of which
     *                    is looked up on the server and set as remote bean.
     *                  - The "timeformat" option, which defines the time
     *                    format when defining start and end times. If none
     *                    specified, it uses the "long" format
     *                    (seconds since 01/01/1970).
     *                  - Filtering options. These are:
     *                          - domain: server domain
     *                          - server: server name
     *                          - source: source (for example, MBeanCmd)
     *                          - probe: probe name
     *                          - value: probe value
     *                          - startTimestamp: time to start at
     *                          - endTimestamp: time to end at
     *                    If you don't need a certain filtering, don't set it.
     *
     * @throws NamingException  If there's a problem resolving the bean.
     * @throws ParseException   If there's a problem parsing times.
     * @throws IOException      On configuration file failure.
     */
    public static void main(final String[] args) throws NamingException, ParseException, IOException {
        new JasmineEventDataExtractor(args);
    }

    /**
     * @see JasmineEventDataExtractor#main(String[])
     */
    protected JasmineEventDataExtractor(final String[] args) throws NamingException, ParseException, IOException {
        ArgumentParser p = new ArgumentParser(args);
        setConfiguration(p.getOptions());
        String format = p.getOption("timeformat");
        if (format != null || "long".equalsIgnoreCase(format)) {
            formatter = new SimpleDateFormat(format);
        }
        int limit = 1000;
        try {
            Integer.parseInt(p.getOption("limit"));
        } catch(Exception e) {
            System.err.println("No limit specified: defaulting to 1000.");
        }

        connect();
        extractEvents(getBean().getEvents(p.getOption("domain"), p.getOption("server"), p.getOption("source"),
                p.getOption("probe"), p.getOption("value"), getTime(p.getOption("startTimestamp")),
                getTime(p.getOption("endTimestamp")), "timestamp", limit));
        disconnect();
    }

    /**
     * Extracts and puts on stdio an array of events.
     *
     * @param events  Events to put on stdio.
     */
    protected void extractEvents(final JasmineEventEB[] events) {
        if(events == null) {
            return;
        }

        Set<String> attributes = new TreeSet<String>();
        for(JasmineEventEB event : events) {
            attributes.add(event.getProbe().substring(event.getProbe().lastIndexOf(':') + 1));
        }

        System.out.print("date;time;server;domain;mbean");
        for(String attribute : attributes) {
            System.out.print(";");
            System.out.print(attribute);
        }
        System.out.println();

        SimpleDateFormat df = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss");
        for(JasmineEventEB event : events) {
            System.out.print(df.format(event.getTimestamp()));
            System.out.print(";");
            System.out.print(event.getTimestamp().getTime());
            System.out.print(";");
            System.out.print(event.getServer());
            System.out.print(";");
            System.out.print(event.getDomain());
            System.out.print(";");
            System.out.print(event.getProbe().substring(0, event.getProbe().lastIndexOf(':')));
            String attribute = event.getProbe().substring(event.getProbe().lastIndexOf(':') + 1);
            for(String cAttribute : attributes) {
                System.out.print(";");
                if(cAttribute.equals(attribute)) {
                    System.out.print(event.getValue());
                }
            }
            System.out.println();
        }
    }

    /**
     * Parses a given time using the formatter, defaults to format "long".
     *
     * @param s  Time to parse, can be null (in that case return is null).
     *
     * @return Parsed time.
     *
     * @throws ParseException  If parsing fails.
     */
    protected Date getTime(final String s) throws ParseException {
        if (s == null) {
            return null;
        }

        if (formatter == null) {
            return new Date(Long.parseLong(s));
        } else {
            return formatter.parse(s);
        }
    }

    /**
     * Date format, null means "long".
     */
    private SimpleDateFormat formatter = null;
}
