/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eos.common.view {
import flash.display.BitmapData;

import flash.events.Event;
import flash.events.IEventDispatcher;
import flash.events.MouseEvent;
import flash.net.URLLoader;
import flash.net.URLRequest;
import flash.net.URLRequestMethod;
import flash.net.navigateToURL;
import flash.utils.ByteArray;

import mx.charts.chartClasses.CartesianChart;
import mx.charts.chartClasses.Series;
import mx.collections.ArrayCollection;
import mx.containers.Canvas;
import mx.controls.Alert;
import mx.core.UIComponent;
import mx.events.CollectionEvent;
import mx.events.DividerEvent;
import mx.events.ResizeEvent;
import mx.graphics.codec.PNGEncoder;

/**
 * Defines the parent class of the MXML components.
 */
public class CartesianChartClass extends Canvas {
    /**
     * Array of Series used for the small Chart
     * its dataProvider is chartData
     */
    [Bindable]
    public var chartSeries:Array;

    /**
     * data Provider for the small Chart
     */
    [Bindable]
    public var chartData:ArrayCollection;

    /**
     * index of Serie having the max of values
     */
    public var LONGEST_SERIE:Number = 0;

    /**
     * nb of values for the LONGEST_SERIE
     */
    public var DATA_LENGTH:Number = 0;

    /**
     * true if the zoom has just been used.
     */
    public var sliderMoved:Boolean = false;

    /**
     * Index of left boundary used for displaying the Big Chart
     */
    [Bindable]
    public var leftBoundary:Number;

    /**
     * Index of right boundary used for displaying the Big Chart
     */
    [Bindable]
    public var rightBoundary:Number;

    public var staticLeftBoundary:Number;
    public var staticRightBoundary:Number;

    public var smallDrag:Boolean = false;
    public var bigDrag:Boolean = false;
    public var mouseXRef:Number;

    public var maxSizeOfWindow:int = 0;

    public var isMontoringFollowed:Boolean = false;
    public var gapBetweenRightBoundaryAndBorder:int = 0;
    public var gapBetweenBoundaries:int = 0;

    [Bindable]
    public var bigChart:CartesianChart;

    [Bindable]
    public var smallChart:CartesianChart;

    [Bindable]
    public var leftBox:Canvas;

    [Bindable]
    public var rightBox:Canvas;


    /**
     * data provider for the big chart
     * init with same data than chartData.
     * then, updated to reflect only a slice [leftBoundary - rightBoundary]
     */
    [Bindable]
    public var mainData:ArrayCollection;

    private const UPLOAD_URL:String = "./UploadFileServlet";
    private const EXPORT_URL:String = "./ExportBitmapServlet";

    [Embed(source="../../../../../../../../assets/divider.png")]
    [Bindable]
    public var dividerClass:Class;
    [Embed(source="../../../../../../../../assets/blank.png")]
    [Bindable]
    public var blankDividerClass:Class;

    public function CartesianChartClass():void {
        addEventListener(ResizeEvent.RESIZE, resizeHandler);
    }

    protected function buildSeries():Series {
        Alert.show("Abstract function: should not be called");
        return null;
    }

    /**
     * Initialise the big chart
     */
    public function chartComplete():void {

        this.chartData = new ArrayCollection();
        this.mainData = new ArrayCollection();

        var bigChartSeries:ArrayCollection = new ArrayCollection();

        for each (var s:Series in chartSeries) {
            var provider:ArrayCollection = s.dataProvider as ArrayCollection;
            provider.addEventListener(CollectionEvent.COLLECTION_CHANGE, renderHandler);
            chartData.addItem(provider);
            var index:int = chartData.getItemIndex(provider);
            // Build a new ArrayCollection on the same Array
            var mainprovider:ArrayCollection = new ArrayCollection(chartData.getItemAt(index).source);
            mainData.addItem(mainprovider);
            var series:Series = buildSeries();
            //series.filterData = false;
            series.displayName = s.displayName;
            index = mainData.getItemIndex(mainprovider);
            series.dataProvider = mainData.getItemAt(index) as ArrayCollection;
            bigChartSeries.addItem(series);
        }

        bigChart.seriesFilters = [];
        bigChart.series = bigChartSeries.toArray();
        this.DATA_LENGTH = chartData.getItemAt(0).length;

        leftBoundary = 0;
        rightBoundary = 0;

        // Initialise the small chart
        smallChart.seriesFilters = [];
        smallChart.series = chartSeries;
        smallChart.width = bigChart.width * 0.9;

        // start with the zoom option unset.
        zoom(false);
    }

    /**
     * resize the two charts
     */
    public function resizeHandler(event:Event):void {
        var chartwith:int = this.parent.width - 34;
        bigChart.width = chartwith;
        smallChart.width = chartwith * 0.9;
    }

    public function zoom(value:Boolean):void {
        if (value) {
            // zoom is on
            smallChart.enabled = true;
            smallChart.height = 80;
        }
        else {
            // zoom is off
            sliderMoved = false;
            smallChart.enabled = false;
            smallChart.height = 0;
        }
    }

    /**
     * Sets the graphs.
     * @param series ArrayCollection containing the Chart series to display
     */
    public function setCharts(series:ArrayCollection):void {
        this.chartSeries = series.toArray();
    }

    /**
     * create a window which will move to show you new data monitored
     * the size of this window will be constant during the execution of this method
     */
    public function followMonitoring(value:Boolean):void {
        if (value) {
            isMontoringFollowed = true;
            gapBetweenRightBoundaryAndBorder = chartData.getItemAt(0).length - rightBoundary;
            gapBetweenBoundaries = rightBoundary - leftBoundary;
        }
        else {
            isMontoringFollowed = false;
        }
    }

    /**
     * New data: redraw the graph
     */
    public function renderHandler(event:Event):void {

        // Compute the longest serie
        this.DATA_LENGTH = 0;
        this.LONGEST_SERIE = 0;
        for (var ser:int = 0; ser < chartData.length; ser++) {
            var len:int = chartData.getItemAt(ser).length;
            if (len > this.DATA_LENGTH) {
                this.DATA_LENGTH = len;
                this.LONGEST_SERIE = ser;
            }
        }

        if (this.sliderMoved == false) {
            rightBoundary = chartData.getItemAt(this.LONGEST_SERIE).length;
        }
        if (this.isMontoringFollowed == true) {
            // ??? TODO
            rightBoundary = chartData.getItemAt(this.LONGEST_SERIE).length - this.gapBetweenRightBoundaryAndBorder;
            leftBoundary = rightBoundary - this.gapBetweenBoundaries;
            this.sliderMoved = true;
        }
        updateMainData();
    }

    public function setMouseDown(big:Boolean):void {
        mouseXRef = mouseX;
        staticLeftBoundary = leftBoundary;
        staticRightBoundary = rightBoundary;
        if (big)
            bigDrag = true;
        else
            smallDrag = true;
        systemManager.addEventListener(MouseEvent.MOUSE_MOVE, moveChart);
        systemManager.addEventListener(MouseEvent.MOUSE_UP, stopDragging);
    }

    private function stopDragging(event:MouseEvent):void {
        smallDrag = false;
        bigDrag = false;
        systemManager.removeEventListener(MouseEvent.MOUSE_MOVE, moveChart);
        systemManager.removeEventListener(MouseEvent.MOUSE_UP, stopDragging);
    }

    /**
     * takes the position of the mouse to update the boundaries values
     */
    private function moveChart(event:MouseEvent):void {
        this.sliderMoved = true;
        if (bigDrag) {
            leftBoundary = staticLeftBoundary + (mouseXRef - mouseX) / (bigChart.width / mainData.getItemAt(0).length);
            rightBoundary = staticRightBoundary + (mouseXRef - mouseX) / (bigChart.width / mainData.getItemAt(0).length);
        }
        else if (smallDrag) {
            leftBoundary = staticLeftBoundary - (mouseXRef - mouseX) / (smallChart.width / chartData.getItemAt(0).length);
            rightBoundary = staticRightBoundary - (mouseXRef - mouseX) / (smallChart.width / chartData.getItemAt(0).length);
        }
        updateMainData(); // TODO ???
    }

    /**
     * Export the Graph in png format
     */
    public function export():void {
        var bd:BitmapData = new BitmapData(bigChart.width, bigChart.height);
        bd.draw(bigChart);


        var encoder:PNGEncoder = new PNGEncoder();

        var ba:ByteArray = encoder.encode(bd);

        var request:URLRequest = new URLRequest();
        request.data = ba;
        request.url = this.UPLOAD_URL;
        request.method = URLRequestMethod.POST;

        var urlLoader:URLLoader = new URLLoader();
        this.configureListeners(urlLoader);
        urlLoader.load(request);
    }

    /**
     * update the position of the slider from the time values selected by user
     * @param from
     * @param to
     */
    public function updateTimeLimits(from:Date, to:Date):void {
        this.sliderMoved = false;

        // We work on the longest series ???? TODO
        var longest:ArrayCollection = chartData.getItemAt(this.LONGEST_SERIE) as ArrayCollection;

        // compute leftBoundary
        if (from == null) {
            leftBoundary = 0;
        }
        else {
            var start:int = 0;
            var end:int = this.DATA_LENGTH - 1;
            var curval:int = start;
            var preval:int = curval;
            while (start < end) {
                curval = start + ((end - start) / 2);
                if (curval == preval) {
                    break;
                }
                var cd:int = compareDate(longest.getItemAt(curval).date, from);
                if (cd == 0) {
                    break;
                }
                if (cd < 0) {
                    start = curval;
                }
                else {
                    end = curval;
                }
                preval = curval;
            }
            leftBoundary = curval;
            this.sliderMoved = true;
        }

        // compute rightBoundary
        if (to == null) {
            rightBoundary = this.DATA_LENGTH - 1;
        }
        else {
            start = leftBoundary;
            end = this.DATA_LENGTH - 1;
            curval = end;
            preval = curval;
            while (start < end) {
                curval = start + ((end - start) / 2);
                if (curval == preval) {
                    break;
                }
                var cd:int = compareDate(longest.getItemAt(curval).date, to);
                if (cd == 0) {
                    break;
                }
                if (cd < 0) {
                    start = curval;
                }
                else {
                    end = curval;
                }
                preval = curval;
            }
            rightBoundary = curval;
            this.sliderMoved = true;
        }
        updateMainData();
    }

    private function compareDate(d1:Date, d2:Date):int {
        if (d1.getFullYear() < d2.getFullYear()) {
            return -1;
        }
        if (d1.getFullYear() > d2.getFullYear()) {
            return 1;
        }
        if (d1.getMonth() < d2.getMonth()) {
            return -1;
        }
        if (d1.getMonth() > d2.getMonth()) {
            return 1;
        }
        if (d1.getDate() < d2.getDate()) {
            return -1;
        }
        if (d1.getDate() > d2.getDate()) {
            return 1;
        }
        if (d1.getHours() < d2.getHours()) {
            return -1;
        }
        if (d1.getHours() > d2.getHours()) {
            return 1;
        }
        if (d1.getMinutes() < d2.getMinutes()) {
            return -1;
        }
        if (d1.getMinutes() > d2.getMinutes()) {
            return 1;
        }
        return 0;
    }

    /**
     * update the big chart when user move the divider
     */
    public function updateBoundariesFromDivider(event:DividerEvent):void {
        // Assumes that values are regularly dispatched on the period.
        // It may not be the case ! TODO
        var dlen:int = chartData.getItemAt(this.LONGEST_SERIE).length;
        leftBoundary = (dlen * leftBox.width) / event.target.width;
        rightBoundary = dlen - (dlen * rightBox.width) / event.target.width;
        this.sliderMoved = true;
        updateMainData();
    }

    /**
     * update the data of the big chart when a change is made on the small char
     */
    private function updateMainData():void {
        for (var serie:int = 0; serie < chartData.length; serie++) {
            var arrdata:Array = chartData.getItemAt(serie).source;
            if (this.sliderMoved) {
                // mainData is a slice of arrdata
                mainData.getItemAt(serie).source = arrdata.slice(leftBoundary, rightBoundary);
            }
            else {
                mainData.getItemAt(serie).source = arrdata;
            }
        }
    }

    private function configureListeners(dispatcher:IEventDispatcher):void {
        dispatcher.addEventListener(Event.COMPLETE, completeHandler);
    }

    private function completeHandler(event:Event):void {
        var loader:URLLoader = URLLoader(event.target);
        var name:String = XML(loader.data.toString()).file.@name;
        var req:String = this.EXPORT_URL + "?fileId=" + name;
        var u:URLRequest = new URLRequest(req);
        navigateToURL(u, "_blank");
    }

}
}
