/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Pod.as 7868 2011-03-25 15:35:16Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eos.common.view {
import flash.display.Graphics;
import flash.display.Sprite;
import flash.events.Event;
import flash.events.MouseEvent;

import mx.containers.HBox;
import mx.containers.Panel;
import mx.controls.Alert;
import mx.controls.Button;
import mx.controls.ComboBox;
import mx.controls.LinkButton;
import mx.controls.Menu;
import mx.events.DragEvent;

import org.ow2.jasmine.monitoring.eos.common.events.PodMenuEvent;
import org.ow2.jasmine.monitoring.eos.common.events.PodStateChangeEvent;

// Events dispatched in case of drag and drop:
// DragEvent.DRAG_START and DragEvent.DRAG_COMPLETE
[Event(name="dragStart", type="mx.events.DragEvent")]
[Event(name="dragComplete", type="mx.events.DragEvent")]

// Events dispatched in case of minimize/maximise
// PodStateChangeEvent.MINIMIZE, PodStateChangeEvent.MAXIMIZE, PodStateChangeEvent.RESTORE
[Event(name="minimize", type="org.ow2.jasmine.monitoring.eos.common.events.PodStateChangeEvent")]
[Event(name="maximize", type="org.ow2.jasmine.monitoring.eos.common.events.PodStateChangeEvent")]
[Event(name="restore", type="org.ow2.jasmine.monitoring.eos.common.events.PodStateChangeEvent")]

// Event dispatched for the menu
[Event(name="menu", type="org.ow2.jasmine.monitoring.eos.common.events.PodMenuEvent")]

/**
 * Container which holds PodContentBase subclasses.
 * Pod are managed by the managers.PodLayoutManager
 * This class is highly drawn from the adobe Flex examples:
 * See http://examples.adobe.com/flex3/devnet/dashboard for original code.
 * - code has been simplified
 * - a menu has been added.
 */
public class Pod extends Panel {
    // Accessed from PodManager
     public static const MINIMIZED_HEIGHT:Number = 22;

    // Possible State Values of the Pod window
    public static const WINDOW_STATE_DEFAULT:Number = -1;
    public static const WINDOW_STATE_MINIMIZED:Number = 0;
    public static const WINDOW_STATE_MAXIMIZED:Number = 1;
    public var windowState:Number;

    // Index within the layout.
    public var index:Number;

    // the HBox holding the menus and Button in the title bar.
    private var controlsHolder:HBox;

    // Elements composing the controlsHolder
    private var menuButton:LinkButton;
    private var minimizeButton:Button;
    private var maximizeRestoreButton:Button;

    // Variables used for dragging the pod.
    private var dragStartMouseX:Number;
    private var dragStartMouseY:Number;
    private var dragStartX:Number;
    private var dragStartY:Number;
    private var dragMaxX:Number;
    private var dragMaxY:Number;

    private var _maximize:Boolean;
    private var _maximizeChanged:Boolean;

    /**
     * Constructor: set things up but do not create children or position things
     */
    public function Pod() {
        super();
        doubleClickEnabled = true;
        setStyle("titleStyleName", "podTitle");

        windowState = WINDOW_STATE_DEFAULT;
        horizontalScrollPolicy = "off";
    }

    /**
     * create the children but do not position or size them
     */
    override protected function createChildren():void {
        super.createChildren();

        if (!controlsHolder) {
            controlsHolder = new HBox();
            controlsHolder.setStyle("paddingRight", getStyle("paddingRight"));
            controlsHolder.setStyle("horizontalAlign", "right");
            controlsHolder.setStyle("verticalAlign", "middle");
            controlsHolder.setStyle("horizontalGap", 3);
            rawChildren.addChild(controlsHolder);
        }

        if (!menuButton) {
            menuButton = new LinkButton();
            menuButton.label = "options";
            controlsHolder.addChild(menuButton);
            menuButton.addEventListener(MouseEvent.CLICK, onClickMenu);
        }

        if (!minimizeButton) {
            minimizeButton = new Button();
            minimizeButton.width = 14;
            minimizeButton.height = 14;
            minimizeButton.styleName = "minimizeButton";
            controlsHolder.addChild(minimizeButton);
            minimizeButton.addEventListener(MouseEvent.CLICK, onClickMinimizeButton);
        }

        if (!maximizeRestoreButton) {
            maximizeRestoreButton = new Button();
            maximizeRestoreButton.width = 14;
            maximizeRestoreButton.height = 14;
            maximizeRestoreButton.styleName = "maximizeRestoreButton";
            controlsHolder.addChild(maximizeRestoreButton);
            maximizeRestoreButton.addEventListener(MouseEvent.CLICK, onClickMaximizeRestoreButton);
        }

        // Used for drag and drop
        titleBar.addEventListener(MouseEvent.MOUSE_DOWN, onMouseDownTitleBar);

        // Double click on title bar = maximize/restore Button.
        titleBar.addEventListener(MouseEvent.DOUBLE_CLICK, onClickMaximizeRestoreButton);

        // Used to restore a window being minimized.
        titleBar.addEventListener(MouseEvent.CLICK, onClickTitleBar);

        // This is needed for maximizing a winwow (?)
        addEventListener(MouseEvent.MOUSE_DOWN, onMouseDown);
    }

    /**
     * This is called each time the Pod is displayed (veryu often)
     * @param unscaledWidth
     * @param unscaledHeight
     */
    override protected function updateDisplayList(unscaledWidth:Number, unscaledHeight:Number):void {
        super.updateDisplayList(unscaledWidth, unscaledHeight);

        // Set controlsHolder position
        controlsHolder.y = titleBar.y;
        controlsHolder.width = unscaledWidth;
        controlsHolder.height = titleBar.height;

        titleTextField.width = titleBar.width - getStyle("paddingLeft") - getStyle("paddingRight");
    }

    private function onMouseDown(event:Event):void {
        // Moves the pod to the top of the z-index.
        parent.setChildIndex(this, parent.numChildren - 1);
    }

    /**
     * Minimize BUTTON
     * @param event
     */
    private function onClickMinimizeButton(event:MouseEvent):void {
        dispatchEvent(new PodStateChangeEvent(PodStateChangeEvent.MINIMIZE));
        // Call after the event is dispatched so the old state is still available.
        minimize();
    }

    /**
     * Called also from LayoutManager
     */
    public function minimize():void {
        controlsHolder.visible = false;
        setStyle("borderSides", "left top right");
        windowState = WINDOW_STATE_MINIMIZED;
        height = MINIMIZED_HEIGHT;
    }

    /**
     * Maximize/Restore BUTTON (toggle)
     * @param event
     */
    private function onClickMaximizeRestoreButton(event:MouseEvent = null):void {
        if (windowState == WINDOW_STATE_DEFAULT) {
            // Maximize the window
            dispatchEvent(new PodStateChangeEvent(PodStateChangeEvent.MAXIMIZE));
            // Call after the event is dispatched so the old state is still available.
            maximize();
        }
        else {
            // reset to normal state
            dispatchEvent(new PodStateChangeEvent(PodStateChangeEvent.RESTORE));
            // Set the state after the event is dispatched so the old state is still available.
            windowState = WINDOW_STATE_DEFAULT;
            maximizeRestoreButton.selected = false;
            controlsHolder.visible = true;
        }
    }

    /**
     * Called also from LayoutManager
     */
    public function maximize():void {
        windowState = WINDOW_STATE_MAXIMIZED;
        _maximize = true;
        maximizeRestoreButton.selected = true;
        controlsHolder.visible = true;
    }

    /**
     * Click on Iconified Window = Restore it.
     * @param event
     */
    private function onClickTitleBar(event:MouseEvent):void {
        if (windowState == WINDOW_STATE_MINIMIZED) {
            // Add the bottom border back in case we were minimized.
            setStyle("borderSides", "left top right bottom");
            onClickMaximizeRestoreButton();
        }
    }

    /**
     * Menu LINKBUTTON
     * @param event
     */
    private function onClickMenu(event:MouseEvent):void {
        dispatchEvent(new PodMenuEvent(PodMenuEvent.MENU, id, event));
    }

    /**
     * Start a drag and drop, only if we are in the default state.
     * @param event
     */
    private function onMouseDownTitleBar(event:MouseEvent):void {
        if (windowState == WINDOW_STATE_DEFAULT)
        {
            dispatchEvent(new DragEvent(DragEvent.DRAG_START));
            dragStartX = x;
            dragStartY = y;
            dragStartMouseX = parent.mouseX;
            dragStartMouseY = parent.mouseY;
            dragMaxX = parent.width - width;
            dragMaxY = parent.height - height;

            // Use the stage so we get mouse events outside of the browser window.
            stage.addEventListener(MouseEvent.MOUSE_MOVE, onMouseMove);
            stage.addEventListener(MouseEvent.MOUSE_UP, onMouseUp);
        }
    }

    /**
     * Drag and Drop started.
     * @param e
     */
    private function onMouseMove(e:MouseEvent):void {
        // Make sure we don't go off the screen on the right.
        var targetX:Number = Math.min(dragMaxX, dragStartX + (parent.mouseX - dragStartMouseX));
        // Make sure we don't go off the screen on the left.
        x = Math.max(0, targetX);

        // Make sure we don't go off the screen on the bottom.
        var targetY:Number = Math.min(dragMaxY, dragStartY + (parent.mouseY - dragStartMouseY));
        // Make sure we don't go off the screen on the top.
        y = Math.max(0, targetY);
    }

    /**
     * End of Drag and Drop
     * @param event
     */
    private function onMouseUp(event:MouseEvent):void {
        dispatchEvent(new DragEvent(DragEvent.DRAG_COMPLETE));

        stage.removeEventListener(MouseEvent.MOUSE_MOVE, onMouseMove);
        stage.removeEventListener(MouseEvent.MOUSE_UP, onMouseUp);
    }

}
}
