/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eos.common.view {
import flash.display.BitmapData;

import flash.events.Event;
import flash.events.MouseEvent;
import flash.net.URLLoader;
import flash.net.URLRequest;
import flash.net.URLRequestMethod;
import flash.net.navigateToURL;
import flash.utils.ByteArray;

import mx.charts.chartClasses.CartesianChart;
import mx.charts.chartClasses.Series;
import mx.collections.ArrayCollection;
import mx.containers.Canvas;
import mx.containers.HDividedBox;
import mx.containers.VBox;
import mx.controls.Alert;
import mx.events.CollectionEvent;
import mx.events.DividerEvent;
import mx.events.ResizeEvent;
import mx.graphics.codec.PNGEncoder;

/**
 * Defines the parent class of the MXML components.
 * Interface:
 * - setCharts(ArrayCollection) : initializes the Series to display
 * - chartComplete() : Init graph when creationComplete event is received
 * - export() : create an image of the graph
 * - zoom(Boolean) : set or unset the zoom (small chart + big chart)
 * - setTimeLimits(Number, Number) : set time limits for big chart -> followData on.
 * - resetTimeLimits() : remove time limits for big chart -> followData off. (default)
 */
public class CartesianChartClass extends Canvas {

    /**
     * Array of Series used for the small Chart
     * its dataProvider is chartData
     */
    [Bindable]
    protected var chartSeries:Array;

    /**
     * data Provider for the small Chart
     * each element represents a Series
     */
    [Bindable]
    protected var chartData:ArrayCollection;

    /**
     * data provider for the big chart
     * each element represents a Series
     * init with same data than chartData.
     * then, updated to reflect only a slice of it (from - to)
     */
    [Bindable]
    protected var mainData:ArrayCollection;

    /**
     * If true, recompute boundaries to follow data on the graph.
     */
    protected var followData:Boolean = false;


    // timestamp of first data currently displayed
    protected var zoomFrom:Number;

    // timestamp of last data currently displayed
    protected var zoomTo:Number;

    protected var zoomOn:Boolean = false;

    // timestamp of first data read
    protected var dateFirst:Number = new Date().getTime() + 100000;

    // timestamp of last data we can read
    protected var dateLast:Number = 1;

    /**
     * Used to draw the small chart
     */
    [Embed(source="../../../../../../../../assets/divider.png")]
    [Bindable]
    protected var dividerClass:Class;
    [Embed(source="../../../../../../../../assets/blank.png")]
    [Bindable]
    protected var blankDividerClass:Class;

    // ----------------------------------------------------------------------------
    // Variables defined in the mxml part. (AreaChartView or LineChartView)
    // ----------------------------------------------------------------------------

    /**
     * The small Chart. (LineChart or AreaChart)
     */
    [Bindable]
    public var smallChart:CartesianChart;

    /**
     * The big Chart. (LineChart or AreaChart)
     */
    [Bindable]
    public var bigChart:CartesianChart;

    [Bindable]
    public var overlayCanvas:HDividedBox;

    /**
     * leftBox in the small chart (zoom)
     */
    [Bindable]
    public var leftBox:Canvas;

    /**
     * rightBox in the small chart (zoom)
     */
    [Bindable]
    public var rightBox:Canvas;

    [Bindable]
    public var exportzone:VBox;

    // -------------------------------------------------------------------
    // Initialization
    // -------------------------------------------------------------------

    /**
     * Constructor
     */
    public function CartesianChartClass():void {
        addEventListener(ResizeEvent.RESIZE, resizeHandler);
    }

    /**
     * Initialise the charts : called at creationComplete.
     */
    public function chartComplete():void {

        // Initialise the big Chart
        this.chartData = new ArrayCollection();
        this.mainData = new ArrayCollection();
        var bigChartSeries:ArrayCollection = new ArrayCollection();
        for each (var s:Series in chartSeries) {
            // init charData and set a method to handle data changes
            var provider:ArrayCollection = s.dataProvider as ArrayCollection;
            provider.addEventListener(CollectionEvent.COLLECTION_CHANGE, changeHandler);
            chartData.addItem(provider);

            // init mainData: Build a new ArrayCollection on the same Array
            var mainprovider:ArrayCollection = new ArrayCollection(provider.source);
            mainData.addItem(mainprovider);

            // Build a Series for the big Chart.
            var series:Series = buildSeries();
            series.displayName = s.displayName;
            series.dataProvider = mainprovider;
            bigChartSeries.addItem(series);

            // Do not use filterData on any chart to improve performance
            //s.filterData = false;
            //series.filterData = false;
        }
        bigChart.series = bigChartSeries.toArray();

        // Initialise the small chart
        zoom(false);
        smallChart.series = chartSeries;
        smallChart.width = bigChart.width * 0.9;
    }

    /**
     * Factory for Series. Abstract method.
     * @return
     */
    protected function buildSeries():Series {
        Alert.show("Abstract function: should not be called");
        return null;
    }

    // -------------------------------------------------------------------
    // Public Interface
    // -------------------------------------------------------------------

    /**
     * Sets the graphs.
     * @param series ArrayCollection containing the Chart series to display
     */
    public function setCharts(series:ArrayCollection):void {
        this.chartSeries = series.toArray();
    }

    /**
     * resize the two charts
     */
    public function resizeHandler(event:Event):void {
        var chartwith:int = this.parent.width - 34;
        bigChart.width = chartwith;
        smallChart.width = chartwith * 0.9;
    }

    /**
     * Set or unset the zoom (i.e. the small chart)
     * @param on true to set the zoom, false to unset it.
     */
    public function zoom(on:Boolean):void {
        zoomOn = on;
        if (on) {
            // zoom is on
            overlayCanvas.visible = true;
            smallChart.visible = true;
            smallChart.enabled = true;
            smallChart.height = 80;
            leftBox.width = leftBoxWith();
            rightBox.width = rightBoxWith();
        }
        else {
            // zoom is off
            zoomFrom = dateFirst;
            zoomTo = dateLast;
            leftBox.width = 0;
            rightBox.width = 0;
            overlayCanvas.visible = false;
            smallChart.enabled = false;
            smallChart.height = 0;
            smallChart.visible = false;
            updateMainData();
        }
    }

    public function leftBoxWith():int {
        if (dateLast - dateFirst <= 0) {
            return 0;
        }
        var retval:int = overlayCanvas.width  * (zoomFrom - dateFirst) / (dateLast - dateFirst);
        return retval;
    }

    public function rightBoxWith():int {
        if (dateLast - dateFirst <= 0) {
            return 0;
        }
        var retval:int = overlayCanvas.width  * (dateLast - zoomTo) / (dateLast - dateFirst);
        return retval;
    }

    /**
     * update the big chart when user move the divider
     */
    protected function updateBoundariesFromDivider(event:DividerEvent):void {
        // compute new zoom limits from leftSize and rightSize
        zoomFrom = dateFirst + leftBox.width * (dateLast - dateFirst) / overlayCanvas.width;
        zoomTo = dateLast - rightBox.width * (dateLast - dateFirst) / overlayCanvas.width;
        updateMainData();
    }

    /**
     * Reset time limits to all data: The followData mode is off.
     */
    public function resetTimeLimits():void {
        if (dateFirst < dateLast) {
            updateTimeLimits(dateFirst, dateLast);
        }
        followData = false;
    }

    /**
     * Set time limits: The followData mode is on.
     * We must redraw the graphs.
     */
    public function setTimeLimits(from:Number, to:Number):void {
        updateTimeLimits(from, to);
        followData = true;
        if (zoomOn) {
            // recompute zoom limits on small chart
            leftBox.width = leftBoxWith();
            rightBox.width = rightBoxWith();
        }
    }

    /**
     * Dataprovider has changed.
     * @param event
     */
    public function changeHandler(event:Event):void {
        if (! followData) {
            updateMainData();
        }
    }

    /**
     * update the position of the slider from the time values selected by user
     * @param from
     * @param to
     */
    private function updateTimeLimits(from:Number, to:Number):void {
        this.zoomFrom = from;
        this.zoomTo = to;
        updateMainData();
    }

    /**
     * update the data of the big chart when a change is made on the small char
     */
    private function updateMainData():void {
        // Assign data for each Serie in the Graph
        for (var serie:int = 0; serie < chartData.length; serie++) {
            var arrdata:Array = new Array();
            for each (var data:Object in chartData.getItemAt(serie).source) {
                // TODO use timestamp
                var timestamp:Number = data.date.getTime();
                // update dateLast and dateFirst
                if (timestamp > dateLast) {
                     dateLast = timestamp;
                    if (!followData) {
                        zoomTo = dateLast;
                    }
                }
                if (timestamp < dateFirst) {
                    dateFirst = timestamp;
                    if (!followData) {
                        zoomFrom = dateFirst;
                    }
                }
                // Check if data must go in the big chart
                if (timestamp < zoomFrom || timestamp > zoomTo) {
                    continue;
                }
                arrdata.push(data);
            }
            mainData.getItemAt(serie).source = arrdata;
        }
    }

    // ------------------------------------------------------------------------
    // Drag and Drop
    // ------------------------------------------------------------------------

    public var bigDrag:Boolean = false;

    public var mouseXRef:Number;

    private var leftRef:int;
    private var rightRef:int;

    /**
     * A drag and Drop has been initialized.
     * @param big true if on the big chart, false if in the small chart.
     */
    public function setMouseDown(big:Boolean):void {
        mouseXRef = mouseX;
        leftRef = leftBox.width;
        rightRef = rightBox.width;
        bigDrag = big;
        systemManager.addEventListener(MouseEvent.MOUSE_MOVE, moveChart);
        systemManager.addEventListener(MouseEvent.MOUSE_UP, stopDragging);
    }

    private function stopDragging(event:MouseEvent):void {
        systemManager.removeEventListener(MouseEvent.MOUSE_MOVE, moveChart);
        systemManager.removeEventListener(MouseEvent.MOUSE_UP, stopDragging);
    }

    private function moveChart(event:MouseEvent):void {
        var move:int;
        if (bigDrag) {
           // drag on big chart
            move =  (mouseXRef - mouseX) * (zoomTo - zoomFrom) / (dateLast - dateFirst);
        } else {
            // drag on small chart
            move =  mouseX - mouseXRef;
        }
        leftBox.width = leftRef + move;
        rightBox.width = rightRef - move;

        // check if we go outside the picture
        if (leftBox.width < 0) {
            rightBox.width = rightBox.width + leftBox.width;
            leftBox.width = 0;
        }
        else if (rightBox.width < 0) {
            leftBox.width = leftBox.width + rightBox.width;
            rightBox.width = 0;
        }

        updateBoundariesFromDivider(null);
    }

    // ------------------------------------------------------------------------
    // Export Graph
    // ------------------------------------------------------------------------

    /**
     * Export the Graph in png format
     */
    public function export():void {
        var bd:BitmapData = new BitmapData(exportzone.width, exportzone.height);
        bd.draw(exportzone);

        var encoder:PNGEncoder = new PNGEncoder();
        var ba:ByteArray = encoder.encode(bd);

        var request:URLRequest = new URLRequest();
        request.data = ba;
        request.url = "./UploadFileServlet";
        request.method = URLRequestMethod.POST;

        var urlLoader:URLLoader = new URLLoader();
        urlLoader.addEventListener(Event.COMPLETE, completeHandler);
        urlLoader.load(request);
    }

    private function completeHandler(event:Event):void {
        var loader:URLLoader = URLLoader(event.target);
        var name:String = XML(loader.data.toString()).file.@name;
        var req:String = "./ExportBitmapServlet?fileId=" + name;
        var u:URLRequest = new URLRequest(req);
        navigateToURL(u, "_blank");
    }

}
}
