/**
 * JASMINe
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JMSConnection.java 5685 2009-12-04 17:02:37Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eos.common.jms;

import java.io.Serializable;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.Context;
import javax.naming.InitialContext;

import org.ow2.jasmine.monitoring.eos.common.EventManager;

/**
 * @author poutchjp
 */
public class JMSConnection {

    public static final String TOPIC_FACTORY = "JTCF";

    private EventManager manager = null;
    protected Topic topic = null;
    protected TopicConnection connection = null;
    protected TopicSession session = null;
    protected TopicSubscriber subscriber = null;

    public JMSConnection(final EventManager evManager) {
        this.manager = evManager;
    }

    public void connect(final String topicName) throws Exception {
        this.connect(topicName, new JMSMessageListener(this.manager));
    }

    public void connect(final String topicName, final MessageListener listener) throws Exception {
        Context context = new InitialContext();

        Object o = context.lookup(TOPIC_FACTORY);
        TopicConnectionFactory tcf = (TopicConnectionFactory) o;
        this.topic = (Topic) context.lookup(topicName);

        this.connection = tcf.createTopicConnection();
        this.session = this.connection.createTopicSession(false, Session.CLIENT_ACKNOWLEDGE);

        this.subscriber = this.session.createSubscriber(this.topic);
        this.subscriber.setMessageListener(listener);

        this.connection.start();
    }

    public void sendObjectMessage(final Serializable object) throws Exception {
        if (this.connection == null) {
            throw new JMSException("Not connected!");
        }

        // Create a new session in order to avoid
        // IllegalStateException: Illegal control thread
        TopicSession session = this.connection.createTopicSession(false, Session.CLIENT_ACKNOWLEDGE);
        MessageProducer producer = session.createProducer(topic);
        try {
            Message m = session.createObjectMessage(object);
            producer.send(m);
        } finally {
            producer.close();
            session.close();
            producer = null;
            session = null;
            System.gc();
        }
    }

    public void disconnect() throws Exception {
        if (this.connection != null) {
            this.subscriber.close();
            this.session.close();
            this.connection.close();

            this.connection = null;
            this.session = null;
            this.topic = null;
            this.subscriber = null;
            System.gc();
        }
    }
}
