/**
 * JASMINe
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JMSConnection.java 5966 2010-01-19 09:42:37Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eos.common.jms;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicSubscriber;
import javax.naming.Context;
import javax.naming.InitialContext;

import org.ow2.jasmine.monitoring.eos.common.EventManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * @author poutchjp
 */
public class JMSConnection implements Serializable {

    public static final String CONNECTION_FACTORY = "JCF";
                          
    private Log logger = LogFactory.getLog(this.getClass());
    
    protected Topic topic = null;
    protected Connection connection = null;
    protected Session session = null;
    protected MessageConsumer subscriber = null;

    private class ConnectionInformation implements Serializable {
        private EventManager manager = null;
        private String topicName = null;
        private MessageListener listener = null;
    }

    private ConnectionInformation information = new ConnectionInformation();

    public JMSConnection(final EventManager evManager) {
        this.information.manager = evManager;
    }

    public void connect(final String topicName) throws Exception {
        this.connect(topicName, new JMSMessageListener(this.information.manager));
    }

    public <SML extends MessageListener, Serializable> void connect(final String topicName, final SML listener)
        throws Exception {
        logger.debug("");

        Context context = new InitialContext();

        Object o = context.lookup(CONNECTION_FACTORY);
        ConnectionFactory tcf = (ConnectionFactory) o;
        this.topic = (Topic) context.lookup(topicName);

        this.connection = tcf.createConnection();
        this.session = this.connection.createSession(false, Session.CLIENT_ACKNOWLEDGE);

        this.subscriber = this.session.createConsumer(this.topic);
        this.subscriber.setMessageListener(listener);

        this.connection.start();

        this.information.topicName = topicName;
        this.information.listener = listener;
    }

    public void sendObjectMessage(final Serializable object) throws Exception {
        if (this.connection == null) {
            throw new JMSException("Not connected!");
        }
        logger.debug("");

        // Create a new session in order to avoid
        // IllegalStateException: Illegal control thread
        Session session = this.connection.createSession(false, Session.CLIENT_ACKNOWLEDGE);
        MessageProducer producer = session.createProducer(topic);
        try {
            Message m = session.createObjectMessage(object);
            producer.send(m);
        } finally {
            producer.close();
            session.close();
            producer = null;
            session = null;
            System.gc();
        }
    }

    public void disconnect() throws Exception {
        logger.debug("");
        if (this.connection != null) {
            this.subscriber.close();
            this.session.close();
            this.connection.close();

            this.connection = null;
            this.session = null;
            this.topic = null;
            this.subscriber = null;

            this.information.topicName = null;
            this.information.listener = null;

            System.gc();
        }
    }

    @SuppressWarnings("unused")
    private void writeObject(ObjectOutputStream out) throws IOException {
        out.writeObject(this.information);
    }

    @SuppressWarnings("unused")
    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        try {
            this.information = (ConnectionInformation) in.readObject();

            if (this.information.topicName != null && this.information.listener != null) {
                this.connect(this.information.topicName, this.information.listener);
            } else {
                this.information.topicName = null;
                this.information.listener = null;
            }
        } catch (Throwable t) {
            // Java 5 doesn't have new IOException(message, cause)
            IOException ioe = new IOException("Cannot read object");
            ioe.initCause(t);
            throw ioe;
        }
    }

}
