/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eos.common.jndilocator;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;


public class ServiceLocator {
    /**
     * Singleton class allowing to locate beans through JNDI.
     * It also implements a cache where all the already used objects are
     * put after a lookup in the JNDI.
     * The object is looked up in the cache and if it's not there, in the
     * JNDi.
     * @throws ServiceLocatorException  If there's a problem finding the bean.
     * @throws ParseException   If there's a problem parsing times.
     * @throws IOException      On configuration file failure.
     */

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(ServiceLocator.class);

    /* InitialContext is stored here */
    private Context initialContext;

    /* Keep cache of distant objects already retrieved */
    private Map<String, Object> cache;

    /* Define singleton */
    private static final ServiceLocator instance = new ServiceLocator();

    public static ServiceLocator getInstance() {
        return instance;
    }

    /**
     * Private constructor of ServiceLocator
     * @throws ServiceLocatorException
     */
    private ServiceLocator() throws ServiceLocatorException {
        try {
            initialContext = getInitialContext();
            cache = new HashMap<String, Object>();
        } catch (Exception e) {
            LOGGER.error("Error starting ServiceLocator", e);
            throw new ServiceLocatorException(e);
        }
    }

    /**
     * Get initial context - Carol JNDI
     * @return Context
     * @throws NamingException, ServiceLocatorException
     */
    public Context getInitialContext() throws ServiceLocatorException {

        try {
            ClassLoader old = Thread.currentThread().getContextClassLoader();
            try {
                Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
                return new InitialContext();
            } finally {
                Thread.currentThread().setContextClassLoader(old);
            }
        } catch (NamingException ne) {
            LOGGER.error("Error getting Initial Context", ne);
            throw new ServiceLocatorException(ne);
        }
    }

    /**
     * Get initial context - Carol JNDI
     * @param url Accepts a string that represents the Context.PROVIDER_URL parameter.
     * @return Context
     * @throws NamingException, ServiceLocatorException
     */
    public Context getInitialContext(final String url) throws ServiceLocatorException {

        try {
            // Get an InitialContext
            Properties h = new Properties();
            h.put(Context.INITIAL_CONTEXT_FACTORY,
                            "org.objectweb.carol.jndi.spi.MultiOrbInitialContextFactory");
            h.put(Context.PROVIDER_URL, url);
            ClassLoader old = Thread.currentThread().getContextClassLoader();
            try {
                Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
                return new InitialContext(h);
            } finally {
                Thread.currentThread().setContextClassLoader(old);
            }
        } catch (NamingException ne) {
            LOGGER.error("Error getting Initial Context", ne);
            throw new ServiceLocatorException(ne);
        }
    }

    /**
     * Gets the remote object, adds it to the cache and returns it.
     * @param jndiName
     * @return Object
     * @throws ServiceLocatorException
     */
    public synchronized Object getRemoteObject(final String jndiName) throws ServiceLocatorException {

        Object remoteObject = cache.get(jndiName);
        if (remoteObject == null) {
            ClassLoader old = Thread.currentThread().getContextClassLoader();
            try {
                Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
                remoteObject = initialContext.lookup(jndiName);
                cache.put(jndiName, remoteObject);
            } catch (Exception e) {
                LOGGER.error("Error getting remote object", e);
                throw new ServiceLocatorException(e);
            }   finally {
                Thread.currentThread().setContextClassLoader(old);
            }
        }
        return remoteObject;
    }

}
