/**
 * JASMINe
 * Copyright (C) 2010-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eos.probemanager.view {
import com.adobe.cairngorm.control.CairngormEventDispatcher;

import mx.collections.ArrayCollection;
import mx.containers.Canvas;

import mx.containers.VBox;
import mx.controls.AdvancedDataGrid;
import mx.controls.Alert;
import mx.controls.LinkButton;
import mx.controls.NumericStepper;
import mx.controls.TextInput;
import mx.core.UIComponent;
import mx.events.CloseEvent;
import mx.managers.PopUpManager;
import mx.resources.ResourceManager;

import org.ow2.jasmine.monitoring.eos.probemanager.event.ProbeCommandEvent;
import org.ow2.jasmine.monitoring.eos.probemanager.event.SimpleCommandEvent;
import org.ow2.jasmine.monitoring.eos.probemanager.model.ProbeManagerModelLocator;
import org.ow2.jasmine.monitoring.eos.probemanager.vo.Probe;
import org.ow2.jasmine.monitoring.eos.probemanager.vo.ProbeIndicator;
import org.ow2.jasmine.monitoring.eos.probemanager.vo.ProbeOutput;
import org.ow2.jasmine.monitoring.eos.probemanager.vo.ProbeTarget;


/**
 * @author Philippe Durieux
 */
public class ProbeEditClass extends Canvas {

    [Bindable]
    protected var model:ProbeManagerModelLocator = ProbeManager.getInstance().getModel();

    [Bindable]
    private var dispatcher:CairngormEventDispatcher = ProbeManager.getInstance().getDispatcher();

    [Bindable]
    protected var probe:Probe;

    /**
     * ope may be one of "new","change","copy"
     */
    [Bindable]
    protected var ope:String;

    private var initDone:Boolean = false;

    // -------------------------------------------------------------------
    // These variables are defined in the mxml part.
    // -------------------------------------------------------------------

    [Bindable]
    public var topLevelBox:VBox;

    [Bindable]
    public var indicatorGrid:AdvancedDataGrid;

    [Bindable]
    public var outputGrid:AdvancedDataGrid;

    [Bindable]
    public var targetGrid:AdvancedDataGrid;

    [Bindable]
    public var probeId:TextInput;

    [Bindable]
    public var npperiod:NumericStepper;

    [Bindable]
    public var removeButton:LinkButton;

    // -------------------------------------------------------------------
    // Module init
    // -------------------------------------------------------------------

    public function ProbeEditClass() {
    }

    public function initView():void {
        if (! initDone && this.ope != null) {
            // Depending on timing, init may not have been achieved.
            initComponent(this.ope, this.probe);
        }
    }

    public function initComponent(o:String, p:Probe):void {
        this.probe = p;
        this.ope = o;
        // Check if window initialized before processing
        if (probeId == null || removeButton == null) {
            // window not initialized yet: cannot achieve init.
            return;
        }
        initDone = true;

        // can only remove a probe that already exists
        removeButton.visible = (this.ope == "change");

        if (this.ope != "change") {
            // name must be changed
            this.probe.probeId = "";
        }
        if (this.ope == "new") {
            // Set a raisonable value for default period
            this.probe.period = 15;
        }
        if (this.ope == "change") {
            // Name must not be changed
            probeId.editable = false;
        }
        if (this.ope != "new") {
             // set selected flag on target, output, indicators used by this probe.
            for each (var obj:Object in targetList) {
                if (probe.hasTarget(obj.Name)) {
                    obj.isSelected = true;
                }
            }
            for each (obj in outputList) {
                if (probe.hasOutput(obj.Name)) {
                    obj.isSelected = true;
                }
            }
            for each (obj in indicatorList) {
                if (probe.hasIndicator(obj.Name)) {
                    obj.isSelected = true;
                }
            }
        }
    }

    /**
     * Close the Window
     */
    public function closeWindow():void {
        ProbeManager.getInstance().mainView.removeTab(this as UIComponent);
    }

    // -------------------------------------------------------------------
    // Target list
    // -------------------------------------------------------------------

    [Bindable]
    protected var targetList:ArrayCollection = new ArrayCollection();

    public function set targets(list:ArrayCollection):void {
        for each (var target:ProbeTarget in list) {
            var obj:Object = {Name: target.name, Type: target.type, Properties: target.properties, isSelected: false};
            if (!knownTarget(obj)) {
                targetList.addItem(obj);
            }
        }
    }

    private function knownTarget(object:Object):Boolean {
        for each (var value:Object in targetList) {
            if (value.Name.toString() == object.Name.toString()) {
                return true;
            }
        }
        return false;
    }

    // -------------------------------------------------------------------
    // Indicator List
    // -------------------------------------------------------------------

    [Bindable]
    protected var indicatorList:ArrayCollection = new ArrayCollection();

    public function set indicators(list:ArrayCollection):void {
        for each (var indicator:ProbeIndicator in list) {
            var obj:Object = {Name: indicator.name, Type: indicator.type, Sources: indicator.sources,  Properties: indicator.properties, isSelected: false};
            if (!knownIndicator(obj)) {
                indicatorList.addItem(obj);
            }
        }
    }

    private function knownIndicator(object:Object):Boolean {
        for each (var value:Object in indicatorList) {
            if (value.Name.toString() == object.Name.toString()) {
                return true;
            }
        }
        return false;
    }


    // -------------------------------------------------------------------
    // Output List
    // -------------------------------------------------------------------

    [Bindable]
    protected var outputList:ArrayCollection = new ArrayCollection();

    public function set outputs(list:ArrayCollection):void {
        for each (var output:ProbeOutput in list) {
            var obj:Object = {Name: output.name, Type: output.type, Properties: output.properties, isSelected: false};
            if (!knownOutput(obj)) {
                outputList.addItem(obj);
            }
        }
    }

    private function knownOutput(object:Object):Boolean {
        for each (var value:Object in outputList) {
            if (value.Name.toString() == object.Name.toString()) {
                return true;
            }
        }
        return false;
    }

    // -------------------------------------------------------------------
    // Actions
    // -------------------------------------------------------------------

    public function showHelp():void {
        var helpWindow:ProbeManagerHelp = PopUpManager.createPopUp(topLevelBox, ProbeManagerHelp, true) as ProbeManagerHelp;
        PopUpManager.centerPopUp(helpWindow);
        // TODO use the TabNavigator, but must adapt the HelpWindow class
        //var helpWindow:ProbeManagerHelp = new ProbeManagerHelp();
        //addTab(helpWindow);
    }

    public function remove():void {
        Alert.show("Do you really want to remove this probe ?",
                "Remove Probe",
                Alert.YES | Alert.NO,
                topLevelBox, removeSelectedProbe, null, Alert.YES);

    }

    private function removeSelectedProbe(event:CloseEvent):void {
        if (event.detail == Alert.YES) {
            removeProbe(probe.probeId);
            // close the tab
            closeWindow();
        }
    }

    private function removeProbe(id:String):void {
        var ev:SimpleCommandEvent;
        ev = new SimpleCommandEvent(SimpleCommandEvent.REMOVE_PROBE_EVENT);
        ev.name = id;
        dispatcher.dispatchEvent(ev);

        // close the tab
        closeWindow();
    }

    /**
     * Add a new Probe. Called when user clic on "Apply"
     */
    public function apply():void {
        // update Probe with values from the form
        if (! updateProbe()) {
            return;
        }

        // Apply the operation
        var ev:ProbeCommandEvent;
        if (ope == "new" || ope == "copy") {
            // dispatch event to create the probe on the server
            ev = new ProbeCommandEvent(ProbeCommandEvent.NEW_PROBE_EVENT);
            ev.probe = probe;
            dispatcher.dispatchEvent(ev);

        }
        if (ope == "change") {
            // dispatch event to create the probe on the server
            ev = new ProbeCommandEvent(ProbeCommandEvent.MODIFY_PROBE_EVENT);
            ev.probe = probe;
            dispatcher.dispatchEvent(ev);
        }

        // close the tab
        closeWindow();
    }

    /**
     * Update Probe and validate its values
     * @return true if it seems correct
     */
    public function updateProbe():Boolean {

        // probe name
        probe.probeId = probeId.text;
        if (probe.probeId.length < 1) {
            Alert.show("The Probe has no name defined");
            return false;
        }

        // period
        probe.period = int(npperiod.value);
        if (probe.period < 1) {
            Alert.show("Bad value for period");
            return false;
        }

        // Build lists of targets
        probe.targetId = new ArrayCollection();
        for each (var obj:Object in targetList) {
            if (obj.isSelected) {
                probe.targetId.addItem(obj.Name);
            }
        }

        // Build lists of outputs
        probe.outputId = new ArrayCollection();
        for each (obj in outputList) {
            if (obj.isSelected) {
                probe.outputId.addItem(obj.Name);
            }
        }

        // Build lists of indicators
        probe.indicatorId = new ArrayCollection();
        for each (obj in indicatorList) {
            if (obj.isSelected) {
                probe.indicatorId.addItem(obj.Name);
            }
        }
        if (probe.indicatorId.length < 1) {
            Alert.show("The Probe has no indicator defined");
            return false;
        }

        return true;
    }

    // -----------------------------------------------------------------------------
    // Internationalization
    // -----------------------------------------------------------------------------

    /**
     * Get localized String
     */
    public function locale(s:String):String {
        return ResourceManager.getInstance().getString("probemanagerResources", s);
    }



}
}
