/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.eos.probemanager.model {
import com.adobe.cairngorm.Consumer;
import com.adobe.cairngorm.model.ModelLocator;

import flash.utils.Dictionary;

import mx.collections.ArrayCollection;
import mx.controls.Alert;
import mx.messaging.events.MessageEvent;
import mx.messaging.messages.AsyncMessage;

import org.ow2.jasmine.monitoring.eos.probemanager.business.MyDelegate;
import org.ow2.jasmine.monitoring.eos.probemanager.vo.*;

[Bindable]
public class ProbeManagerModelLocator implements ModelLocator {

    public function ProbeManagerModelLocator() {
        super();
    }

    /**
     * Use only one delegate.
     */
    public var delegate:MyDelegate = null;

    public function getDelegate():MyDelegate {
        if (delegate == null) {
            delegate = new MyDelegate();
        }
        return delegate;
    }

    // -------------------------------------------------------------
    // jasmineProbe topic listener
    // -------------------------------------------------------------

    // Be sure not to use mx std flex Consumer
    private var consumer:com.adobe.cairngorm.Consumer = null;

    public function subscribeJasmineProbeTopic():void {
        var mgr:ProbeManager = ProbeManager.getInstance();
        consumer = mgr.getServices().getConsumer("asynchProbeManagerService");
        if (consumer == null) {
            Alert.show("Cannot get asynchProbeManagerService", "ERROR");
            return;
        }
        consumer.topic = "jasmineProbe";
        consumer.subscribe();
        consumer.addEventListener(MessageEvent.MESSAGE, probeEventHandler);
    }

    public function unsubscribeJasmineProbeTopic():void {
        consumer.unsubscribe();
    }

    /**
     * received an Event from server:
     * it could be a probe state change, or a probe result.
     * @param event
     */
    public function probeEventHandler(event:MessageEvent):void {
        var msg:AsyncMessage = event.message as AsyncMessage;
        if (msg.body is ProbeEvent) {
            // Probe state change
            var probeEvent:ProbeEvent = msg.body as ProbeEvent;
            var probe:Probe = getProbe(probeEvent.probeId);
            if (probe != null) {
                probe.state = probeEvent.state;
                probe.error = probeEvent.error;
                if (probe.state == Probe.RUNNING) {
                    // Probe running implies its targets are OK
                    for each (var t:ProbeTarget in probe.targets) {
                        t.state = ProbeTarget.TARGET_RUNNING;
                    }
                }
            }
            // We must set the currentState in probeDetailView if exists.
            // TODO send an event that will be caught by probeDetailView ? not sure
            return;
        }
        if (msg.body is ProbeResult) {
            // Probe Result received
            var probeResult:ProbeResult = msg.body as ProbeResult;
            probe = getProbe(probeResult.probeId);
            if (probe != null)
            {
                probe.addResult(probeResult);
            }
            return;
        }
        Alert.show("Received an unknown type of message on topic jasmineProbe");
    }


    // -------------------------------------------------------------
    // Probes
    // -------------------------------------------------------------

    /**
     * Selected probe ident
     */
    private var _selectedProbeIdent:String;

    public function get selectedProbeIdent():String {
        return _selectedProbeIdent;
    }

    public function set selectedProbeIdent(value:String):void {
        _selectedProbeIdent = value;
    }

    /**
     *  Array of Probe
     */
    private var _probes:ArrayCollection = new ArrayCollection();

    public function get probes():ArrayCollection {
        return _probes;
    }

    public function set probes(value:ArrayCollection):void {
        this._probes = value;
        if (value.length == 0)
            return;

        // targets, outputs and indicators have not been converted.
        // We must build these lists now.
        buildTargetLists();
        buildOutputLists();
        buildIndicatorLists();

        // Set the selected Probe and reset all selected flags
        var id:String = "";
        var first:Boolean = true;
        var pn:ArrayCollection = new ArrayCollection();
        for each (var probe:Probe in _probes) {
            pn.addItem(probe.probeId);
            if (first) {
                // set the default selected probe as the first one
                first = false;
                id = probe.probeId;
            }
            if (probe.selected == true) {
                id = probe.probeId;
                probe.selected = false;
            }
        }
        // We do use the setter here to force the binding, because
        // the view has to be refreshed.
        probeNames = pn;

        // forces the binding
        _selectedProbeIdent = "";
        this.selectedProbeIdent = id;
    }

    public function getProbe(name:String):Probe {
        var ret:Probe = null;
        for each (var probe:Probe in _probes) {
            if (probe.probeId == name) {
                ret = probe;
                break;
            }
        }
        return ret;
    }

    private var _probeNames:ArrayCollection;

    public function get probeNames():ArrayCollection {
        return _probeNames;
    }

    public function set probeNames(value:ArrayCollection):void {
        _probeNames = value;
    }

// -------------------------------------------------------------
    // ProbeTargets
    // -------------------------------------------------------------

    /**
     *  Array of ProbeTarget
     */
    private var _targets:ArrayCollection = new ArrayCollection();

    public function get targets():ArrayCollection {
        return _targets;
    }

    public function set targets(value:ArrayCollection):void {
        this._targets = value;
        var tn:ArrayCollection = new ArrayCollection();
        for each (var target:ProbeTarget in _targets) {
            tn.addItem(target.name);
        }
        // We do use the setter here to force the binding, because
        // the view has to be refreshed.
        targetNames = tn;
        // rebuild target lists in probes
        buildTargetLists();
    }

    private var _targetNames:ArrayCollection;

    public function get targetNames():ArrayCollection {
        return _targetNames;
    }

    public function set targetNames(value:ArrayCollection):void {
        this._targetNames = value;
    }

    private var _targetTypes:ArrayCollection;

    public function get targetTypes():ArrayCollection {
        return _targetTypes;
    }

    public function set targetTypes(value:ArrayCollection):void {
        _targetTypes = value;
    }

    private var _targetPropertyInfos:ArrayCollection;

    public function get targetPropertyInfos():ArrayCollection {
        return _targetPropertyInfos;
    }

    public function set targetPropertyInfos(value:ArrayCollection):void {
        _targetPropertyInfos = value;
    }

    public function getTarget(name:String):ProbeTarget {
        var ret:ProbeTarget = null;
        for each (var target:ProbeTarget in targets) {
            if (target.name == name) {
                ret = target;
                break;
            }
        }
        return ret;
    }

     public function setUnreachable(name:String):void {
        for each (var target:ProbeTarget in targets) {
            if (target.name == name) {
                target.state = ProbeTarget.TARGET_FAILED;
                break;
            }
        }
    }


    // -------------------------------------------------------------
    // ProbeOutputs
    // -------------------------------------------------------------

    /**
     *  Array of ProbeOutput
     */
    private var _outputs:ArrayCollection = new ArrayCollection();

    public function get outputs():ArrayCollection {
        return _outputs;
    }

    public function set outputs(value:ArrayCollection):void {
        this._outputs = value;
        var on:ArrayCollection = new ArrayCollection();
        for each (var output:ProbeOutput in _outputs) {
            on.addItem(output.name);
        }
        on.addItemAt("", 0);
        // We do use the setter here to force the binding, because
        // the view has to be refreshed.
        outputNames = on;
        // rebuild output lists
        buildOutputLists();
    }

    private var _outputNames:ArrayCollection;

    public function get outputNames():ArrayCollection {
        return _outputNames;
    }

    public function set outputNames(value:ArrayCollection):void {
        this._outputNames = value;
    }

    private var _outputTypes:ArrayCollection;

    public function get outputTypes():ArrayCollection {
        return _outputTypes;
    }

    public function set outputTypes(value:ArrayCollection):void {
        _outputTypes = value;
    }

    private var _outputPropertyInfos:ArrayCollection;

    public function get outputPropertyInfos():ArrayCollection {
        return _outputPropertyInfos;
    }

    public function set outputPropertyInfos(value:ArrayCollection):void {
        _outputPropertyInfos = value;
    }

    public function getOutput(name:String):ProbeOutput {
        var ret:ProbeOutput = null;
        for each (var output:ProbeOutput in _outputs) {
            if (output.name == name) {
                ret = output;
                break;
            }
        }
        return ret;
    }

    // -------------------------------------------------------------
    // ProbeIndicators
    // -------------------------------------------------------------

    /**
     *  Array of ProbeIndicator
     */
    private var _indicators:ArrayCollection = new ArrayCollection();

    public function get indicators():ArrayCollection {
        return _indicators;
    }

    public function set indicators(value:ArrayCollection):void {
        this._indicators = value;
        var on:ArrayCollection = new ArrayCollection();
        for each (var indicator:ProbeIndicator in _indicators) {
            on.addItem(indicator.name);
        }
        on.addItemAt("", 0);
        // We do use the setter here to force the binding, because
        // the view has to be refreshed.
        indicatorNames = on;
        // rebuild indicator lists
        buildIndicatorLists();
    }

    private var _indicatorNames:ArrayCollection;

    public function get indicatorNames():ArrayCollection {
        return _indicatorNames;
    }

    public function set indicatorNames(value:ArrayCollection):void {
        this._indicatorNames = value;
    }

    private var _indicatorTypes:ArrayCollection;

    public function get indicatorTypes():ArrayCollection {
        return _indicatorTypes;
    }

    public function set indicatorTypes(value:ArrayCollection):void {
        _indicatorTypes = value;
        // default = no type selected.
        _indicatorTypes.addItemAt("", 0);
    }

    public function getIndicator(name:String):ProbeIndicator {
        var ret:ProbeIndicator = null;
        for each (var indicator:ProbeIndicator in _indicators) {
            if (indicator.name == name) {
                ret = indicator;
                break;
            }
        }
        return ret;
    }

    // Key is a type, Item is a CollectorInfo
    private var _collectorInfo:Dictionary = new Dictionary();

    public function getCollectorInfo(type:String):CollectorInfo {
        return _collectorInfo[type];
    }

    public function get collectorInfo():Dictionary {
        return _collectorInfo;
    }

    public function setCollectorInfo(type:String, value:CollectorInfo):void {
        _collectorInfo[type] = value;
    }

// ---------------------------------------------------------------
    // MBeans List
    // ---------------------------------------------------------------

    private var _mbeanlist:ArrayCollection;

    public function get mbeanlist():ArrayCollection {
        return _mbeanlist;
    }

    public function set mbeanlist(value:ArrayCollection):void {
        this._mbeanlist = value;
    }

    // ---------------------------------------------------------------
    // private functions
    // ---------------------------------------------------------------

    /**
     * Build the target list of each Probe
     * must be called when either probe list or target list has changed.
     */
    private function buildTargetLists():void {
        for each (var probe:Probe in _probes) {
            // Build its target list
            var targetlist:ArrayCollection = new ArrayCollection();
            for each (var target:ProbeTarget in _targets) {
                for each (var tid:String in probe.targetId) {
                    if (target.name == tid) {
                        targetlist.addItem(target);
                        break;
                    }
                }
            }
            probe.targets = targetlist.source;
        }
    }

    /**
     * Build the output list of each Probe
     * must be called when either probe list or output list has changed.
     */
    private function buildOutputLists():void {
        for each (var probe:Probe in _probes) {
            // Build its output list
            var outputlist:ArrayCollection = new ArrayCollection();
            for each (var output:ProbeOutput in _outputs) {
                for each (var oid:String in probe.outputId) {
                    if (output.name == oid) {
                        outputlist.addItem(output);
                        break;
                    }
                }
            }
            probe.outputs = outputlist.source;
        }
    }

    /**
     * Build the indicator list of each Probe
     * must be called when either probe list or indicator list has changed.
     */
    private function buildIndicatorLists():void {
        for each (var probe:Probe in _probes) {
            // Build its indicator list
            var indicatorlist:ArrayCollection = new ArrayCollection();
            for each (var indicator:ProbeIndicator in _indicators) {
                for each (var oid:String in probe.indicatorId) {
                    if (indicator.name == oid) {
                        indicatorlist.addItem(indicator);
                        break;
                    }
                }
            }
            probe.indicators = indicatorlist.source;
        }
    }
}
}
