/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ProbeManagerService.java 9860 2012-02-14 12:58:01Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.monitoring.eos.probemanager.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Instantiate;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;

import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.jasmine.probe.jmx.JmxBrowserService;
import org.ow2.kerneos.core.service.KerneosSimpleService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Server part of the ProbeManager EoS module.
 * Works with a JasmineProbeManager that can be either mbeancmd-osgi or JasmineProbe
 * @author durieuxp
 */
@Component
@Instantiate
@Provides
public class ProbeManagerService implements KerneosSimpleService {

    protected static final Log logger = LogFactory.getLog(ProbeManagerService.class);

    @ServiceProperty(name = KerneosSimpleService.ID, value = "probeManagerService")
    private String id;

    protected JasmineListener jasmineListener = null;
    protected ProbeTopicProducer producer = null;

    @Requires
    protected JasmineProbeManager probeService;

    @Requires
    protected JmxBrowserService jmxBrowserService;

    private ProbeManagerService() {
        logger.debug("");
        producer = new ProbeTopicProducer();
        jasmineListener = new JasmineListener(producer);

    }

    @Validate
    private void start()
    {
        logger.debug("");
        probeService.addProbeListener(producer);
    }

    @Invalidate
    private void stop()
    {
        logger.debug("");
        probeService.removeProbeListener(producer);
    }

    // ------------------------------------------------------------------------
    // public interface of the service
    // ------------------------------------------------------------------------

    /**
     * Retrieve list of targets defined on this server
     * @return Array of targets
     */
    public ArrayList<ProbeTarget> getTargets() throws ProbeManagerException {

        logger.debug("get Targets");

        Collection<JasmineTarget> targets = null;
        try {
            targets = probeService.getTargets();
        } catch (Exception e) {
            logger.error("Cannot get list of targets:" + e);
            throw new ProbeManagerException("Cannot get list of targets: " + e.getMessage());
        }
        logger.debug("found " + targets.size() + " targets");
        ArrayList<ProbeTarget> result = new ArrayList<ProbeTarget>();
        for (JasmineTarget jt : targets) {
            result.add(newProbeTarget(jt));
        }
        return result;
    }

    /**
     * Create a new target
     * @return Array of targets
     */
    public ArrayList<ProbeTarget> newTarget(final ProbeTarget target) throws ProbeManagerException {

        logger.debug("create new target");

        // Create target and get the new list
        Collection<JasmineTarget> targets = null;
        try {
            JasmineTarget jt = newJasmineTarget(target);
            probeService.createTarget(jt);
            targets = probeService.getTargets();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot create target:" + e);
            throw new ProbeManagerException("Cannot create target: " + e.getMessage());
        }
        ArrayList<ProbeTarget> result = new ArrayList<ProbeTarget>();
        for (JasmineTarget jt : targets) {
            result.add(newProbeTarget(jt));
        }
        return result;
    }

    /**
     * Remove a target
     * @return Array of targets
     */
    public ArrayList<ProbeTarget> removeTarget(final String name) throws ProbeManagerException {

        logger.debug("remove target");

        // Remove target and get the new list
        Collection<JasmineTarget> targets = null;
        try {
            probeService.removeTarget(name);
            targets = probeService.getTargets();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot remove target:" + e);
            throw new ProbeManagerException("Cannot remove target: " + e.getMessage());
        }
        ArrayList<ProbeTarget> result = new ArrayList<ProbeTarget>();
        for (JasmineTarget jt : targets) {
            result.add(newProbeTarget(jt));
        }
        return result;
    }


    /**
     * modify a Target
     * @param target value object
     * @return new Array of targets defined
     */
    public ArrayList<ProbeTarget> modifyTarget(final ProbeTarget target) throws ProbeManagerException {

        logger.debug("");

        Collection<JasmineTarget> targets = null;
        try {
            probeService.changeTarget(newJasmineTarget(target));
            targets = probeService.getTargets();
        } catch (Exception e) {
            logger.error("Cannot modify Target:" + e);
            throw new ProbeManagerException("Cannot modify Target: " + e.getMessage());
        }
        ArrayList<ProbeTarget> result = new ArrayList<ProbeTarget>();
        for (JasmineTarget jt : targets) {
            result.add(newProbeTarget(jt));
        }
        return result;
    }

    /**
     * Get the list of possible types for a Target
     * @return List of types
     * @throws ProbeManagerException
     */
    public ArrayList<String> getTargetTypes() throws ProbeManagerException {
        logger.debug("");

        Set<String> types = null;
        try {
            types = probeService.getTargetTypes();
        } catch (Exception e) {
            logger.error("Cannot get list of target types:" + e);
            throw new ProbeManagerException("Cannot get list of target types: " + e.getMessage());
        }
        logger.debug("found " + types.size() + " target types");
        ArrayList<String> result = new ArrayList<String>();
        for (String type : types) {
            result.add(type);
        }
        return result;
    }

    /**
     * Get the list of possible properties for a target type
     * @param type
     * @return List of PropertyInfo
     * @throws ProbeManagerException
     */
    public List<JasminePropertyInfo> getTargetPropertyInfos(final String type) throws ProbeManagerException {
        logger.debug(type);

        List<JasminePropertyInfo> infos;
        try {
            infos = probeService.getTargetPropertyInfos(type);
        } catch (Exception e) {
            logger.error("Cannot get list of target property infos:" + e);
            throw new ProbeManagerException("Cannot get list of target property infos: " + e.getMessage());
        }
        logger.debug("found " + infos.size() + " target properties");
        return infos;
    }


    /**
     * Retrieve list of indicators defined on this server
     * @return Array of indicators
     */
    public ArrayList<ProbeIndicator> getIndicators() throws ProbeManagerException {

        logger.debug("");

        Collection<JasmineIndicator> indicators = null;
        try {
            indicators = probeService.getIndicators();
        } catch (Exception e) {
            logger.error("Cannot get list of indicators:" + e);
            throw new ProbeManagerException("Cannot get list of indicators: " + e.getMessage());
        }
        logger.debug("found " + indicators.size() + " indicators");
        ArrayList<ProbeIndicator> result = new ArrayList<ProbeIndicator>();
        for (JasmineIndicator jt : indicators) {
            result.add(newProbeIndicator(jt));
        }
        return result;
    }

    /**
     * Create a new indicator
     * @return Array of indicators
     */
    public ArrayList<ProbeIndicator> newIndicator(final ProbeIndicator indicator) throws ProbeManagerException {

        logger.debug("");

        // Create indicator and get the new list
        Collection<JasmineIndicator> indicators = null;
        try {
            JasmineIndicator jt = newJasmineIndicator(indicator);
            probeService.createIndicator(jt);
            indicators = probeService.getIndicators();
        } catch (Exception e) {
            logger.error("Cannot create indicator:" + e);
            throw new ProbeManagerException("Cannot create indicator: " + e.getMessage());
        }
        ArrayList<ProbeIndicator> result = new ArrayList<ProbeIndicator>();
        for (JasmineIndicator jt : indicators) {
            result.add(newProbeIndicator(jt));
        }
        return result;
    }

    /**
     * Remove an indicator
     * @return Array of indicators
     */
    public ArrayList<ProbeIndicator> removeIndicator(final String name) throws ProbeManagerException {

        logger.debug("");

        // Remove indicator and get the new list
        Collection<JasmineIndicator> indicators = null;
        try {
            probeService.removeIndicator(name);
            indicators = probeService.getIndicators();
        } catch (Exception e) {
            logger.error("Cannot remove indicator:" + e);
            throw new ProbeManagerException("Cannot remove indicator: " + e.getMessage());
        }
        ArrayList<ProbeIndicator> result = new ArrayList<ProbeIndicator>();
        for (JasmineIndicator jt : indicators) {
            result.add(newProbeIndicator(jt));
        }
        return result;
    }


    /**
     * modify an indicator
     * @param indicator value object
     * @return new Array of indicators defined
     */
    public ArrayList<ProbeIndicator> modifyIndicator(final ProbeIndicator indicator) throws ProbeManagerException {

        logger.debug("");

        Collection<JasmineIndicator> indicators = null;
        try {
            probeService.changeIndicator(newJasmineIndicator(indicator));
            indicators = probeService.getIndicators();
        } catch (Exception e) {
            logger.error("Cannot modify Indicator:" + e);
            throw new ProbeManagerException("Cannot modify Indicator: " + e.getMessage());
        }
        ArrayList<ProbeIndicator> result = new ArrayList<ProbeIndicator>();
        for (JasmineIndicator jt : indicators) {
            result.add(newProbeIndicator(jt));
        }
        return result;
    }


    /**
     * Get the list of possible types for an Indicator
     * @return List of JasmineCollectorInfo
     * @throws ProbeManagerException
     */
    public ArrayList<JasmineCollectorInfo> getIndicatorTypes() throws ProbeManagerException {
        logger.debug("");

        Set<String> types = null;
        Set<JasmineCollectorInfo> ret = null;
        try {
            types = probeService.getIndicatorTypes();
        } catch (Exception e) {
            logger.error("Cannot get list of indicator types:" + e);
            throw new ProbeManagerException("Cannot get list of indicator types: " + e.getMessage());
        }
        logger.debug("found " + types.size() + " indicator types");
        ArrayList<JasmineCollectorInfo> result = new ArrayList<JasmineCollectorInfo>();
        for (String type : types) {
            JasmineCollectorInfo info = probeService.getIndicatorTypeInfo(type);
            result.add(info);
        }
        return result;
    }

    /**
     * Retrieve list of outputs defined on this server
     * @return Array of outputs
     */
    public ArrayList<ProbeOutput> getOutputs() throws ProbeManagerException {

        logger.debug("get Outputs");

        Collection<JasmineOutput> outputs = null;
        try {
            outputs = probeService.getOutputs();
        } catch (Exception e) {
            logger.error("Cannot get list of outputs:" + e);
            throw new ProbeManagerException("Cannot get list of outputs: " + e.getMessage());
        }
        logger.debug("found " + outputs.size() + " outputs");
        ArrayList<ProbeOutput> result = new ArrayList<ProbeOutput>();
        for (JasmineOutput jo : outputs) {
            result.add(newProbeOutput(jo));
        }
        return result;
    }

    /**
     * Create a new output
     * @return Array of outputs
     */
    public ArrayList<ProbeOutput> newOutput(final ProbeOutput output) throws ProbeManagerException {

        logger.debug("create new output");

        // Create output and get the new list
        Collection<JasmineOutput> outputs = null;
        try {
            probeService.createOutput(newJasmineOutput(output));
            outputs = probeService.getOutputs();
        } catch (Exception e) {
            logger.error("Cannot create new output:" + e);
            throw new ProbeManagerException("Cannot Create new output: " + e.getMessage());
        }
        ArrayList<ProbeOutput> result = new ArrayList<ProbeOutput>();
        for (JasmineOutput jo : outputs) {
            result.add(newProbeOutput(jo));
        }
        return result;
    }

    /**
     * Remove an output
     * @return Array of outputs
     */
    public ArrayList<ProbeOutput> removeOutput(final String name) throws ProbeManagerException {

        logger.debug("remove output");

        // Remove output and get the new list
        Collection<JasmineOutput> outputs = null;
        try {
            probeService.removeOutput(name);
            outputs = probeService.getOutputs();
        } catch (Exception e) {
            logger.error("Cannot remove output:" + e);
            throw new ProbeManagerException("Cannot remove output: " + e.getMessage());
        }
        ArrayList<ProbeOutput> result = new ArrayList<ProbeOutput>();
        for (JasmineOutput jo : outputs) {
            result.add(newProbeOutput(jo));
        }
        return result;
    }

    /**
     * modify an Output
     * @param output value object
     * @return new Array of outputs defined
     */
    public ArrayList<ProbeOutput> modifyOutput(final ProbeOutput output) throws ProbeManagerException {

        logger.debug("modify Output");

        Collection<JasmineOutput> outputs = null;
        try {
            probeService.changeOutput(newJasmineOutput(output));
            outputs = probeService.getOutputs();
        } catch (Exception e) {
            logger.error("Cannot modify Output:" + e);
            throw new ProbeManagerException("Cannot modify Output: " + e.getMessage());
        }
        ArrayList<ProbeOutput> result = new ArrayList<ProbeOutput>();
        for (JasmineOutput jo : outputs) {
            result.add(newProbeOutput(jo));
        }
        return result;
    }

    /**
     * Get the list of possible types for an Output
     * @return List of types
     * @throws ProbeManagerException
     */
    public ArrayList<String> getOutputTypes() throws ProbeManagerException {
        logger.debug("");

        Set<String> types = null;
        try {
            types = probeService.getOutputTypes();
        } catch (Exception e) {
            logger.error("Cannot get list of output types:" + e);
            throw new ProbeManagerException("Cannot get list of output types: " + e.getMessage());
        }
        logger.debug("found " + types.size() + " output types");
        ArrayList<String> result = new ArrayList<String>();
        for (String type : types) {
            result.add(type);
        }
        return result;
    }

    /**
     * Get the list of possible properties for an output type
     * @param type
     * @return List of PropertyInfo
     * @throws ProbeManagerException
     */
    public List<JasminePropertyInfo> getOutputPropertyInfos(final String type) throws ProbeManagerException {
        logger.debug(type);

        List<JasminePropertyInfo> infos;
        try {
            infos = probeService.getOutputPropertyInfos(type);
        } catch (Exception e) {
            logger.error("Cannot get list of output property infos:" + e);
            throw new ProbeManagerException("Cannot get list of output property infos: " + e.getMessage());
        }
        logger.debug("found " + infos.size() + " output properties");
        return infos;
    }

    /**
     * Create a new probe
     * @return Array of probes
     */
    public ArrayList<Probe> newProbe(final Probe probe) throws ProbeManagerException {

        logger.debug("create new probe");

        // Create probe and get the new list
        Collection<JasmineProbe> probes = null;
        String id = null;
        try {
            id = probeService.createProbe(newJasmineProbe(probe));
        } catch (Exception e) {
            logger.error("Cannot create probe:" + e);
            throw new ProbeManagerException("Cannot create probe: " + e.getMessage());
        }
        try {
            probes = probeService.getProbes();
        } catch (Exception e) {
            logger.error("Cannot get list of probes:" + e);
            throw new ProbeManagerException("Cannot get list of probes: " + e.getMessage());
        }
        ArrayList<Probe> result = new ArrayList<Probe>();
        for (JasmineProbe jp : probes) {
            Probe p = newProbe(jp);
            p.setSelected(jp.getId().equals(id));
            result.add(p);
        }
        return result;
    }

    /**
     * Retrieve list of probes defined on this server
     * @return Array of probes defined
     */
    public ArrayList<Probe> getProbes() throws ProbeManagerException {

        logger.debug("get Probes");

        // Get the list
        Collection<JasmineProbe> probes = null;
        try {
            probes = probeService.getProbes();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot get list of probes:" + e);
            throw new ProbeManagerException("Cannot get list of probes: " + e.getMessage());
        }
        logger.debug("found " + probes.size() + " probes");
        ArrayList<Probe> result = new ArrayList<Probe>();
        int select = 0;
        for (JasmineProbe jp : probes) {
            Probe p = newProbe(jp);
            p.setSelected(select++ == 0);
            result.add(p);
        }
        return result;
    }

    /**
     * remove a Probe
     * @param ident Probe Ident
     * @return new Array of probes defined
     */
    public ArrayList<Probe> removeProbe(final String ident) throws ProbeManagerException {

        // Get the updated list
        Collection<JasmineProbe> probes = null;
        try {
            logger.debug("remove Probe");
            probeService.removeProbe(ident);
            probes = probeService.getProbes();
        } catch (Exception e) {
            logger.error("Cannot remove Probe:" + e);
            throw new ProbeManagerException("Cannot remove Probe: " + e.getMessage());
        }
        ArrayList<Probe> result = new ArrayList<Probe>();
        int select = 0;
        for (JasmineProbe jp : probes) {
            Probe p = newProbe(jp);
            p.setSelected(select++ == 0);
            result.add(p);
        }
        return result;
    }

    /**
     * start a Probe
     * @param ident Probe Ident
     * @return new Array of probes defined
     */
    public void startProbe(final String ident) throws ProbeManagerException {

        try {
            logger.debug("start Probe");
            probeService.startProbe(ident);
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot start Probe:" + e);
            throw new ProbeManagerException("Cannot start Probe: " + e.getMessage());
        }
    }

    /**
     * start all Probes
     * @return new Array of probes defined
     */
    public void startAllProbes() throws ProbeManagerException {

        try {
            logger.debug("start All Probes");
            probeService.startAllProbes();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot start Probe:" + e);
            throw new ProbeManagerException("Cannot start Probe: " + e.getMessage());
        }
    }

    /**
     * stop a Probe
     * @param ident Probe Ident
     * @return new Array of probes defined
     */
    public void stopProbe(final String ident) throws ProbeManagerException {

        try {
            logger.debug("stop Probe");
            probeService.stopProbe(ident);
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot stop Probe:" + e);
            throw new ProbeManagerException("Cannot stop Probe: " + e.getMessage());
        }
    }

    /**
     * stop all started Probes
     * @return new Array of probes defined
     */
    public void stopAllProbes() throws ProbeManagerException {

        try {
            logger.debug("stop All Probes");
            probeService.stopAllProbes();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot stop Probe:" + e);
            throw new ProbeManagerException("Cannot stop Probe: " + e.getMessage());
        }
    }

    public ArrayList<Probe> removeAllProbes() throws ProbeManagerException {

        Collection<JasmineProbe> probes = null;
        try {
            logger.debug("remove All Probes");
            probeService.removeAllProbes();
            probes = probeService.getProbes();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot remove Probe:" + e);
            throw new ProbeManagerException("Cannot remove Probe: " + e.getMessage());
        }
        ArrayList<Probe> result = new ArrayList<Probe>();
        for (JasmineProbe jp : probes) {
            Probe p = newProbe(jp);
            result.add(p);
        }
        return result;
    }

    /**
     * modify a probe
     * @param probe value object
     * @return new Array of probes defined
     */
    public ArrayList<Probe> modifyProbe(final Probe probe) throws ProbeManagerException {

        // Get the updated list
        Collection<JasmineProbe> probes = null;
        String id = probe.getProbeId();
        try {
            logger.debug("modify Probe");
            probeService.changeProbe(newJasmineProbe(probe));
            probes = probeService.getProbes();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot modify Probe:" + e);
            throw new ProbeManagerException("Cannot modify Probe: " + e.getMessage());
        }
        ArrayList<Probe> result = new ArrayList<Probe>();
        for (JasmineProbe jp : probes) {
            Probe p = newProbe(jp);
            p.setSelected(jp.getId().equals(id));
            result.add(p);
        }
        return result;
    }

    /**
     * Change value of probe period, without stopping the probe if it is running.
     * @param ident Probe Ident
     * @param period
     * @throws ProbeManagerException
     */
    public void changeProbePeriod(final String ident, final int period) throws ProbeManagerException {
        try {
            logger.debug("change Probe period");
            probeService.changeProbePeriod(ident, period);
        } catch (Exception e) {
            logger.error("Cannot change Probe period:" + e);
            throw new ProbeManagerException("Cannot change Probe period: " + e.getMessage());
        }
    }

    /**
     * copy a Probe
     * @param probe Probe to be duplicated
     * @return new Array of probes defined
     */
    public ArrayList<Probe> copyProbe(final Probe probe) throws ProbeManagerException {

        logger.debug("copy Probe");

        Collection<JasmineProbe> probes = null;
        String id = null;
        try {
            id = probeService.createProbe(newJasmineProbe(probe));
            probes = probeService.getProbes();
        } catch (Exception e) {
            logger.error("Cannot copy Probe:" + e);
            throw new ProbeManagerException("Cannot copy Probe: " + e.getMessage());
        }
        logger.debug("found " + probes.size() + " probes");
        ArrayList<Probe> result = new ArrayList<Probe>();
        for (JasmineProbe jp : probes) {
            Probe p = newProbe(jp);
            p.setSelected(jp.getId().equals(id));
            result.add(p);
        }
        return result;
    }

    /**
     * Save configuration in xml
     */
    public void saveConfig() throws ProbeManagerException {

        logger.debug("save configuration");

        try {
            probeService.saveConfig(null);
        } catch (Exception e) {
            logger.error("Cannot save configuration:" + e);
            throw new ProbeManagerException("Cannot save configuration: " + e.getMessage());
        }
    }

    /**
     * Start listening probe results
     */
    public void startListen(final String probeId) throws ProbeManagerException {

        logger.debug("start listening");

        try {
            jasmineListener.startListening(probeId);
        } catch (Exception e) {
            logger.error("Cannot start listening:" + e);
            throw new ProbeManagerException("Cannot start listening: " + e.getMessage());
        }
    }

    /**
     * Stop listening probe results
     */
    public void stopListen(final String probeId) throws ProbeManagerException {

        logger.debug("stop listening");

        try {
            jasmineListener.stopListening(probeId);
        } catch (Exception e) {
            logger.error("Cannot stop listening:" + e);
            throw new ProbeManagerException("Cannot stop listening: " + e.getMessage());
        }
    }

    public ArrayList<Probe> loadConf(final String url) throws ProbeManagerException {

        logger.debug("load configuration file: " + url);

        // Get the updated list
        Collection<JasmineProbe> probes = null;
        try {
            probeService.loadConfig(url);
            probes = probeService.getProbes();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot load Config:" + e);
            throw new ProbeManagerException("Cannot load config: " + e.getMessage());
        }
        ArrayList<Probe> result = new ArrayList<Probe>();
        int select = 0;
        for (JasmineProbe jp : probes) {
            Probe p = newProbe(jp);
            p.setSelected(select++ == 0);
            result.add(p);
        }
        return result;
    }

    public ArrayList<String> getMBeans(final String filter, final ProbeTarget target) throws ProbeManagerException {

        logger.debug("get Mbeans matching: " + filter);

        Set<ObjectName> mbeans = null;
        try {
            // TODO mbeans = jmxBrowserService.getMBeans(newJasmineTarget(target), filter);
        } catch (Exception e) {
            logger.error("Cannot get mbeans: " + e);
            throw new ProbeManagerException("Cannot get mbeans: " + e.getMessage());
        }
        ArrayList<String> result = new ArrayList<String>();
        for (ObjectName mbean : mbeans) {
            result.add(mbean.getCanonicalName());
        }
        return result;
    }

    // -------------------------------------------------------------------------------------
    // private methods
    // -------------------------------------------------------------------------------------

    // -------------------------------------------------------------------------------------
    // converters
    // -------------------------------------------------------------------------------------

    /**
     * Convert JasmineTarget to ProbeTarget
     * @param jt JasmineTarget
     * @return ProbeTarget
     * @throws ProbeManagerException
     */
    private ProbeTarget newProbeTarget(final JasmineTarget jt) throws ProbeManagerException {
        ProbeTarget t = new ProbeTarget();
        t.setName(jt.getName());
        t.setType(jt.getType());
        t.setProperties(map2list(jt.getProperties()));

        // set target state
        switch (jt.getState()) {
            case JasmineTarget.TARGET_FAILED:
                t.setState(ProbeTarget.TARGET_FAILED);
                break;
            case JasmineTarget.TARGET_RUNNING:
                t.setState(ProbeTarget.TARGET_RUNNING);
                break;
            default:
                t.setState(ProbeTarget.TARGET_UNKNOWN);
                break;
        }

        return t;
    }

    /**
     * Convert ProbeTarget to JasmineTarget
     * @param t ProbeTarget
     * @return JasmineTarget
     * @throws ProbeManagerException
     */
    private JasmineTarget newJasmineTarget(final ProbeTarget t) throws ProbeManagerException {
        JasmineTarget jt = new JasmineTarget();
        jt.setName(t.getName());
        jt.setType(t.getType());
        Map<String, String> properties = list2map(t.getProperties());
        jt.setProperties(properties);
        // Do not pass target state: it is better known by the server side.
        return jt;
    }

    /**
     * Convert JasmineOutput to ProbeOutput
     * @param jo JasmineOutput
     * @return ProbeOutput
     * @throws ProbeManagerException
     */
    private ProbeOutput newProbeOutput(final JasmineOutput jo) throws ProbeManagerException {
        ProbeOutput o = new ProbeOutput();
        o.setName(jo.getName());
        o.setType(jo.getType());
        o.setProperties(map2list(jo.getProperties()));
        return o;
    }

    /**
     * Convert ProbeOutput to JasmineOutput
     * @param output ProbeOutput
     * @return JasmineOutput
     * @throws ProbeManagerException
     */
    private JasmineOutput newJasmineOutput(final ProbeOutput output) throws ProbeManagerException {
        JasmineOutput jo = new JasmineOutput();
        jo.setName(output.getName());
        jo.setType(output.getType());
        jo.setProperties(list2map(output.getProperties()));
        return jo;
    }

    /**
     * Convert JasmineIndicator to ProbeIndicator
     * @param jo JasmineIndicator
     * @return ProbeIndicator
     * @throws ProbeManagerException
     */
    private ProbeIndicator newProbeIndicator(final JasmineIndicator jo) throws ProbeManagerException {
        ProbeIndicator o = new ProbeIndicator();
        o.setName(jo.getName());
        o.setType(jo.getType());
        o.setScale(jo.getScale());
        o.setSources(jo.getSources());
        o.setProperties(map2list(jo.getProperties()));
        return o;
    }

    /**
     * Convert ProbeIndicator to JasmineIndicator
     * @param indicator ProbeIndicator
     * @return JasmineIndicator
     * @throws ProbeManagerException
     */
    private JasmineIndicator newJasmineIndicator(final ProbeIndicator indicator) throws ProbeManagerException {
        JasmineIndicator jo = new JasmineIndicator();
        jo.setName(indicator.getName());
        jo.setType(indicator.getType());
        jo.setScale(indicator.getScale());
        jo.setSources(indicator.getSources());
        jo.setProperties(list2map(indicator.getProperties()));
        return jo;
    }

    /**
     * Convert JasmineProbe to Probe
     * @param jp JasmineProbe
     * @return Probe
     * @throws ProbeManagerException
     */
    private Probe newProbe(final JasmineProbe jp) throws ProbeManagerException {
        Probe p = new Probe();
        p.setProbeId(jp.getId());
        p.setPeriod(jp.getPeriod());
        p.setOutputId(jp.getOutputList());
        p.setTargetId(jp.getTargetList());
        p.setIndicatorId(jp.getIndicatorList());

        // set probe state
        switch (jp.getStatus()) {
            case JasmineProbe.PROBE_FAILED:
                p.setState(Probe.PROBE_FAILED);
                break;
            case JasmineProbe.PROBE_RUNNING:
                p.setState(Probe.PROBE_RUNNING);
                break;
            case JasmineProbe.PROBE_STARTED:
                p.setState(Probe.PROBE_STARTED);
                break;
            case JasmineProbe.PROBE_STOPPED:
                p.setState(Probe.PROBE_STOPPED);
                break;
        }
        return p;
    }

    /**
     * Convert Probe to JasmineProbe
     * @param probe
     * @return JasmineProbe
     * @throws ProbeManagerException
     */
    private JasmineProbe newJasmineProbe(final Probe probe) throws  ProbeManagerException {
        JasmineProbe jp = new JasmineProbe();
        jp.setId(probe.getProbeId());
        jp.setPeriod(probe.getPeriod());
        jp.setTargetList(probe.getTargetId());
        jp.setOutputList(probe.getOutputId());
        jp.setIndicatorList(probe.getIndicatorId());
        return jp;
    }

    /**
     * Convert a Map to a list of properties
     */
    private List<String> map2list(Map<String, String> map) {
        List<String> ret = new ArrayList<String>();
        if (map != null) {
            for (String key : map.keySet()) {
                String value = map.get(key);
                ret.add(key+"="+value);
            }
        }
        return ret;
    }

    /**
     * Convert a list of properties to a Map
     */
    private Map<String, String> list2map(List<String> list) {
        Map<String, String> ret = new HashMap<String, String>();
        if (list != null) {
            for (String property : list) {
                int index = property.indexOf("=");
                String key = property.substring(0, index);
                String value = property.substring(index + 1);
                ret.put(key, value);
            }
        }
        return ret;
    }
}
