package org.ow2.jasmine.monitoring.eos.probemanager.service;

import java.io.Serializable;
import java.util.StringTokenizer;

import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.Topic;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.ow2.jasmine.event.messages.JasmineEvent;
import org.ow2.jasmine.event.messages.JasmineEventDetails;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeListener;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This class receives events from JasmineListener and resend them on the jasmineProbe Topic
 * @author durieuxp
 */
public class ProbeTopicProducer implements JasmineProbeListener {

    protected Log logger = LogFactory.getLog(this.getClass());

    // Producer of "jasmineProbe" topic
    protected Connection probeconn = null;
    protected Session probesess = null;
    protected Topic probetopic = null;
    protected MessageProducer probeProducer = null;

    public ProbeTopicProducer() {
    }

    // ---------------------------------------------------------------------------------
    // JasmineProbeListener interface
    // ---------------------------------------------------------------------------------

    /**
     * Notifies a Probe state change
     * @param probe
     */
    public void notifyEvent(final JasmineProbe probe) {
        logger.debug("Publish State Change to " + probe.getStatus());
        ProbeEvent event = new ProbeEvent(probe.getId(), probe.getStatus(), probe.getError());
        sendMessageToTopic(event);
    }

    // ---------------------------------------------------------------------------------
    // Public methods
    // ---------------------------------------------------------------------------------

    /**
     * Send a message on the Topic for flex clients
     * @param event Event sent by the eventswitch (JasmineEvent format)
     */
    public void publishResult(final JasmineEvent event) {
        logger.debug("publish result on Probe Topic");

        if (! getProducerForProbeTopic()) {
            return;
        }
        try {
            for (JasmineEventDetails detail : event.getEvents()) {
                ProbeResult result = new ProbeResult(event.getProbeId());
                // target
                result.setTarget(event.getServer());
                // probe = mbean:attribute
                int index = detail.getProbe().lastIndexOf(':');
                String mbean = detail.getProbe().substring(0, index);
                String attrib = detail.getProbe().substring(index + 1);
                result.setAttrname(attrib);
                index = mbean.indexOf("name=");
                if (index >= 0) {
                    // If any, keep only the name part
                    String name = mbean.substring(index + 5);
                    StringTokenizer st = new StringTokenizer(name, ":;=, ");
                    result.setMbean(st.nextToken());
                } else {
                    // Else, keep all the mbean name
                    result.setMbean(mbean);
                }
                // attribute value
                result.setAttrvalue(detail.getValue());
                // timestamp
                result.setTimestamp(detail.getTimestamp());
                // send message
                Message message = probesess.createObjectMessage(result);
                probeProducer.send(message);
            }
        } catch (JMSException e) {
            logger.warn("Cannot send message to topic: " + e);
        }
    }

    public void releaseProbeConnection() {
        if (probesess != null) {
            try {
                probesess.close();
                probesess = null;
            } catch (JMSException ignore) {
            }
        }
        if (probeconn != null) {
            try {
                probeconn.close();
                probeconn = null;
            } catch (JMSException ignore) {
            }
        }
    }

    // ---------------------------------------------------------------------------------
    // Private methods
    // ---------------------------------------------------------------------------------

    /**
     * Open a Session to send a message on the jasmineProbe Topic
     * @param message message to send
     */
    private void sendMessageToTopic(final Serializable message) {
        if (! getProducerForProbeTopic()) {
            return;
        }
        try {
            Message objm = probesess.createObjectMessage(message);
            probeProducer.send(objm);
        } catch (JMSException e) {
            logger.warn("Cannot send message to topic: " + e);
        } catch (ClassCastException e) {
            logger.warn("Cannot send message to topic: " + e);
        }
    }

    private boolean getProducerForProbeTopic() {
        // First of all, we need a Connection 
        if (probeconn == null) {
            probesess = null;
            InitialContext ictx = null;
            ConnectionFactory factory = null;
            try {
                ictx = new InitialContext();
                factory = (ConnectionFactory) ictx.lookup("CF");
                probeconn = factory.createConnection();
            } catch (Exception e) {
                logger.error("Cannot get a connection to send message on topic: " + e);
                return false;
            }
        }
        // Then, get a Session on this Connection
        if (probesess == null) {
            probeProducer = null;
            try {
                probesess = probeconn.createSession(false, Session.AUTO_ACKNOWLEDGE);
            } catch (Exception e) {
                logger.error("Cannot get a Session to send message on topic: " + e);
                return false;
            }
        }
        // Get the Producer for the Topic on this session
        if (probeProducer == null) {
            try {
                probeProducer = probesess.createProducer(getProbeTopic());
            } catch (Exception e) {
                logger.error("Cannot get a Producer to send message on topic: " + e);
                return false;
            }
        }
        return true;
    }

    private Topic getProbeTopic() {
        if (probetopic == null) {
            InitialContext ictx = null;
            try {
                ictx = new InitialContext();
                probetopic = (Topic) ictx.lookup("jasmineProbe");
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return probetopic;
    }

}
