/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.monitoring.tests.dbfill;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * @author joaninh
 * 
 */

public class Cmd {


    static final String PUBLISHER_TYPE_PROPERTY_NAME = "publisher.type";
    static final String PUBLISHER_TYPE_EJB = "ejb";
    static final String PUBLISHER_TYPE_MULE = "mule";
    static final String PUBLISHER_TYPE_JDBC = "jdbc";
    static final String DEFAULT_PUBLISHER_TYPE = "ejb";

    static private Log logger = LogFactory.getLog(Cmd.class);

    /**
     * @param args
     */
    public static void main(String[] args) {
        
        if (args.length > 0) {
            if ("-help".equals(args[0])) {
                usage();
                return;
            }
        }
        
        IPublisher publisher;

        logger.info(">> Beginning of fill database.");
        
        // Create the Publisher
        String publisherType = System.getProperty(PUBLISHER_TYPE_PROPERTY_NAME, DEFAULT_PUBLISHER_TYPE);
        logger.info(">> Publisher type = " + publisherType.toUpperCase());
        if (publisherType.equalsIgnoreCase(PUBLISHER_TYPE_EJB)) {
            publisher = new EjbPublisher();
        } else if (publisherType.equalsIgnoreCase(PUBLISHER_TYPE_MULE)) {
            publisher = new MulePublisher();
        } else if (publisherType.equalsIgnoreCase(PUBLISHER_TYPE_JDBC)) {
            publisher = new JdbcPublisher();
        } else {
            logger.error("ERROR: unknown publisher type");
            throw new Error("unknown publisher type");
        }

        // Get the configuration
        DbFillConfig config = new DbFillConfig();

        // Get the configuration common to all the events
        long timeTo = config.getTimeFrom();
        long timeFrom = config.getTimeTo();
        String domain = config.getDomain();
        String server = config.getServer();
        String sName = config.getSName();
        String source = config.getSource();

        // Get the configuration for each event
        for (JTEventConfig eventConfig : config.getEventConfigList()) {

            logger.info(">> Publish datas for the event config.: " + eventConfig.toString());

            int prevValue;
            int n = eventConfig.getVDeltaNeg()*(eventConfig.getVMax()-eventConfig.getVMin());
            int d = eventConfig.getVDeltaPos()+eventConfig.getVDeltaNeg();
            int value;
            if (d != 0) {
                value = eventConfig.getVMin() + (n / d);
            } else {
                value = (eventConfig.getVMax()+eventConfig.getVMin())/2;
            }
            long timeCurrent = timeTo;
            JTEvent event = new JTEvent(domain, server, sName, source, eventConfig.getProbeId(), eventConfig.getMBean(), eventConfig.getVName(), timeCurrent, value);

            // Create and publish multiple events from a event configuration
            // from the timestamp timeFrom to the timestamp timeTo
            while (timeCurrent <= timeFrom) {
                event.setTime(timeCurrent);
                event.setValue(value);
                publisher.publishData(event);
                logger.debug(">> Published event.: " + event.toString());
                prevValue = value;
                value = JTEvent.computeNextValue(eventConfig.getVMin(), eventConfig.getVMax(), eventConfig.getVDeltaNeg(), eventConfig.getVDeltaPos(), prevValue);
                timeCurrent += eventConfig.getPeriod() * 1000;
            }

        }

        // dispose the muleOuter otherwise the program does not end
        publisher.close();

        logger.info(">> End of fill database.");
    }

    static void usage() {
        System.out.println("");
        System.out.println("Usage:");
        System.out.println("");
        System.out.println("  java [ property ]* -jar ..../dbfill-${VERSION}-jar-with-dependencies.jar");
        System.out.println("");
        System.out.println("    -Ddbfill.config=/path/my-configfile.xml (default dbfill-config.xml in the dbfill jar file)");
        System.out.println("    -Dpublisher.type=[ejb|mule|jdbc]        (default 'ejb')");
        System.out.println("    ");
        System.out.println("    if publisher.type=ejb:");
        System.out.println("     -Dsmart.host=<host> (default 'localhost')");
        System.out.println("     -Dsmart.port=<port> (default '2503')");
        System.out.println("    ");
        System.out.println("    if publisher.type=mule:");
        System.out.println("     -Djprobe.dispatcher.host=<host> (default 'localhost')");
        System.out.println("     -Djprobe.dispatcher.port=<port> (default '18564')");
        System.out.println("    ");
        System.out.println("    if publisher.type=jdbc:");
        System.out.println("     -Djdbc.url=<url>       (default 'jdbc:h2:tcp://localhost:19001/db_jonas')");
        System.out.println("     -Djdbc.user=<user>     (default 'jonas')");
        System.out.println("     -Djdbc.passwd=<passwd> (default 'jonas')");
        System.out.println("");
    }
}
