/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.monitoring.tests.dbfill;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.Enumeration;


import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

public class JdbcPublisher implements IPublisher {

    static final String JDBC_URL_PROPERTY_NAME = "jdbc.url";
    static final String DEFAULT_JDBC_URL = "jdbc:h2:tcp://localhost:19001/db_jonas";
    //static final String DEFAULT_JDBC_URL = "jdbc:hsqldb:hsql://localhost:19001/db_jonas
    //static final String DEFAULT_JDBC_URL = "jdbc:oracle:thin:@localhost:1521/db_jonas";
    //static final String DEFAULT_JDBC_URL = "jdbc:postgresql://localhost:5432/db_jonas";
    //static final String DEFAULT_JDBC_URL = "jdbc:mysql://localhost:3306/db_jonas";
    
    static final String JDBC_USER_PROPERTY_NAME = "jdbc.user";
    static final String DEFAULT_JDBC_USER = "jonas";
    
    static final String JDBC_PASSWD_PROPERTY_NAME = "jdbc.passwd";
    static final String DEFAULT_JDBC_PASSWD = "jonas";
    
    static final String H2_JDBC_DRIVER = "org.h2.Driver";
    static final String HSQLDB_JDBC_DRIVER = "org.hsqldb.jdbcDriver";
    static final String MYSQL_JDBC_DRIVER = "com.mysql.jdbc.Driver";
    static final String ORACLE_JDBC_DRIVER = "oracle.jdbc.OracleDriver";
    static final String POSTGRESQL_JDBC_DRIVER = "org.postgresql.Driver";

    private Log logger = LogFactory.getLog(JdbcPublisher.class);

    Connection connection;
    PreparedStatement insertStatement;
    long nextPkId = -1;

    public JdbcPublisher() {
        
        String sDriver;
        String sUrl = System.getProperty(JDBC_URL_PROPERTY_NAME, DEFAULT_JDBC_URL);
        String sUser = System.getProperty(JDBC_USER_PROPERTY_NAME, DEFAULT_JDBC_USER);
        String sPasswd = System.getProperty(JDBC_PASSWD_PROPERTY_NAME, DEFAULT_JDBC_PASSWD);
        if (sUrl.startsWith("jdbc:h2")) {
            sDriver = H2_JDBC_DRIVER;
        } else if (sUrl.startsWith("jdbc:hsqldb")) {
            sDriver = HSQLDB_JDBC_DRIVER;
        } else if (sUrl.startsWith("jdbc:mysql")) {
            sDriver = MYSQL_JDBC_DRIVER;
        } else if (sUrl.startsWith("jdbc:oracle")) {
            sDriver = ORACLE_JDBC_DRIVER;
        } else if (sUrl.startsWith("jdbc:postgresql")) {
            sDriver = POSTGRESQL_JDBC_DRIVER;
        } else {
            logger.error("ERROR: Unknown database type (not h2, mysql, oracle, paostgresql): ");
            throw new Error("Unknown database type (not h2, mysql, oracle, paostgresql)");
        }
            
        logger.info(">> JDBC driver = " + sDriver);
        logger.info(">> JDBC url    = " + sUrl);
        logger.info(">> JDBC user   = " + sUser);
        logger.info(">> JDBC passwd = " + sPasswd);
        
        try {
            // Get the driver
            Class drvClass = JdbcPublisher.class.getClassLoader().loadClass(sDriver);
            Driver drv = (Driver) drvClass.newInstance();
            Enumeration enume = DriverManager.getDrivers();
            while (enume.hasMoreElements()) {
                enume.nextElement();
            }
            // Get a connection
            connection = DriverManager.getConnection(sUrl, sUser, sPasswd);
            // Create the PrepareStement for the insert
            insertStatement = connection.prepareStatement("INSERT INTO jm_events (id, domain, probe, server, sname, source, timestamp, value) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
            // Get the value of the initial PK id
            Statement selectStatement = connection.createStatement();
            ResultSet rset = selectStatement.executeQuery("SELECT MIN(id) FROM jm_events");
            while (rset.next ()) {
                nextPkId = rset.getLong(1) - 1;
            }
            logger.info(">> nextPkId    = " + nextPkId);
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("ERROR: Cannot connect to the database: " + e);
            throw new Error("Cannot connect to the database", e);
        }
    }

    public void publishData(JTEvent event) {
        logger.debug(">>>>>> event to publish via the EJB: " + event.toString());
        try {
            insertStatement.setLong(1, nextPkId);
            nextPkId =  nextPkId - 1;
            insertStatement.setString(2, event.getDomain());
            insertStatement.setString(3, event.getMBean()+":"+event.getVName());
            insertStatement.setString(4, event.getServer());
            insertStatement.setString(5, event.getSName());
            insertStatement.setString(6, event.getSource()+"-dbfill-jdbc");
            insertStatement.setTimestamp(7, new Timestamp(event.getTime()));
            insertStatement.setObject(8, event.getValue());
            insertStatement.execute();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("ERROR: Cannot insert an event into the database: " + e);
            throw new Error("Cannot  insert an event into the database", e);
        }
    }

    public void close() {
        // Nothing to do
        try {
            insertStatement.close();
            connection.close();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("ERROR: Cannot close the JdbcPublisher: " + e);
            throw new Error("Cannot close the JdbcPublisher", e);
        }
    }


}
