/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * $Id$
 */
package org.objectweb.jasmine.rules.dispatcher;

import org.objectweb.jasmine.rules.Notification;
import org.ow2.jasmine.event.api.IJasmineEventNotification;
import org.ow2.jasmine.event.beans.JasmineEventNotification;
import org.ow2.jasmine.event.messages.JasmineEventNotificationMail;
import org.ow2.jasmine.monitoring.eventswitch.client.DispatcherContext;
import org.ow2.jasmine.monitoring.eventswitch.client.JasmineEventNotificationDispatcher;
import org.objectweb.jasmine.rules.action.GenerateJMXNotification;

/**
 * @author Jean-Pierre Poutcheu
 * Dispatch notification events.
 * Some actions go through the EventSwitch(Mail, DB and JMS)
 * others do not(JMX and SNMP export).
 */
public class NotificationDispatcher {

    /**
     * List of possible actions
     * on a notification event.
     */
    public enum EventAction {
        SEND_MAIL,
        SAVE_IN_DATABASE,
        JMX_EXPORT,
        JMS_TOPIC;
    }

    /**
     * The client event manager.
     */
    private static EventDispatcherManager<DispatcherContext> manager = null;

    /**
     * The processor event.
     */
    private static JasmineEventNotificationDispatcher processor = null;

    /**
     * Default private constructor.
     */
    private NotificationDispatcher() {

    }

    /**
     * Dispatch notification event to the predefined endpoint URI for the given
     * action type. If you want to send to a specific URI, use
     * @link{NotificationDispatcher#dispatchEvent(String, JasmineEventNotification)}
     * @param action the action to do.
     * @param notification the event to dispatch.
     */
    public static void dispatchEvent(final EventAction action, final JasmineEventNotification notification) {

        if (action == EventAction.SAVE_IN_DATABASE) {
            // dispatch the event to the default notification database endpoint
            dispatchEvent("vm://JasmineEventNotificationEJB3Wrapper", notification);
        }
        else if (action == EventAction.JMS_TOPIC) {
            // dispatch the event to the default JMS topic endpoint
            dispatchEvent("vm://JasmineNotificationJMSComponent", notification);
        }
        else if (action == EventAction.JMX_EXPORT) {
            // export the event as a JMX notification
            // JMX_EXPORT action does not pass through the EventSwitch
            exportEventAsJMXNotification(notification);
        }
        else if (action == EventAction.SEND_MAIL) {
            throw new IllegalArgumentException("EventAction " + EventAction.SEND_MAIL
                + " requires FROM_ADDRESS and TO_ADDRESS. Use " + NotificationDispatcher.class.getSimpleName()
                + ".dispatchEvent(" + EventAction.class.getSimpleName() + ", " + JasmineEventNotification.class.getSimpleName()
                + ", String, String) instead.");
        }
        else {
            throw new IllegalArgumentException("EventAction " + action + " not supported!");
        }

    }

    /**
     * Dispatch a given event to the EventSwitch.
     * @param URI the inbound resource locator.
     * @param notification the event to dispatch.
     */
    public static void dispatchEvent(final String URI, final IJasmineEventNotification notification) {

        if (manager == null && processor == null) {
            manager = new EventDispatcherManager <  DispatcherContext  > ();
            processor = new JasmineEventNotificationDispatcher();
            manager.setProcessor(processor);
        }
        // get the initial dispatcher context
        DispatcherContext context = new DispatcherContext();
        // setting the dispatching URI
        context.setJasmineURI(URI);
        manager.setContext(context);
        try {
            // start the context
            context.start();
            // dispatch the event
            manager.dispatchEvent(notification);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }


    /**
     * Export the given notification as a JMX notification.
     * @param eventNotification the notification event
     */
    private static void exportEventAsJMXNotification(final JasmineEventNotification eventNotification) {
        Notification notification = new Notification(eventNotification.getDate(), eventNotification.getLevel(), eventNotification.getMessage());
        GenerateJMXNotification.exportAsJMXNotification(notification);
    }


    /**
     * Dispatch event to the EventSwitch in the case the needed action is
     * SEND_MAIL to the predefined endpoint URI for the given action type. If
     * you want to send to a specific URI, use
     * @link{NotificationDispatcher#dispatchEvent(String, JasmineEventNotification, String, String)}.
     * SEND_MAIL needs more informations like the FROM_ADDRESS and TO_ADDRESS.
     * @param action the action to do.
     * @param endpoint the endpoint where event is dispatched.
     * @param notification the event to dispatch.
     * @param fromAddress the from address.
     * @param toAddress the destination address.
     */
    public static void dispatchEvent(final String endpoint, final JasmineEventNotification notification, final String fromAddress, final String toAddress) {

        // transform the notification event to a mail notification
        JasmineEventNotificationMail mailNotification = new JasmineEventNotificationMail(notification.getId(),
                notification.getDate(), notification.getLevel(), notification.getMessage(), notification.getStatus(), 
                notification.getRule(), fromAddress, toAddress);
        // dispatch the event to the mail endpoint URI
        dispatchEvent(endpoint, mailNotification);
    }

    /**
     * Dispatch event to the EventSwitch in the case the needed action is SEND_MAIL.
     * SEND_MAIL needs more informations like the FROM_ADDRESS and TO_ADDRESS.
     * @param action the action to do.
     * @param notification the event to dispatch.
     * @param fromAddress the from address.
     * @param toAddress the destination address.
     */
    public static void dispatchEvent(final EventAction action, final JasmineEventNotification notification, final String fromAddress, final String toAddress) {

        if (action == EventAction.SEND_MAIL) {
            dispatchEvent("vm://JasmineEventNotificationMailWrapper", notification, fromAddress, toAddress);
        }
        else {
            dispatchEvent(action, notification);
        }

    }


}
