/**
 * JASMINe
 * Copyright (C) 2006 Bull S.A.S.
 * Contact: jasmine@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: LogInterface.java 4798 2009-08-07 12:30:47Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.objectweb.jasmine.rules.logs.impl;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ejb.Local;
import javax.ejb.Remote;
import javax.ejb.Stateful;
import javax.interceptor.Interceptors;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.persistence.TemporalType;

import org.objectweb.jasmine.rules.logs.beans.JMSLogInterceptor;
import org.objectweb.jasmine.rules.logs.beans.LogInterfaceLocal;
import org.objectweb.jasmine.rules.logs.beans.LogInterfaceRemote;
import org.objectweb.jasmine.rules.logs.beans.LogInterfaceRulesRemote;
import org.objectweb.jasmine.rules.logs.util.AlarmLog;
import org.objectweb.jasmine.rules.logs.util.DBException;

@Stateful(mappedName="log_interface")
@Interceptors({JMSLogInterceptor.class})
@Remote({LogInterfaceRemote.class, LogInterfaceRulesRemote.class})
@Local({LogInterfaceLocal.class})
public class LogInterface implements LogInterfaceRemote,
        LogInterfaceRulesRemote, LogInterfaceLocal {

    @PersistenceContext
    private EntityManager entityManager = null;

    public int addLogEntity(Date date, String level, String message)
            throws DBException {
        LogEntity logEntity = new LogEntity(date, level, message);

        try {
            entityManager.persist(logEntity);
        } catch (Exception e) {
            throw new DBException("The logEntiy " + message
                    + " cannot be added");
        }

        return logEntity.getLogEntityId();
    }

    public void deleteMessageBeforeDate(Date date) throws DBException {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        entityManager.createQuery(
                "delete from LogEntity le where le.date <= '"
                        + formatter.format(date) + "'").executeUpdate();
    }


    
    
    /**
     * Retrieves the alarm with the specified id.
     * 
     * @param id id of the alarm to be retrieved.
     * @return the alarm with the specified id or null if no alarms match the id.
     */
    public AlarmLog getLog(int id) {
        LogEntity entity = entityManager.find(LogEntity.class, id);

        return alarmFromEntity(entity);
    }
    
    /**
     * Retrieves all the alarms contained in the DB.<br>
     * 
     * @return a list of alarms or an empty list if no alarms have been raised.
     */
    @SuppressWarnings("unchecked")
    public List<AlarmLog> getAllLogs() {
        Query query = entityManager.createQuery("select le from LogEntity as le order by le.logEntityId desc");

        return alarmListFromEntityList((List<LogEntity>)query.getResultList());
    }
    
    /**
     * Retrieves the last alarms.<br>
     * The maximum number of alarms to retrieve must be specified.
     * 
     * @param nb the maximum number of alarm to retrieve.
     * @return a list of alarms or an empty list if no alarms have been raised.
     */
    @SuppressWarnings("unchecked")
    public List<AlarmLog> getLastLogs(int nb) {
        Query query = entityManager.createQuery("select le from LogEntity as le order by le.logEntityId desc");
        query.setMaxResults(nb);
        
        return alarmListFromEntityList((List<LogEntity>)query.getResultList());
    }

    /**
     * Retrieves the alarms with an id contained in the specified range of id inclusive.
     * 
     * @param lowid the low threshold.
     * @param highid the high threshold.
     * @return a list of alarms or an empty list if no alarms match.
     */
    @SuppressWarnings("unchecked")
    public List<AlarmLog> getLogRange(int lowid, int highid) {
        Query query = entityManager.createQuery("select le from LogEntity as le where le.logEntityId between " + lowid + " and " + highid);
        
        return alarmListFromEntityList((List<LogEntity>)query.getResultList());
    }

    /**
     * Retrieves the alarms matching the specified criteria.
     * All alarms which a level in the list and raised between provided dates will be returned. 
     * 
     * @param levels a list of levels.
     * @param startDate the lowest date.
     * @param endDate the highest date.
     * @return a list of alarms or an empty list if no alarms match.
     */
    @SuppressWarnings("unchecked")
    public List<AlarmLog> getLogRange(String[] levels, Date startDate, Date endDate) {
        
        // Prepare levels
        String levelList = inFromArray(levels);
        if(levelList.length() == 0) return new ArrayList<AlarmLog>();

        // Prepare Query
        String queryString = "select le from LogEntity as le";
        queryString += " where le.level in (" + levelList + ")";
        if(startDate != null) queryString += " and le.date >= :startdate";  
        if(endDate != null) queryString += " and le.date <= :enddate";  
        
        // Create Query
        Query query = entityManager.createQuery(queryString);
        if(startDate != null) query.setParameter("startdate", startDate, TemporalType.TIMESTAMP);
        if(endDate != null) query.setParameter("enddate", endDate, TemporalType.TIMESTAMP);

        // Query DB
        return alarmListFromEntityList((List<LogEntity>)query.getResultList());
    }



    /**
     * Deletes the alarm with the specified id.
     * 
     * @param id id of the alarm to be deleted.
     */
    public void delLog(int id) {
        entityManager.createQuery(
                "delete from LogEntity le where le.logEntityId = '" + id + "'"
        ).executeUpdate();
    }

    /**
     * Deletes all the alarms contained in the DB.<br>
     */
    public void delAllLogs() {
        entityManager.createQuery("delete from LogEntity").executeUpdate();
    }
    
    /**
     * Deletes the alarms matching the specified criteria.
     * All alarms which a level in the list and raised between provided dates will be deleted. 
     * 
     * @param levels a list of levels.
     * @param startDate the lowest date.
     * @param endDate the highest date.
     */
    public void delLogRange(String[] levels, Date startDate, Date endDate) {

        // Prepare levels
        String levelList = inFromArray(levels);
        if(levelList.length() == 0) return;

        // Prepare Query
        String queryString = "delete from LogEntity le";
        queryString += " where le.level in (" + levelList + ")";
        if(startDate != null) queryString += " and le.date >= :startdate";  
        if(endDate != null) queryString += " and le.date <= :enddate";  
        
        // Create Query
        Query query = entityManager.createQuery(queryString);
        if(startDate != null) query.setParameter("startdate", startDate, TemporalType.TIMESTAMP);
        if(endDate != null) query.setParameter("enddate", endDate, TemporalType.TIMESTAMP);

        // Query DB
        query.executeUpdate();
    }



    /**
     * Utility method that convert an array of string to an 'in' list in JPQL.<br>
     * 
     * @param entity the entity to convert.
     * @return the resulted alarm.
     */
    private String inFromArray(String[] strings) {
        String list = "";
        
        for(String element:strings)
            if(!"".equals(element)) list += "'" + element.toUpperCase() + "',";
        
        if(list.length() == 0) return "";
        else return list.substring(0, list.length() - 1);
    }
    
    /**
     * Utility method that convert an entity to an alarm.<br>
     * The entity has been retrieved from the entity manager.
     * 
     * @param entity the entity to convert.
     * @return the resulted alarm.
     */
    private AlarmLog alarmFromEntity(LogEntity entity) {
        if(entity == null) return null;
        return new AlarmLog(
            entity.getLogEntityId(),
            new Date(entity.getDate().getTime()),
            entity.getLevel(),
            entity.getMessage()
        );
    }
    
    /**
     * Utility method that convert a list of entities to a list of alarms.<br>
     * This uses the alarmFromEntity method to convert each entity of the list to an alarm.
     * 
     * @param logEntities the list of entities to convert.
     * @return the resulted list of alarms.
     */
    private List<AlarmLog> alarmListFromEntityList(List<LogEntity> logEntities) {
        List<AlarmLog> logTable = new ArrayList<AlarmLog>();
        for(LogEntity entity:logEntities) 
            logTable.add(alarmFromEntity(entity));
        
        return logTable;
    }
}
