/**
 * JASMINe
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineProbeManager.java 7202 2010-11-26 09:55:28Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.api;

import java.util.Collection;

/**
 * Interface for JASMINe Probe Manageranager.
 * @author danesa
 * @author durieuxp
 */
public interface JasmineProbeManager {

    /**
     * Create a new probe defined by its data
     * @param probe probe description
     * @return the probeId to be used to reference this probe later.
     */
    int createProbe(final JasmineProbe probe) throws JasmineProbeException;

    /**
     * Change a Probe with new parameters
     * The Probe must be of the same type (stat/poll)
     * @param probe probe description
     */
    void changeProbe(final JasmineProbe probe) throws JasmineProbeException;

    /**
     * Remove a probe.
     * @param id probe identifier.
     */
    void removeProbe(final int id) throws JasmineProbeException;

    /**
     * Start a probe.
     * @param id probe identifier.
     * @throws a JasmineProbeException is thrown if the probe could not
     * be started.
     */
    void startProbe(final int id) throws JasmineProbeException;

    /**
     * Stop a probe.
     * @param id probe identifier.
     * @throws a JasmineProbeException is thrown if the probe could not
     * be stopped.
     */
    void stopProbe(final int id) throws JasmineProbeException;

    /**
     * Start all the managed probes.
     * @throws JasmineProbeException
     */
    void startAllProbes() throws JasmineProbeException;

    /**
     * Stop all the managed probes.
     * @throws JasmineProbeException
     */
    void stopAllProbes() throws JasmineProbeException;

    /**
     * Remove all the managed probes.
     * @throws JasmineProbeException
     */
    void removeAllProbes() throws JasmineProbeException;

    /**
     * @return all the probe definitions
     */
    Collection<JasmineProbe> getProbes();

    /**
     * Execute a command.
     * @param command command definition containing the command's arguments
     * @return the command's result if any, null otherwise.
     * @exception JasmineProbeException ..
     */
    Collection<String> execCommand(JasmineCommand command) throws JasmineProbeException;

    /**
     * Register a ProbeListener in order to be notified by probe state changes.
     * Maybe other events will be considered...
     * @param listener object that treats the probe state change
     */
    void addProbeListener(JasmineProbeListener listener);

    /**
     * Create a target with a given name. The name must be unique, otherwise
     *  an exception is thrown.
     * @param targetName the name of the target
     * @throws a JasmineProbeException is thrown if the targetName is already
     *  in use and ...
     */
    void createTarget(final JasmineTarget target) throws JasmineProbeException;

    /**
     * Recreate a target with a given name.
     * The name must exist, otherwise an exception is thrown.
     * The target must not be used by a probe, otherwise an exception is thrown.
     * @param targetName the name of the target
     * @throws a JasmineProbeException is thrown if the targetName doesn't exist
     * or if the target is used by a running probe.
     */
    void changeTarget(final JasmineTarget target) throws JasmineProbeException;

    /**
     * Return a target object identified by its name.
     * @param targetNamen a target name
     * @return the target definition or null if does not exist.
     */
    JasmineTarget findTarget(final String targetName);

    /**
     * get the list of all defined targets.
     * @return List of the targets
     * @exception JasmineProbeException ...
     */
    Collection<JasmineTarget> getTargets() throws JasmineProbeException;

    /**
     * remove an target
     * @param name that identify the target to remove
     */
    void removeTarget(final String name) throws JasmineProbeException;

    /**
     * Create a new Output
     * @param output output description
     */
    void createOutput(final JasmineOutput output) throws JasmineProbeException;

    /**
     * Change an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     * @param output output description
     * @throws a JasmineProbeException is thrown if the output does not exist,
     * or if it is used by a running probe.
     */
    void changeOutput(final JasmineOutput output) throws JasmineProbeException;

    /**
     * Return the output definition corresponding to an output name.
     * @param name output name
     * @return the output definition or null if does not exist.
     */
    JasmineOutput findOutput(final String name);

    /**
     * remove an output
     * @param name ident of the output to remove
     */
    void removeOutput(final String name) throws JasmineProbeException;

    /**
     * get the list of all defined outputs.
     * @return List of the outputs
     */
    Collection<JasmineOutput> getOutputs() throws JasmineProbeException;

    /**
     * Save the current configuration in the specified xml file
     * @param path the xml file, or null if saved in the default configuration file.
     * @throws JasmineProbeException
     */
    void saveConfig(String path) throws JasmineProbeException;

    /**
     * Load the specified xml configuration file
     * The configuration will be merged with the current one.
     * @param path the xml file, or null if load the default configuration file.
     * @throws JasmineProbeException
     */
    void loadConfig(String path) throws JasmineProbeException;
}
