/**
 * OW2 Util
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

importPackage(org.osgi.framework);

/**
 * BundleInfo acts as a facade against a real Bundle object.
 * @param identifier bundle identifier (ID or symbolicName + version couple)
 * @constructor
 */
function BundleInfo(identifier) {

    this.bundle = this.findBundle(identifier);

    if (this.bundle != null) {
        this.headers = Converters.convertDictionaryToObject(this.bundle.headers);
        this.bundleId = this.bundle.bundleId;
        this.state = this.bundle.state;
        this.location = this.bundle.location;
        this.symbolicName = this.bundle.symbolicName;
        this.lastModified = this.bundle.lastModified;
        if (this.bundle.version != null) {
            this.version = this.bundle.version.toString();
        }
    }

    this.exists = (this.bundle != null);
    this.uninstalled = this.checkState([Bundle.UNINSTALLED]);
    this.installed = this.checkState([Bundle.INSTALLED, Bundle.RESOLVED, Bundle.STARTING, Bundle.STOPPING, Bundle.ACTIVE]);
    this.resolved = this.checkState([Bundle.RESOLVED, Bundle.STARTING, Bundle.STOPPING, Bundle.ACTIVE]);
    this.starting = this.checkState([Bundle.STARTING]);
    this.stopping = this.checkState([Bundle.STOPPING]);
    this.active = this.checkState([Bundle.ACTIVE]);

}

BundleInfo.prototype.findBundle = function(identifier) {

    // Is the identifier the BundleID ?
    if (typeof identifier == 'number') {
        var b = bundleContext.getBundle(Number(identifier));
        if (b) {
            return b;
        }
        // The given numerical ID do nto matches an installed Bundle
        return null;
    }

    // Otherwise it is something of the format:
    // symbolicName[:bundleVersion]

    var fragments = String(identifier).split(":");
    var name = fragments[0];
    var version = fragments[1];

    var bundles = bundleContext.bundles;
    for (var i = 0; i < bundles.length ; i++) {
        var b = bundles[i];
        if (name == b.symbolicName) {
            if (version != undefined) {
                // Version matching is required
                if (version == b.version) {
                    return b;
                }
            } else {
                // No constraining version, so matching only use the symbolic name
                return b;
            }
        }
    }

    return null;

}

BundleInfo.prototype.checkState = function(expected) {
    if (this.bundle == null) {
        return false;
    }

    for (var i = 0; i < expected.length; i++) {
        if (this.bundle.state == expected[i]) {
            return true;
        }
    }
    return false;
}
