/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.itests;

import java.io.File;
import java.util.ArrayList;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.management.JMX;
import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.ow2.jasmine.probe.JProbeManagerMXBean;
import org.ow2.jonas.depmonitor.MonitoringService;
import org.ow2.jonas.launcher.jonas.JOnAS;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import org.testng.Assert;
import org.testng.annotations.AfterSuite;
import org.testng.annotations.BeforeSuite;

/**
 * Abstract class for starting/stopping JasmineProbe and providing an access to
 * OSGi through BundleContext.
 * 
 * @author Florent Benoit
 */
public class JOnASLauncher {

    /**
     * Logger.
     */
    protected static Log logger = LogFactory.getLog(JOnASLauncher.class);

    /**
     * One second delay.
     */
    private static final long ONE_SECOND = 1000L;

    /**
     * Waiting time = 45 seconds.
     */
    private static final long WAITING_SERVICES_TIME = 45 * ONE_SECOND;

    /**
     * Total Collector Services number
     */
    private static final int COLLECTOR_SERVICES_NUMBER = 9;

    /**
     * Total Outer Services number
     */
    private static final int OUTER_SERVICES_NUMBER = 3;

    /**
     * Lock.
     */
    private volatile static Lock lock = new ReentrantReadWriteLock().writeLock();

    /**
     * Counter of start of JasmineProbe.
     */
    private static int count = 0;

    /**
     * JasmineProbe instance.
     */
    private static JOnAS jpServer = null;

    /**
     * OSGi bundle context.
     */
    private static BundleContext bundleContext = null;

    /**
     * Server ready.
     */
    private static boolean ready = false;

    /**
     * Server is stopping.
     */
    private static boolean stopping = false;

    /**
     * Server is stopping soon.
     */
    private static boolean lastCallStopping = false;

    /**
     * Jasmine MXBean name Probe Manager
     */
    private static String JASMINE_PROBE_MANAGER_MXBEAN_NAME = "jasmine:dest=probe-manager";

    /**
     * Jasmine MXBean Probe Manager
     */
    private static JProbeManagerMXBean proxyMXBean = null;

    /**
     * Path of the tmp directory used for the output files
     */
    private static String tmpDirPath;

    /**
     * Path of the configuration directory
     */
    private static String configurationDirPath;

    /**
     * Relative name of the Jmx jprobe client jar file
     */
    private static String JMX_CLIENT_JAR_NAME = "jprobe-client.jar";

    /**
     * Absolute name of the Jmx jprobe client jar file
     */
    private static String jmxClientJarName;

    /**
     * Carol port number
     */
    private static final String PORT_DEFAULT = "4099";
    private static int portNumber;

    /**
     * Server name
     */
    private static final String SERVER_DEFAULT = "jasmine-probe";
    private static String serverName;

    /**
     * Start the JasmineProbe server.
     * 
     * @throws Exception if server cannot be started.
     */
    @BeforeSuite
    public void startServer() throws Exception {
        try {
            lock.lock();
            if (jpServer == null) {
                jpServer = new JOnAS(true);
                // Get Bundle Context
                bundleContext = jpServer.getFramework().getBundleContext();

                // Start JasmineProbe
                new Thread() {
                    @Override
                    public void run() {
                        try {
                            jpServer.start();
                        } catch (Exception e) {
                            logger.error("Cannot start the server", e);
                            throw new IllegalStateException("Cannot start the server", e);
                        }
                    }
                }.start();

                // wait for depMonitor service
                final long maxWaitTime = System.currentTimeMillis() + WAITING_SERVICES_TIME;
                ServiceReference serviceManagerServiceReference = null;
                while (serviceManagerServiceReference == null && System.currentTimeMillis() < maxWaitTime) {
                    serviceManagerServiceReference = getBundleContext().getServiceReference(MonitoringService.class.getName());
                    if (serviceManagerServiceReference == null) {
                        Thread.sleep(ONE_SECOND);
                    }
                }
                // wait for jasmine probe services
                waitJasmineProbeServices();
                ready = true;
                logger.info("The JasmineProbe server is ready, tests can be launched...");
            } else {
                logger.info("The JasmineProbe server is already launched, tests can be launched...");
            }
            count++;
        } finally {
            lock.unlock();
        }
    }

    /**
     * Test the availability of the Jasmine Probe Services.
     * 
     * @throws InterruptedException if waiting fails
     */
    public void waitJasmineProbeServices() throws InterruptedException, org.osgi.framework.InvalidSyntaxException {

        // Scheduler Service
        long maxWaitTime = System.currentTimeMillis() + WAITING_SERVICES_TIME;
        ServiceReference serviceReference;
        do {
            serviceReference = getBundleContext().getServiceReference(org.ow2.jasmine.probe.probescheduler.SchedulerService.class.getName());
            if (serviceReference == null) {
                Thread.sleep(ONE_SECOND);
            }
        } while (serviceReference == null && System.currentTimeMillis() < maxWaitTime);
        Assert.assertNotNull(serviceReference, "Scheduler Service is not registered");

        // JMX Connection Service
        maxWaitTime = System.currentTimeMillis() + WAITING_SERVICES_TIME;
        do {
            serviceReference = getBundleContext().getServiceReference(org.ow2.jasmine.probe.jmxconnection.JmxConnectionService.class.getName());
            if (serviceReference == null) {
                Thread.sleep(ONE_SECOND);
            }
        } while (serviceReference == null && System.currentTimeMillis() < maxWaitTime);
        Assert.assertNotNull(serviceReference, "JMX Connection Service is not registered");

        // All Outer Services
        ServiceReference[] serviceReferences;
        maxWaitTime = System.currentTimeMillis() + WAITING_SERVICES_TIME;
        do {
            serviceReferences = getBundleContext().getServiceReferences(org.ow2.jasmine.probe.outer.JasmineOuterService.class.getName(), null);
            if ((serviceReferences == null) || (serviceReferences.length < OUTER_SERVICES_NUMBER)) {
                Thread.sleep(ONE_SECOND);
            }
        } while (((serviceReferences == null) || (serviceReferences.length < OUTER_SERVICES_NUMBER)) && (System.currentTimeMillis() < maxWaitTime));
        Assert.assertNotNull(serviceReferences, "Output Service not registered");
        Assert.assertEquals(serviceReferences.length, OUTER_SERVICES_NUMBER, "All the Outer Services are not registered");

        // All Collector Services
        maxWaitTime = System.currentTimeMillis() + WAITING_SERVICES_TIME;
        do {
            serviceReferences = getBundleContext().getServiceReferences(org.ow2.jasmine.probe.collector.JasmineCollectorService.class.getName(), null);
            if ((serviceReferences == null) || (serviceReferences.length <= COLLECTOR_SERVICES_NUMBER)) {
                Thread.sleep(ONE_SECOND);
            }
        } while (((serviceReferences == null) || (serviceReferences.length <= COLLECTOR_SERVICES_NUMBER)) && (System.currentTimeMillis() < maxWaitTime));
        Assert.assertNotNull(serviceReferences, "Collector Service not registered");
        Assert.assertEquals(serviceReferences.length, COLLECTOR_SERVICES_NUMBER, "All the Collectors Services are not registered");

        logger.info("The jasmine-probe services are ready.");
    }

    /**
     * Get the output tmp directory path used for the test Get a proxy to the
     * Jasmine Probe Manager MXBean
     * 
     * @throws MalformedObjectNameException
     */
    @BeforeSuite(dependsOnMethods="startServer")
    public void setup() throws MalformedObjectNameException {
        // Get the configuration directory path 
        configurationDirPath = System.getProperty("jonas.base");
        Assert.assertNotNull(configurationDirPath, "jonas.base directory path not defined");
        configurationDirPath = configurationDirPath.concat(File.separator + "conf" + File.separator);
        logger.info("The configuration directory used for probe-config files is : '" + configurationDirPath + "'");
        
        // Get the output tmp directory path used for the test
        tmpDirPath = System.getProperty("java.io.tmp");
        Assert.assertNotNull(tmpDirPath, "tmp directory path not defined");
        tmpDirPath = tmpDirPath.concat(File.separator);
        logger.info("The tmp directory used for output files is : '" + tmpDirPath + "'");

        // Get the name of the Jmx jprobe client jar file
        jmxClientJarName = System.getProperty("jonas.root");
        Assert.assertNotNull(configurationDirPath, "jonas.root directory path not defined");
        jmxClientJarName = jmxClientJarName.concat(File.separator + JMX_CLIENT_JAR_NAME);
        logger.info("The JMX JProbe client jar file used is : '" + jmxClientJarName + "'");
        
        // Get the portNumber, serverName and JMX URL
        portNumber = Integer.parseInt(System.getProperty("the.carol.port", PORT_DEFAULT));
        serverName = System.getProperty("jonas.name", SERVER_DEFAULT);

        // Get the MBeanServer
        ArrayList<MBeanServer> mbServers = MBeanServerFactory.findMBeanServer(null);
        if (mbServers.size() == 0) {
            Assert.fail("No MBeanServer !!!");
        }
        if (mbServers.size() > 1) {
            logger.info("Warning: There is severals MBeanServer, get the first one");
        }
        MBeanServer mbeanServer = mbServers.get(0);
        // Get a proxy to the Jasmine Probe Manager MXBean
        ObjectName objectName = new ObjectName(JASMINE_PROBE_MANAGER_MXBEAN_NAME);
        proxyMXBean = JMX.newMXBeanProxy(mbeanServer, objectName, JProbeManagerMXBean.class);
        logger.info("Probe Manager MXBean proxy initialized.");
    }

    /**
     * Stop the JasmineProbe server.
     * 
     * @throws Exception if server cannot be stopped
     */
    @AfterSuite
    public void stopServer() throws Exception {
        lock.lock();
        count--;
        if (count == 1) {
            lastCallStopping = true;
        }
        // last call to the stop is performing the stop;
        if (count == 0) {
            if (jpServer != null) {
                stopping = true;
                logger.info("The JasmineProbe server is being stopped.");
                jpServer.stop();
                jpServer = null;
            }
        }
        lock.unlock();
    }

    /**
     * @return true if server is ready
     */
    public boolean isReady() {
        return ready;
    }

    /**
     * Waits that the server is ready.
     * 
     * @throws InterruptedException
     *             if exception
     */
    public void waitReady() throws InterruptedException {
        while (!ready) {
            Thread.sleep(ONE_SECOND);
        }
    }

    /**
     * Waits that the server is ready.
     * 
     * @throws InterruptedException
     *             if exception
     */
    public void waitStopping() throws InterruptedException {
        while (!stopping) {
            Thread.sleep(ONE_SECOND);
        }
    }

    /**
     * Waits that the server is ready.
     * 
     * @throws InterruptedException
     *             if exception
     */
    public void waitStoppingSoon() throws InterruptedException {
        while (!lastCallStopping) {
            Thread.sleep(ONE_SECOND);
        }
    }

    /**
     * @return the bundle context.
     */
    protected BundleContext getBundleContext() {
        return bundleContext;
    }

    /**
     * @return the Probe Manager MXBean proxy.
     */
    protected JProbeManagerMXBean getProxyMXBean() {
        return proxyMXBean;
    }

    /**
     * @return the tmp directory path.
     */
    protected String getTmpDirPath() {
        return tmpDirPath;
    }

    /**
     * @return the configuration directory path.
     */
    protected String getConfigurationDirPath() {
        return configurationDirPath;
    }

    /**
     * @return the jmx client jar name
     */
    protected String getJmxClientJarName() {
        return jmxClientJarName;
    }
    
    
    /**
     * @return the port
     */
    protected int getPortNumber() {
        return portNumber;
    }
    
    /**
     * @return the serverName
     */
    protected String getServerName() {
        return serverName;
    }
    
    /**
     * @return the JMX URL
     */
    protected String getJmxUrl() {
        return ("service:jmx:rmi:///jndi/rmi://localhost:"+portNumber+"/jrmpconnector_"+serverName);
    }
    
    
}
