/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.itests.rest;

import java.net.HttpURLConnection;
import java.util.List;

import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.GenericType;
import com.sun.jersey.api.client.WebResource;

import org.ow2.jasmine.probe.api.generated.JsrType;
import org.ow2.jasmine.probe.api.generated.TypeNameType;

import org.testng.Assert;
import org.testng.annotations.Test;

public class DataTest  extends AbsRestTest {

    /**
     * Create a data task
     */
    @Test
    public void createDataTask() throws Exception {
        WebResource webResource;
        ClientResponse response;
        
        // Create a data task for the probe 'pData'
        String sUri = urlBase + "/probe/pData/data";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_ACCEPTED, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
        }
    }

    /**
     * Get data of the task 'pData'
     */
    @Test (dependsOnMethods={"createDataTask"})
    public void getData() throws Exception {
        WebResource webResource;
        ClientResponse response;

        logger.info("Wait a while to get time to do one collect ...");
        Thread.sleep(9*1000);

        // Get data from the task 'pData'
        String sUri = urlBase + "/data/pData";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // Check the results of one collect of the 'pData' probe
        List<JsrType> jsrs = response.getEntity(new GenericType<List<JsrType>>() {});
        Assert.assertEquals(jsrs.size(), 2, "Incorrect JSR number");
        for (JsrType jsr : jsrs) {
            Assert.assertEquals(jsr.getType(), TypeNameType.INT, "Incorrect JSR type for " + jsr.getName());
            if ("C_data_5".equals(jsr.getName())) {
                Assert.assertEquals(jsr.getValue().getIntvalue().intValue(), 5, "Incorrect JSR value for" + jsr.getName());
            } else if ("C_data_10".equals(jsr.getName())) {
                Assert.assertEquals(jsr.getValue().getIntvalue().intValue(), 10, "Incorrect JSR value for" + jsr.getName());
            } else {
                Assert.fail("Incorrect JSR name " + jsr.getName());
            }
        }
    }

    /**
     * Remove a data task
     */
    @Test (dependsOnMethods={"getData"})
    public void deleteDataTask() throws Exception {
        WebResource webResource;
        ClientResponse response;
        
        // Delete the data task for the probe 'pData'
        String sUri = urlBase + "/probe/pData/data";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.delete(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        
        // Check no more data can be getting
        logger.info("Wait a while to get time to do one collect ...");
        Thread.sleep(14*1000);
        // Try to get data from the task 'pData'
        sUri = urlBase + "/data/pData";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status when checking");
    }

    /**
     * Create a data task for the 'poll-cpusun' probe', get data, delete data task.
     * This test allows to check the data names.
     */
    @Test
    public void getDataPollCpu() throws Exception {
        WebResource webResource;
        ClientResponse response;
        
        // Create a data task for the probe 'pData'
        String sUri = urlBase + "/probe/poll-cpusun/data";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_ACCEPTED, "Bad HTTP status");

        // Get data from the task 'poll-cpusun'
        logger.info("Wait a while to get time to do two collects ...");
        Thread.sleep(24*1000);
        sUri = urlBase + "/data/poll-cpusun";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // Check the results of one collect of the 'poll-cpusun' probe
        List<JsrType> jsrs = response.getEntity(new GenericType<List<JsrType>>() {});
        Assert.assertEquals(jsrs.size(), 3, "Incorrect JSR number");
        for (JsrType jsr : jsrs) {
            if (!"processCpuTime".equals(jsr.getName()) && !"currentCpuTime".equals(jsr.getName()) && !"currentCpuLoad".equals(jsr.getName())) {
                Assert.fail("Incorrect JSR name " + jsr.getName());
            }
        }

        // Delete the data task for the probe 'poll-cpusun'
        sUri = urlBase + "/probe/poll-cpusun/data";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.delete(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
    }

    /**
     * Try to create a data task on an unknown probe
     */
    @Test
    public void createDataTaskUnknownProbe() throws Exception {
        WebResource webResource;
        ClientResponse response;
        
        // Try to create a data task on an unknown probe
        String sUri = urlBase + "/probe/unknown/data";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
    }

    /**
     * Try to delete an unknown data task
     */
    @Test
    public void deleteUnknownDataTask() throws Exception {
        WebResource webResource;
        ClientResponse response;
        
        // Try to delete an unknown data task
        String sUri = urlBase + "/probe/unknown/data";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.delete(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
    }

}
