/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */

package org.ow2.jasmine.probe.collectors.df.internal;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.util.CsvConverter;
import org.ow2.jasmine.probe.util.MetaData;

import java.util.*;

/**
 * Collector implementation for df.
 * This implementation works with a separate thread used to pre collect the results.
 * A simpler implementation could be done by not using a thread.
 * Using a thread is interesting when we do not want to be stuck in case info
 * cannot be got.
 * @author durieuxp
 */
public class DfCollector extends JCollector {

    /**
     * Disk if specified, or null if no disk specified (= all disks)
     */
    private Collection<String> diskList = null;

    /**
     * Value ready to be returned to user.
     */
    private JasmineIndicatorValue cache = null;

    /**
     * Collected values.
     */
    private JasmineIndicatorValue metrics = null;

    /**
     * Number of users (consumers) of indicator's value in the current probe
     */
    private int nbUsers = 1;

    /**
     * Counts the users that already consumed the
     */
    private int nbUsed = 0;

    /**
     * Time to consider data as obsolete if not got by probe (in millisec).
     * TODO : make it configurable ...??
     */
    private static final long OBSOLATE_TIME = 3000L;

    /**
     * Pass stopped to true if the probe using this collector is stopped.
     */
    private boolean stopped = false;

    /**
     * Constructor
     * @param indicator definition
     */
    public DfCollector(String probeId, JasmineIndicator indicator, int period) {
        super(probeId, indicator, period);
        // specific part of JasmineIndicator (property list)
        Map<String, String> props = indicator.getProperties();
        this.diskList = CsvConverter.csv2list(props.get("disk"));
    }

    /**
     * Get the disk to be observed
     * null if we are interested on all disks
     * @return
     */
    public Collection<String> getDiskList() {
        return diskList;
    }

    // ----------------------------------------------------------
    // JasmineCollector implementation
    // ----------------------------------------------------------

    /**
     * Retrieve the last results for this indicator
     * This method return a List of results in case indicator represents
     * actually a list of value (for example: df for ALL the disks)
     * @return List of JasmineIndicatorValue
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        logger.debug("Nb users = {0}", nbUsers);
        if (nbUsers == 1) {
            // Create the JasmineIndicatorValue instance to return at each getLastResult() call
            JasmineIndicatorValue jiv = getJiv();
            metrics = null;
            return jiv;
        }
        // value has to be cached and returned for each user
        if (nbUsed == 0) {
            // first call
            cache = getJiv();
        }
        // increment used number
        nbUsed++;
        if (nbUsed == nbUsers) {
            // this is the last time cached value need to be returned
            // force metrics to null in order to be refreshed by a worker calling needResult(), then addResult()
            logger.debug("Last cached result will be returned because nbUsed = {0}", nbUsed);
            metrics = null;
            nbUsed = 0;
        }
        logger.debug("Returned cached result");
        return cache;
    }

    /**
     * Stop polling
     */
    public void stopPolling() {
        logger.debug("Stop " + indicator.getName() + " in probe " + probeId);
        stopped = true;
    }

    /**
     * Restart polling
     */
    public void startPolling() {
        logger.debug("Start " + indicator.getName() + " in probe " + probeId);
        stopped = false;
    }
    // ----------------------------------------------------------
    // Interface used by the CollectorService
    // ----------------------------------------------------------

    /**
     * Check if this Collector needs a new result.
     * A stopped collector returns false.
     * Otherwise, it returns true if the cache is empty or outdated
     * regarding to the poll period and the OBSOLATE_TIME constant.
     * @return true if it needs one
     */
    public boolean needResult() {
        if (stopped) {
            logger.debug("STOPPED, need no result for " + indicator.getName() + " in probe " + probeId);
            return false;
        }
        if (metrics == null) {
            return true;
        }
        long timestamp = metrics.getTimestamp();
        if (oldvalue(timestamp)) {
            // forget this value too old.
            metrics = null;
            return true;
        }
        return false;
    }

    /**
     * Add a new result to the list.
     * @param value
     */
    public void addResult(JasmineIndicatorValue value) {
        logger.debug(value.getName());
        metrics = value;
    }

    // ----------------------------------------------------------
    // Private methods
    // ----------------------------------------------------------

    /**
     * @return true if this timestamp is too old for the period.
     */
    private boolean oldvalue(long timestamp) {
        long now = System.currentTimeMillis();
        return (now - timestamp > period * 1000 + OBSOLATE_TIME);
    }

    /**
     * @return The JIV from cache decorated with metadata.
     * @throws JasmineCollectorException
     */
    private JasmineIndicatorValue getJiv() throws JasmineCollectorException {
        if (metrics == null) {
            logger.debug("No value");
            return null;
        }
        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        jiv.setName(metrics.getName());
        jiv.setValues(metrics.getValues());
        jiv.setMultiValue(metrics.isMultiValue());

        logger.debug("value=" + jiv);

        // Apply scale factor to the result, if not 1.
        if (indicator.getScale() != 1) {
            for (JasmineSingleResult jsr : jiv.getValues()) {
                long value = longValue((Number) jsr.getValue()) / indicator.getScale();
                jsr.setValue(value);
            }
        }

        // Set domain and server names in JIV's metadata
        jiv.addMetadata(MetaData.SERVER, getServerName());
        jiv.addMetadata(MetaData.DOMAIN, getDomainName());

        // Set target in JIV
        jiv.setTarget(getDefaultTarget());

        return jiv;
    }

}
