/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.jmxclient;

import java.io.*;
import java.net.MalformedURLException;
import java.util.*;

import javax.management.JMX;
import javax.management.MalformedObjectNameException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import org.ow2.jasmine.probe.*;

/**
 * Client to access Jasmine Probe Manager via Jmx.
 * @author durieuxp
 */
public class JmxClient {

    private static String host = "localhost";
    private static String port = "4099";
    private static String server = "jasmine-probe";
    private static String MXBEAN = "jasmine:dest=probe-manager";

    private static JProbeManagerMXBean proxy = null;

    private static Map<String, String> params = new HashMap<String, String>();

    private static Map<String, String> properties = new HashMap<String, String>();

    private static List<String> outputs = new ArrayList<String>();

    private static List<String> targets = new ArrayList<String>();

    private static List<String> indicators = new ArrayList<String>();

    private static boolean startProbe = false;

    private static boolean verbose = false;

    private static String id = null;

    public static void main(String[] args) {

        // parse command line
        if (args.length == 0) {
            help();
            return;
        }
        String command = null;
        for (int argn = 0; argn < args.length; argn++) {
            String arg = args[argn];
            boolean nextArgument = argn < args.length - 1;
            if (arg.equals("-host") && nextArgument) {
                // host name in URL
                host = args[++argn];
                continue;
            }
            if (arg.equals("-port") && nextArgument) {
                // port number in URL
                port = args[++argn];
                continue;
            }
            if (arg.equals("-server") && nextArgument) {
                // Jmx Server name in URL
                server = args[++argn];
                continue;
            }
            if (arg.equals("-d") && nextArgument) {
                // property
                String keyval = args[++argn];
                int index = keyval.indexOf("=");
                if (index < 1) {
                    System.err.println("Bad syntax: -d should define a property");
                    continue;
                }
                String key = keyval.substring(0, index);
                String value = keyval.substring(index + 1);
                properties.put(key, value);
                continue;
            }
            if (arg.equals("-o") && nextArgument) {
                // output
                outputs.add(args[++argn]);
                continue;
            }
            if (arg.equals("-target") && nextArgument) {
                // target
                targets.add(args[++argn]);
                continue;
            }
            if (arg.equals("-i") && nextArgument) {
                // indicator
                indicators.add(args[++argn]);
                continue;
            }
            if (arg.equals("-n") && nextArgument) {
                // name
                if (id == null) {
                    id = args[++argn];
                } else {
                   System.err.println("Bad syntax: name defined twice. Take: " + id);
                }
                continue;
            }
            if (arg.equals("-v") || arg.equals("-verbose")) {
                verbose = true;
                continue;
            }
            if (arg.equals("-start")) {
                startProbe = true;
                continue;
            }
            if (arg.startsWith("-") && nextArgument) {
                // other params
                params.put(arg.substring(1), args[++argn]);
                continue;
            }
            // Check if property definition
            int index = arg.indexOf("=");
            if (index > 0) {
                String key = arg.substring(0, index);
                String value = arg.substring(index + 1);
                properties.put(key, value);
                continue;
            }
            // assumes this arg is the command name, or if already defined, the id.
            if (command == null) {
                command = arg;
                continue;
            }
            if (id == null) {
                id = arg;
                continue;
            }
            System.err.println("Bad syntax: <" + arg + "> has been ignored");
        }

        // Connect to the Jmx Server ans get an MXBean proxy
        getMXBeanProxy();
        if (proxy == null) {
            // Cannot connect to server
            System.exit(1);
        }

        // Call the appropriate command
        try {
            if ("help".equals(command) || command == null) {
                help();
            } else if ("indicator-create".equals(command)) {
                createIndicator();
            } else if ("indicator-change".equals(command)) {
                changeIndicator();
            } else if ("indicator-list".equals(command)) {
                listIndicators();
            } else if ("indicator-remove".equals(command)) {
                removeIndicator();
            } else if ("indicator-types".equals(command)) {
                listIndicatorTypes();
            } else if ("indicator-properties".equals(command)) {
                listIndicatorProperties();
            } else if ("indicator-info".equals(command)) {
                listIndicatorInfo();
            } else if ("output-create".equals(command)) {
                createOutput();
            } else if ("output-change".equals(command)) {
                changeOutput();
            } else if ("output-list".equals(command)) {
                listOutputs();
            } else if ("output-remove".equals(command)) {
                removeOutput();
            } else if ("output-types".equals(command)) {
                listOutputTypes();
            } else if ("output-properties".equals(command)) {
                listOutputProperties();
            } else if ("target-create".equals(command)) {
                createTarget();
            } else if ("target-list".equals(command)) {
                listTargets();
            } else if ("target-remove".equals(command)) {
                removeTarget();
            } else if ("probe-create".equals(command)) {
                createProbe();
            } else if ("probe-change".equals(command)) {
                changeProbe();
            } else if ("probe-list".equals(command)) {
                listProbes();
            } else if ("probe-remove".equals(command)) {
                removeProbe();
            } else if ("probe-start".equals(command)) {
                startProbe();
            } else if ("probe-stop".equals(command)) {
                stopProbe();
            } else if ("config-save".equals(command)) {
                saveConfig();
            } else if ("config-load".equals(command)) {
                loadConfig();
            } else {
                System.err.println("command not found: " + command);
            }
        } catch (JasmineProbeException e) {
            System.err.println("Operation failed: " + e.getMessage());
            System.exit(2);
        }
        System.exit(0);
    }

    private static void createIndicator() throws JasmineProbeException {
        JasmineIndicator indic = new JasmineIndicator();

        // Mandatory Ident
        if (id == null) {
            throw new JasmineProbeException("option -n missing");
        }
        indic.setName(id);

        // type: -t <type>
        String type = params.get("t");
        if (type == null) {
            throw new JasmineProbeException("option -t missing");
        }
        indic.setType(type);

        // scale: -scale <int_value>
        int scale = 1;
        if (params.get("scale") != null) {
            scale = new Integer(params.get("scale"));
        }
        indic.setScale(scale);

        // Sources: [-i indicator]*
        indic.setSources(indicators);

        // properties: list of "key=value"
        indic.setProperties(properties);

        // Create the Indicator on the server.
        try {
            proxy.createIndicator(indic);
        } catch (TypeNotSupportedException e) {
            System.out.println(e.getMessage());
            System.out.println("Supported indicator types are:");
            listIndicatorTypes();
            throw new JasmineProbeException("Type Not Supported");
        } catch (PropertyMissingException e) {
            System.out.println(e.getMessage());
            listIndicatorProperties();
            throw new JasmineProbeException("Missing Property");
        } catch (SourceNumberException e) {
            System.out.println(e.getMessage());
            throw new JasmineProbeException(e.getMessage());
        }
        System.out.println("indicator " + id + " created");
    }

    private static void changeIndicator() throws JasmineProbeException {
        JasmineIndicator indic = new JasmineIndicator();
        if (id == null) {
            throw new JasmineProbeException("option -n missing");
        }
        indic.setName(id);

        // type is null if not changed
        String type = params.get("t");
        indic.setType(type);

        // source indicators
        indic.setSources(indicators);

        // pass only properties changed or new one.
        indic.setProperties(properties);

        try {
            proxy.changeIndicator(indic);
        } catch (TypeNotSupportedException e) {
            System.out.println(e.getMessage());
            System.out.println("Supported Indicator types :");
            listIndicatorTypes();
            throw new JasmineProbeException("Type Not Supported");
        } catch (PropertyMissingException e) {
            System.out.println(e.getMessage());
            listIndicatorProperties();
            throw new JasmineProbeException("Missing Property");
        } catch (SourceNumberException e) {
            System.out.println(e.getMessage());
            throw new JasmineProbeException(e.getMessage());
        }
        System.out.println("indicator " + id + " changed");
    }

    private static void listIndicators() throws JasmineProbeException {
        if (id == null) {
            // list indicators
            String type = params.get("t");
            String [] idList = proxy.listIndicators(type);
            if (idList.length == 0) {
                if (type == null) {
                    System.out.println("No JasmineIndicator found");
                } else {
                    System.out.println("No JasmineIndicator found for type " + type);
                }
                return;
            }
            for (int i = 0; i < idList.length; i++) {
                if (verbose) {
                    JasmineIndicator indic = proxy.getIndicator(idList[i]);
                    System.out.println(indic.toLine());
                } else {
                    System.out.println(idList[i]);
                }
            }
        } else {
           // show an indicator
            JasmineIndicator indic = proxy.getIndicator(id);
            if (indic == null) {
                throw new JasmineProbeException(id + " not found");
            }
            System.out.println(indic.toString());
            if (verbose) {
                // Show current
                System.out.println("List of probes using this indicator:");
                Set<String> probes = proxy.getIndicatorCurrentUse(id);
                for (String pid : probes) {
                    JasmineProbe probe = proxy.getProbe(pid);
                    if (probe == null) {
                        System.err.println(id + " not found");
                        continue;
                    }
                    System.out.println(pid + "   (" + probe.printStatus() + ")");
                }
            }
        }
    }

    private static void listIndicatorTypes() throws JasmineProbeException {
        Set<String> liste = proxy.getIndicatorTypes();
        for (String type : liste) {
            if (verbose) {
                JasmineCollectorInfo info = proxy.getIndicatorTypeInfo(type);
                String form = (type + "            ").substring(0, 12) + info.getShortDesc();
                System.out.println(form);
            } else {
                System.out.println(type);
            }
        }
    }

    private static void listIndicatorProperties() throws JasmineProbeException {
        String type = params.get("t");
        if (type == null) {
            throw new JasmineProbeException("option -t missing");
        }
        JasmineCollectorInfo info = proxy.getIndicatorTypeInfo(type);
        if (info == null) {
            throw new JasmineProbeException("invalid type; " + type);
        }
        List<JasminePropertyInfo> props = info.getPropertyInfos();
        if (! props.isEmpty()) {
            System.out.println("The properties flagged with * are mandatory:");
            for (JasminePropertyInfo prop : props) {
                System.out.println(prop.toString());
            }
        }
    }

    private static void listIndicatorInfo() throws JasmineProbeException {
        String type = params.get("t");
        if (type == null) {
            throw new JasmineProbeException("option -t missing");
        }
        JasmineCollectorInfo info = proxy.getIndicatorTypeInfo(type);
        if (info == null) {
            throw new JasmineProbeException("invalid type; " + type);
        }
        System.out.println(info.getShortDesc());
        System.out.println(info.getLongDesc());
        // Print info about sources only if sources might exist
        if (info.getSourceMax() > 0) {
            System.out.println(info.getSourceDesc());
        }
        // properties
        List<JasminePropertyInfo> props = info.getPropertyInfos();
        if (! props.isEmpty()) {
            System.out.println("The properties flagged with * are mandatory:");
            for (JasminePropertyInfo prop : props) {
                System.out.println(prop.toString());
            }
        }
    }

    private static void removeIndicator() throws JasmineProbeException {
        if (id == null) {
            throw new JasmineProbeException("option -n missing");
        }
        proxy.removeIndicator(id);
        System.out.println("indicator removed");
    }

    private static void createTarget() throws JasmineProbeException {
        JasmineTarget target = new JasmineTarget();

        if (id == null) {
            throw new JasmineProbeException("option -n missing (name)");
        }
        target.setName(id);

        String url = params.get("u");
        if (id == null) {
            throw new JasmineProbeException("option -u missing (url)");
        }
        target.setUrl(url);

        proxy.createTarget(target);
        System.out.println("target " + id + " created");
    }

    private static void listTargets() throws JasmineProbeException {
        if (id == null) {
            // list all targets
            String [] idList = proxy.listTargets();
            if (idList.length == 0) {
                System.out.println("No JasmineTarget found");
                return;
            }
            for (int i = 0; i < idList.length; i++) {
                if (verbose) {
                    JasmineTarget target = proxy.getTarget(idList[i]);
                    System.out.println(target.toLine());
                } else {
                    System.out.println(idList[i]);
                }
            }
        } else {
           // show a target
            JasmineTarget target = proxy.getTarget(id);
            if (target == null) {
                throw new JasmineProbeException(id + " not found");
            }
            System.out.println(target.toString());
        }
    }

    private static void removeTarget() throws JasmineProbeException {
        if (id == null) {
            throw new JasmineProbeException("option -n missing");
        }
        proxy.removeTarget(id);
        System.out.println("target removed");
    }

    private static void createOutput() throws JasmineProbeException {
        JasmineOutput output = new JasmineOutput();
        if (id == null) {
            throw new JasmineProbeException("option -n missing");
        }
        output.setName(id);
        String type = params.get("t");
        if (type == null) {
            throw new JasmineProbeException("option -t missing");
        }
        output.setType(type);
        output.setProperties(properties);
        try {
            proxy.createOutput(output);
        } catch (TypeNotSupportedException e) {
            System.out.println(e.getMessage());
            System.out.println("Supported Output types :");
            listOutputTypes();
            throw new JasmineProbeException("Type not supported: " + type);
        } catch (PropertyMissingException e) {
            System.out.println(e.getMessage());
            listOutputProperties();
            throw new JasmineProbeException("Missing property");
        }
        System.out.println("output " + id + " created");
    }

    private static void changeOutput() throws JasmineProbeException {
        JasmineOutput output = new JasmineOutput();
        if (id == null) {
            throw new JasmineProbeException("option -n missing");
        }
        output.setName(id);

        // type is null if not changed
        String type = params.get("t");
        output.setType(type);

        // pass only properties changed or new one.
        output.setProperties(properties);

        try {
            proxy.changeOutput(output);
        } catch (TypeNotSupportedException e) {
            System.out.println(e.getMessage());
            System.out.println("Supported Output types :");
            listOutputTypes();
            throw new JasmineProbeException("Type not supported: " + type);
        } catch (PropertyMissingException e) {
            System.out.println(e.getMessage());
            listOutputProperties();
            throw new JasmineProbeException("Missing property");
        }
        System.out.println("output " + id + " changed");
    }

    private static void listOutputs() throws JasmineProbeException {
        if (id == null) {
            // list outputs
            String type = params.get("t");
            String [] idList = proxy.listOutputs(type);
            if (idList.length == 0) {
                if (type == null) {
                    System.out.println("No JasmineOutput found");
                } else {
                    System.out.println("No JasmineOutput found for type " + type);
                }
                return;
            }
            for (int i = 0; i < idList.length; i++) {
                if (verbose) {
                    JasmineOutput output = proxy.getOutput(idList[i]);
                    System.out.println(output.toLine());
                } else {
                    System.out.println(idList[i]);
                }
            }
        } else {
            // show an output
            JasmineOutput output = proxy.getOutput(id);
            if (output == null) {
                throw new JasmineProbeException(id + " not found");
            }
            System.out.println(output.toString());
            if (verbose) {
                // Show current
                System.out.println("List of probes using this output:");
                Set<String> probes = proxy.getOutputCurrentUse(id);
                for (String pid : probes) {
                    JasmineProbe probe = proxy.getProbe(pid);
                    if (probe == null) {
                        System.err.println(id + " not found");
                        continue;
                    }
                    System.out.println(pid + "   (" + probe.printStatus() + ")");
                }
            }
        }
    }

    private static void listOutputTypes() throws JasmineProbeException {
        Set<String> liste = proxy.getOutputTypes();
        for (String type : liste) {
             System.out.println(type);
        }
    }

    private static void listOutputProperties() throws JasmineProbeException {
        String type = params.get("t");
        if (type == null) {
            throw new JasmineProbeException("option -t missing");
        }
        List<JasminePropertyInfo> props = proxy.getOutputPropertyInfos(type);
        if (props == null) {
            throw new JasmineProbeException("invalid type; " + type);
        }
        if (! props.isEmpty()) {
            System.out.println("The properties flagged with * are mandatory:");
            for (JasminePropertyInfo prop : props) {
                System.out.println(prop.toString());
            }
        }
    }

    private static void removeOutput() throws JasmineProbeException {
        if (id == null) {
            throw new JasmineProbeException("option -n missing");
        }
        proxy.removeOutput(id);
        System.out.println("output removed");
    }

    private static void loadConfig() throws JasmineProbeException {
        String path = params.get("f");
        if (path == null) {
            throw new JasmineProbeException("option -f missing");
        }
        proxy.loadConfig(path);
        System.out.println("config has been loaded");
    }

    private static void saveConfig() throws JasmineProbeException {
        String path = params.get("f");
        proxy.saveConfig(path);
        System.out.println("config has been saved");
    }

    private static void createProbe() throws JasmineProbeException {
        JasmineProbe probe = new JasmineProbe();
        probe.setId(id);

        int period = 10;
        if (params.get("p") != null) {
            period = new Integer(params.get("p"));
        }
        probe.setPeriod(period);

        probe.setOutputList(outputs);
        probe.setTargetList(targets);
        probe.setIndicatorList(indicators);

        id =  proxy.createProbe(probe);
        System.out.println("probe " + id + " created");

        if (startProbe) {
            proxy.startProbe(id);
            System.out.println("probe " + id + " started");
        }
    }

    private static void changeProbe() throws JasmineProbeException {
        JasmineProbe probe = new JasmineProbe();
        if (id == null) {
            throw new JasmineProbeException("option -n missing");
        }
        probe.setId(id);

        int period = -1;
        if (params.get("p") != null) {
            period = new Integer(params.get("p"));
        }
        probe.setPeriod(period);

        probe.setOutputList(outputs);
        probe.setTargetList(targets);
        probe.setIndicatorList(indicators);

        proxy.changeProbe(probe);
        System.out.println("probe " + id + " changed");
    }

    private static void listProbes() throws JasmineProbeException {
        if (id == null) {
            // list all probes
            String [] idList = proxy.listProbes();
            for (int i = 0; i < idList.length; i++) {
                if (verbose) {
                    JasmineProbe probe = proxy.getProbe(idList[i]);
                    System.out.println(probe.toLine());
                } else {
                    System.out.println(idList[i]);
                }
            }
        } else {
           // show a probe
            JasmineProbe probe = proxy.getProbe(id);
            if (probe == null) {
                throw new JasmineProbeException(id + " not found");
            }
            System.out.println(probe.toString());
        }
    }

    private static void startProbe() throws JasmineProbeException {
        if (id == null) {
            // start all probes
            proxy.startAllProbes();
            System.out.println("all probes started");
        } else {
            proxy.startProbe(id);
            System.out.println("probe started");
        }
    }

    private static void stopProbe() throws JasmineProbeException {
        if (id == null) {
            // stop all probes
            proxy.stopAllProbes();
            System.out.println("all probes stopped");
        } else {
            proxy.stopProbe(id);
            System.out.println("probe stopped");
        }
    }

    private static void removeProbe() throws JasmineProbeException {
        if (id == null) {
            // remove all probes
            proxy.removeAllProbes();
            System.out.println("all probes removed");
        } else {
            proxy.removeProbe(id);
            System.out.println("probe removed");
        }
    }

    /**
     * print the help file
     */
    private static void help() {
        String filename = "help.txt";
        if (id != null) {
            filename = id + "-help.txt";
        }
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        InputStream in0 = classLoader.getResourceAsStream(filename);
        if (in0 == null) {
            System.out.println("This help file is not available: " + filename);
            return;
        }
        BufferedReader in = new BufferedReader(new InputStreamReader(in0));
        OutputStreamWriter out = new OutputStreamWriter(System.out);
        try {
            boolean goOn = true;
            char[] buf = new char[2048];
            while (goOn) {
                int read = in.read(buf);
                if (read > 0) {
                    out.write(buf, 0, read);
                    out.flush();
                } else {
                    goOn = false;
                    System.out.println();
                }
            }
        } catch (IOException e) {
            System.err.println("Error reading help: " + e);
        } finally {
            if (in != null) {
                try {
                    in.close();
                } catch (IOException e) {
                    System.err.println("Error closing the BufferedReader: " + e);
                }
            }
        }
    }

    private static void getMXBeanProxy() {
        String url = "service:jmx:rmi:///jndi/rmi://" + host + ":" + port + "/jrmpconnector_" + server;
        try {
            JMXServiceURL jmxUrl = new JMXServiceURL(url);
            JMXConnector connector = JMXConnectorFactory.connect(jmxUrl);
            MBeanServerConnection mbsc = connector.getMBeanServerConnection();
            ObjectName objectName = new ObjectName(MXBEAN);
            proxy = JMX.newMXBeanProxy(mbsc, objectName, JProbeManagerMXBean.class);
        } catch (MalformedURLException e) {
            System.err.println("Cannot build JMXServiceURL with "
                    + url + ":\n" + e.getMessage());
        } catch (MalformedObjectNameException e) {
            System.err.println(MXBEAN + " is a bad object name.: " + e.getMessage());
        } catch (IOException e) {
            System.err.println("Cannot connect to " + url + ":\n" + e.getMessage());
        }
    }


}
